package com.efuture.ocp.common.callnumber;

import cn.hutool.core.thread.ExecutorBuilder;
import com.efuture.common.utils.ServiceLogs;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.function.Consumer;

/**
 * 叫号执行器
 * submit ：提交一个任务到线程池，根据maxWorker 创建线程并按maxWorker去获取号码并执行
 * call:直接获取一个号并执行一次任务
 */
public class CallNumberExecutor
{

    CallNumberHandle handle;
    /**
     * 一个叫号器，一个线程池
     */
    ConcurrentMap<String, ExecutorService> executors = new ConcurrentHashMap<>();

    ConcurrentMap<String, List<CallNumberRuner>> workers = new ConcurrentHashMap<>();

    public CallNumberExecutor(CallNumberHandle handle)
    {
        this.handle = handle;
    }

    public void submit(String bizName, int maxNum, int maxWorker, Consumer<NumberInfo> worker)
    {
        ExecutorService executorService = executors.computeIfAbsent(bizName,
        (key) -> {
            return (ExecutorService) ExecutorBuilder.create().setMaxPoolSize(maxWorker).setCorePoolSize(0).build();
        });
        List<CallNumberRuner> workList = workers.computeIfAbsent(bizName,
        (key) -> {
            List<CallNumberRuner> runers = new ArrayList<>();

            for (int i = 0; i < maxWorker; i++)
            {
                CallNumberRuner run = new CallNumberRuner(key, maxNum, i - 1, worker);
                runers.add(run);
            }
            return runers;
        });

        if (workList.size() < maxWorker) {
            for (int i = workList.size() - 1; i < maxWorker; i++) {
                CallNumberRuner run = new CallNumberRuner(bizName, maxNum, i, worker);
                workList.add(run);
            }
        }

        for (CallNumberRuner runer : workList) {
            executorService.submit(runer);
        }
    }

    public boolean watchPrint(String bizName)
    {
        List<CallNumberRuner> workList = workers.get(bizName);
        boolean ibRun = false;

        for (CallNumberRuner runer : workList) {
            ibRun = runer.isRunning();
            ServiceLogs.truedebuglog("watchPrint", runer.watch(), 0);
        }

        return ibRun;
    }

    public CallNumberRuner call(String bizName, int maxNumer, Consumer<NumberInfo> worker)
    {
        CallNumberRuner run = new CallNumberRuner(bizName, maxNumer, 0, worker);
        run.run();
        return run;
    }

    /**
     * 获取一个号码
     *
     * @param bizName
     * @param maxNum
     * @param waitSecond
     * @return
     */
    public NumberInfo getNumber(String bizName, int maxNum, int waitSecond)
    {
        NumberInfo num = new NumberInfo(bizName, maxNum, -99);
        int rtn = handle.next(num, waitSecond);

        if (rtn < 0) {
            return null;
        }
        else {
            return num;
        }
    }

    /**
     * 获取下一个号码
     *
     * @param num
     * @param waitSecond
     * @return
     */
    public int getNextNumber(NumberInfo num, int waitSecond)
    {
        if (num.isRunning()) {
            handle.complete(num);
        }

        return handle.next(num, waitSecond);
    }

    public void destroy()
    {
        for (ExecutorService service : executors.values()) {
            service.shutdown();
        }
    }

    private class CallNumberRuner implements Runnable
    {
        public String curJobKey = "";
        private NumberInfo number;
        private Consumer<NumberInfo> worker;
        private String lineSeparator = System.lineSeparator();

        public CallNumberRuner(NumberInfo number, Consumer<NumberInfo> worker)
        {
            this.number = number;
            this.worker = worker;
        }

        public CallNumberRuner(String bizName, int maxNum, int start, Consumer<NumberInfo> worker)
        {
            this.number = new NumberInfo(bizName, maxNum, start);
            this.worker = worker;
        }

        public NumberInfo getNumber()
        {
            return number;
        }

        public Consumer<NumberInfo> getWorker()
        {
            return worker;
        }

        public String watch()
        {
            StringBuilder sb = new StringBuilder();
            sb.append(lineSeparator);
            sb.append("number:" + number.toString());
            sb.append(lineSeparator);
            sb.append("isRuning:" + number.isRunning());
            return sb.toString();
        }

        public boolean isRunning()
        {
            return number.isRunning();
        }

        @Override
        public void run()
        {
            curJobKey = Thread.currentThread().getName() + "-" + this.number.toString();
            ServiceLogs.setCurLogKey(curJobKey);
            ServiceLogs.info("CallNumberRuner", "开始运行[{0}]", 0, this.number.toString());

            if (this.number.isRunning()) {
                ServiceLogs.info("CallNumberRuner", "当前序号[{0}]正在执行,不继续执行", 0, this.number.toString());
                return;
            }

            //循环叫号处理，只要叫不到号了，就直接循环
            int num = -99;

            do {
                try {
                    num = handle.next(this.number, 10);

                    if (num >= 0) {
                        ServiceLogs.info("CallNumberRuner", "开始执行[{0}]", 0, this.number.toString());
                        this.worker.accept(this.number);
                    }
                    else {
                        ServiceLogs.info("CallNumberRuner", "开始执行[{0}]", 0, this.number.toString());
                    }
                }
                catch (Exception e) {
                    e.printStackTrace();
                }
                finally {
                    handle.complete(this.number);
                }
            } while (num >= 0);
        }
    }
}
