package com.efuture.ocp.common.distributedLock;

import com.jarvis.cache.serializer.StringSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.RedisStringCommands;
import org.springframework.data.redis.core.RedisConnectionUtils;
import org.springframework.data.redis.core.types.Expiration;

import java.util.concurrent.TimeUnit;

public class SpringRedisDistributedLockHandle extends AbstractDistributedLockHandle
{

    private static final Logger logger = LoggerFactory.getLogger(SpringRedisDistributedLockHandle.class);
    private static final StringSerializer STRING_SERIALIZER = new StringSerializer();

    private final RedisConnectionFactory redisConnectionFactory;

    public SpringRedisDistributedLockHandle(RedisConnectionFactory redisConnectionFactory)
    {
        this.redisConnectionFactory = redisConnectionFactory;
    }

    private RedisConnection getConnection()
    {
        return RedisConnectionUtils.getConnection(redisConnectionFactory);
    }

    @Override
    public boolean doNaiveCachePut(String key, DLockInfo lockInfo, int lockExpire, int waitMillSeconds)
    {
        if (null == redisConnectionFactory || null == key || key.isEmpty()) {
            return false;
        }

        RedisConnection redisConnection = getConnection();

        try {
            String val = String.valueOf(lockInfo.getLockKey());
            Expiration expiration = Expiration.from(lockExpire, TimeUnit.SECONDS);
            // 采用redisson做客户端时，set key value [EX | PX] [NX | XX] 会因为条件不满足无法设值成功而返回null导致拆箱空指针
            Boolean locked = redisConnection.stringCommands().set(STRING_SERIALIZER.serialize(key), STRING_SERIALIZER.serialize(val), expiration,
                             RedisStringCommands.SetOption.SET_IF_ABSENT);
            return locked == null ? false : locked;
        }
        catch (Exception ex) {
            logger.error(ex.getMessage(), ex);
        }
        finally {
            RedisConnectionUtils.releaseConnection(redisConnection, redisConnectionFactory, false);
        }

        return false;
    }

    @Override
    public boolean doNaiveCacheRenewal(DLockInfo lockInfo)
    {
        if (null == redisConnectionFactory || null == lockInfo) {
            return false;
        }

        RedisConnection redisConnection = getConnection();

        try {
            String key = lockInfo.getKey();
            String val = String.valueOf(lockInfo.getLockKey());
            Expiration expiration = Expiration.from(lockInfo.getLeaseTime(), TimeUnit.SECONDS);
            // 采用redisson做客户端时，set key value [EX | PX] [NX | XX] 会因为条件不满足无法设值成功而返回null导致拆箱空指针
            Boolean succ = redisConnection.stringCommands().set(STRING_SERIALIZER.serialize(key), STRING_SERIALIZER.serialize(val), expiration,
                           RedisStringCommands.SetOption.SET_IF_PRESENT);
            return succ == null ? false : succ;
        }
        catch (Exception ex) {
            logger.error(ex.getMessage(), ex);
        }
        finally {
            RedisConnectionUtils.releaseConnection(redisConnection, redisConnectionFactory, false);
        }

        return false;
    }

    @Override
    public void doNaiveCacheDelete(String key)
    {
        if (null == redisConnectionFactory || null == key || key.length() == 0) {
            return;
        }

        RedisConnection redisConnection = getConnection();

        try {
            redisConnection.keyCommands().del(STRING_SERIALIZER.serialize(key));
        }
        catch (Exception ex) {
            logger.error(ex.getMessage(), ex);
        }
        finally {
            RedisConnectionUtils.releaseConnection(redisConnection, redisConnectionFactory, false);
        }
    }

    @Override
    public long doNaiveCacheGet(String key)
    {
        long rtn = 0;

        if (null == redisConnectionFactory || null == key || key.length() == 0) {
            return rtn;
        }

        RedisConnection redisConnection = getConnection();

        try {
            String str = STRING_SERIALIZER.deserialize(redisConnection.stringCommands().get(STRING_SERIALIZER.serialize(key)), String.class);

            if (str != null) {
                rtn = Long.valueOf(str);
            }
        }
        catch (Exception ex) {
            logger.error(ex.getMessage(), ex);
        }
        finally {
            RedisConnectionUtils.releaseConnection(redisConnection, redisConnectionFactory, false);
        }

        return rtn;
    }
}
