package com.efuture.msboot.service.impl;

import com.alibaba.fastjson.util.TypeUtils;
import com.efuture.msboot.clouds.RestBeanAccess;
import com.efuture.msboot.core.bean.EasyBeanWrapper;
import com.efuture.msboot.core.utils.ParamCheckUtil;
import com.efuture.msboot.core.utils.SpringContextHolder;
import com.efuture.msboot.data.DataAccess;
import com.efuture.msboot.data.ViewAccess;
import com.efuture.msboot.data.bean.CURDEnum;
import com.efuture.msboot.data.bean.DataPage;
import com.efuture.msboot.data.bean.Query;
import com.efuture.msboot.data.query.SqlBuilder;
import com.efuture.msboot.data.query.bean.SearchInfo;
import com.efuture.msboot.data.utils.EntityUtils;
import com.efuture.msboot.data.utils.ViewUtils;
import com.efuture.msboot.service.BaseListService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import java.io.Serializable;
import java.lang.reflect.ParameterizedType;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by wzm on 2019/2/18.
 */
@Slf4j
public abstract class BaseListServiceImpl<T> implements BaseListService<T> {
    @Lazy
    @Autowired(required = false)
    protected DataAccess dataAccess;

    @Lazy
    @Autowired(required = false)
    protected ViewAccess viewAccess;

    @Autowired
    protected SqlBuilder sqlBuilder;

    /**
     * T 对应的 class
     */
    protected Class<T> beanClazz;

    /**
     * 是否支持通过rest的方式填充bean里面的字段
     */
    protected Boolean supportRestBean = true;

    protected RestBeanAccess restBeanAccess;

    @PostConstruct
    public void init(){
        ParameterizedType type = (ParameterizedType) this.getClass().getGenericSuperclass();
        this.beanClazz = (Class<T>) type.getActualTypeArguments()[0];
    }

    @Override
    public T add(T entity) {
        Assert.notNull(entity, "entry 不能为空");

        EntityUtils.initInsert(entity);

        dataAccess.insert(entity);

        Serializable id = EntityUtils.getIdValue(entity);

        return get(id);
    }

    @Override
    public T update(T entity) {
        Assert.notNull(entity, "entity 不能为空");

        Serializable id = EntityUtils.getIdValue(entity);

        Assert.notNull(id, "id 不能为空");

        EntityUtils.initUpdate(entity);

        dataAccess.updateById(entity);

        return get(id);
    }

    @Override
    public int delete(Serializable id) {
        Assert.notNull(id, "id 不能为空");

        return dataAccess.deleteById(id, beanClazz);
    }

    @Override
    public int deleteByMap(Map<String,Object> params){
        Assert.notNull(params, "params 不能为空");

        return dataAccess.deleteByMap(params, beanClazz);
    }

    @Override
    public int batchDelete(List<Serializable> ids) {
        return dataAccess.deleteBatchIds(ids, beanClazz);
    }

    /**
     * 根据query删除
     * @param query
     * @return
     */
    @Override
    public int deleteByQuery(Query<T> query){
        return dataAccess.delete(query, this.beanClazz);
    }

    @Override
    public List<Serializable> batchAdd(List<T> entityList) {
        return batchAdd(entityList, false);
    }

    @Transactional
    @Override
    public List<Serializable> batchAdd(List<T> entityList, Boolean validate) {
        Assert.notEmpty(entityList, "entityList 不能为空");

        List<Serializable> ids = new ArrayList<Serializable>();

        for(T entity:entityList){
            if(validate){
                ParamCheckUtil.validate(entity);
            }

            EntityUtils.initInsert(entity);

            dataAccess.insert(entity);

            ids.add(EntityUtils.getIdValue(entity));
        }

        return ids;
    }


    /**
     * 批量保存
     * @param entityList
     * @return
     */
    @Transactional
    @Override
    public List<T> save(List<T> entityList){
        Assert.notEmpty(entityList, "entityList 不能为空");

        // 先删除
        for(T entity:entityList){
            EasyBeanWrapper beanWrapper = new EasyBeanWrapper(entity);

            // 没有这个字段，默认不处理
            if(!beanWrapper.getPropertiesSet().contains("curd_flag")){
                continue;
            }

            String curd_flag = EntityUtils.getCurdFlag(entity);
            if(CURDEnum.DELETE.toString().equalsIgnoreCase(curd_flag)){
                Serializable idValue = EntityUtils.getIdValue(entity);
                dataAccess.deleteById(idValue, entity.getClass());
            }
        }

        for(Object entity:entityList){
            //更新标识
            String curd_flag = EntityUtils.getCurdFlag(entity);

            //新增
            if(CURDEnum.INSERT.toString().equalsIgnoreCase(curd_flag)){
                EntityUtils.initInsert(entity);

                dataAccess.insert(entity);
            }else if(CURDEnum.UPDATE.toString().equalsIgnoreCase(curd_flag)){
                EntityUtils.initUpdate(entity);

                dataAccess.updateById(entity);
            }else{
                log.warn("无法处理标识 "+curd_flag);
            }
        }

        handleRestBean(entityList);

        return entityList;
    }

    @Override
    public T get(Serializable id) {
        Assert.notNull(id, "id 不能为空");

        if(!ViewUtils.isView(this.beanClazz)) {
            return (T) dataAccess.selectById(id, beanClazz);
        }

        T entity = viewAccess.selectById(id, beanClazz);

        handleRestBean(entity);

        return entity;
    }

    @Override
    public boolean exist(Serializable id) {
        T entry = get(id);
        return entry != null;
    }

    @Override
    public DataPage<T> search(SearchInfo searchInfo) {
        if(searchInfo == null){
            searchInfo = new SearchInfo();
        }
        searchInfo.setEntityClazz(this.beanClazz);

        String sqlTemplate = sqlBuilder.buildSqlTemplate(searchInfo);
        log.info(">>>> \n >> searchInfo: {}, \n >> sql: {}", searchInfo, sqlTemplate);

        DataPage<T> dataPage = dataAccess.selectSqlPage(sqlTemplate, searchInfo.getParams(), searchInfo.getPageNo(), searchInfo.getPageSize(), beanClazz);

        handleRestBean(dataPage.getPageData());

        return dataPage;
    }

    @Override
    public List<T> listAll() {
        List<T> result = dataAccess.selectList(new Query<T>(), beanClazz);

        handleRestBean(result);

        return result;
    }

    @Override
    public List<T> listByMap(Map<String,Object> columnMap){
        List<T> resultList = dataAccess.selectByMap(columnMap, beanClazz);

        handleRestBean(resultList);

        return resultList;
    }

    @Override
    public List<T> listByIds(List<Serializable> ids){
        List<T> resultList =  dataAccess.selectBatchIds(ids, this.beanClazz);

        handleRestBean(resultList);

        return resultList;
    }

    /**
     * 根据id获取
     * @param query
     * @return
     */
    @Override
    public List<T> listByQuery(Query<T> query){
        List<T> resultList = dataAccess.selectList(query, this.beanClazz);

        handleRestBean(resultList);

        return resultList;
    }


    /**
     * 根据map查询返回
     * @param columnMap
     * @return
     */
    @Override
    public T getByMap(Map<String, Object> columnMap){
        T entity = dataAccess.selectOneByMap(columnMap, this.beanClazz);

        handleRestBean(entity);

        return entity;
    }

    @Override
    public int countAll(){
        return dataAccess.selectCount(new Query<>(), this.beanClazz);
    }


    /**
     * 根据查询条件返回字典
     * @param paramsList
     * @return
     */
    @Override
    public Map<String, T> dict(List<Map<String, Object>> paramsList){
        if(CollectionUtils.isEmpty(paramsList)){
            return null;
        }

        Map<String, T> dictMap = new HashMap<>();

        for(Map<String, Object> params:paramsList){
            StringBuilder sb = new StringBuilder();

            params.values().forEach(val->{
                sb.append(TypeUtils.castToString(val));
            });

            String key = sb.toString();
            if(dictMap.containsKey(key)){
                continue;
            }

            T entity = getByMap(params);

            dictMap.put(key, entity);
        }

        return dictMap;
    }

    /**
     * 对 bean 里面的 restField 字段进行填值
     * @param obj
     */
    protected void handleRestBean(Object obj){
        if(!this.supportRestBean){
            return;
        }

        if(this.restBeanAccess == null){
            Map map = SpringContextHolder.getApplicationContext().getBeansOfType(RestBeanAccess.class);
            if(CollectionUtils.isEmpty(map)){
                this.supportRestBean = false;
                return;
            }
            this.restBeanAccess = SpringContextHolder.getBean(RestBeanAccess.class);
        }

        restBeanAccess.handle(obj);
    }
}
