package com.efuture.msboot.service.impl;

import com.efuture.msboot.core.bean.EasyBeanWrapper;
import com.efuture.msboot.core.utils.ExceptionUtils;
import com.efuture.msboot.core.utils.ParamCheckUtil;
import com.efuture.msboot.data.DataAccess;
import com.efuture.msboot.data.ViewAccess;
import com.efuture.msboot.data.bean.CURDEnum;
import com.efuture.msboot.data.bean.DataPage;
import com.efuture.msboot.data.bean.Query;
import com.efuture.msboot.data.query.SqlBuilder;
import com.efuture.msboot.data.query.bean.SearchInfo;
import com.efuture.msboot.data.utils.EntityUtils;
import com.efuture.msboot.data.utils.SearchUtils;
import com.efuture.msboot.data.utils.ViewUtils;
import com.efuture.msboot.service.BaseListService;
import com.efuture.msboot.service.annotation.Search;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.PostConstruct;
import java.io.Serializable;
import java.lang.reflect.ParameterizedType;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * Created by wzm on 2019/2/18.
 */
@Slf4j
public abstract class BaseListServiceImpl<T> implements BaseListService<T> {
    @Lazy
    @Autowired(required = false)
    protected DataAccess dataAccess;

    @Lazy
    @Autowired(required = false)
    protected ViewAccess viewAccess;

    @Autowired
    protected SqlBuilder sqlBuilder;

    /**
     * T 对应的 class
     */
    protected Class<T> beanClazz;

    /**
     * search标签
     */
    protected Search searchAnnotation;

    @PostConstruct
    public void init(){
        ParameterizedType type = (ParameterizedType) this.getClass().getGenericSuperclass();
        this.beanClazz = (Class<T>) type.getActualTypeArguments()[0];
        this.searchAnnotation = this.getClass().getAnnotation(Search.class);
    }

    @Override
    public T add(T entity) {
        return add(entity,true);
    }

    @Override
    public T add(T entity, Boolean fleshGet) {
        Assert.notNull(entity, "entry 不能为空");

        EntityUtils.initInsert(entity);

        dataAccess.insert(entity);

        Serializable id = EntityUtils.getIdValue(entity);

        return fleshGet?get(id):entity;
    }

    @Override
    public T update(T entity) {
        return update(entity, true);
    }

    @Override
    public T update(T entity, Boolean fleshGet) {
        Assert.notNull(entity, "entity 不能为空");

        Serializable id = EntityUtils.getIdValue(entity);

        Assert.notNull(id, "id 不能为空");

        EntityUtils.initUpdate(entity);

        dataAccess.updateById(entity);

        return fleshGet?get(id):entity;
    }

    @Override
    public int delete(Serializable id) {
        Assert.notNull(id, "id 不能为空");

        return dataAccess.deleteById(id, beanClazz);
    }

    @Override
    public int batchDelete(List<Serializable> ids) {
        return dataAccess.deleteBatchIds(ids, beanClazz);
    }

    /**
     * 根据query删除
     * @param query
     * @return
     */
    @Override
    public int deleteByQuery(Query<T> query){
        return dataAccess.delete(query, this.beanClazz);
    }

    @Override
    public List<Serializable> batchAdd(List<T> entityList) {
        return batchAdd(entityList, false);
    }

    @Transactional
    @Override
    public List<Serializable> batchAdd(List<T> entityList, Boolean validate) {
        Assert.notEmpty(entityList, "entityList 不能为空");

        List<Serializable> ids = new ArrayList<Serializable>();

        for(T entity:entityList){
            if(validate){
                ParamCheckUtil.validate(entity);
            }

            EntityUtils.initInsert(entity);

            dataAccess.insert(entity);

            ids.add(EntityUtils.getIdValue(entity));
        }

        return ids;
    }


    /**
     * 批量保存
     * @param entityList
     * @return
     */
    @Transactional
    @Override
    public List<T> save(List<T> entityList){
        Assert.notEmpty(entityList, "entityList 不能为空");

        // 先删除
        for(T entity:entityList){
            EasyBeanWrapper beanWrapper = new EasyBeanWrapper(entity);

            // 没有这个字段，默认不处理
            if(!beanWrapper.getPropertiesSet().contains("curd_flag")){
                continue;
            }

            String curd_flag = EntityUtils.getCurdFlag(entity);
            if(CURDEnum.DELETE.toString().equalsIgnoreCase(curd_flag)){
                Serializable idValue = EntityUtils.getIdValue(entity);
                dataAccess.deleteById(idValue, entity.getClass());
            }
        }

        for(Object entity:entityList){
            //更新标识
            String curd_flag = EntityUtils.getCurdFlag(entity);

            //新增
            if(CURDEnum.INSERT.toString().equalsIgnoreCase(curd_flag)){
                EntityUtils.initInsert(entity);

                dataAccess.insert(entity);
            }else if(CURDEnum.UPDATE.toString().equalsIgnoreCase(curd_flag)){
                EntityUtils.initUpdate(entity);

                dataAccess.updateById(entity);
            }else{
                log.warn("无法处理标识 "+curd_flag);
            }
        }

        return entityList;
    }

    @Override
    public T get(Serializable id) {
        Assert.notNull(id, "id 不能为空");

        if(!ViewUtils.isView(this.beanClazz)) {
            return (T) dataAccess.selectById(id, beanClazz);
        }

        return viewAccess.selectById(id, beanClazz);
    }

    @Override
    public boolean exist(Serializable id) {
        T entry = get(id);
        return entry != null;
    }

    @Override
    public DataPage<T> search(SearchInfo searchInfo) {
        if(searchInfo == null){
            searchInfo = new SearchInfo();
        }
        searchInfo.setEntityClazz(this.beanClazz);

        String sqlTemplate = sqlBuilder.buildSqlTemplate(searchInfo);
        log.info(">>>> \n >> searchInfo: {}, \n >> sql: {}", searchInfo, sqlTemplate);

        return dataAccess.selectSqlPage(sqlTemplate, searchInfo.getParams(), searchInfo.getPageNo(), searchInfo.getPageSize(), beanClazz);
    }

    @Override
    public List<T> listAll() {
        return dataAccess.selectList(new Query<T>(), beanClazz);
    }

    @Override
    public List<T> listByMap(Map<String,Object> columnMap){
        return dataAccess.selectByMap(columnMap, beanClazz);
    }

    @Override
    public List<T> listByIds(List<Serializable> ids){
        return dataAccess.selectBatchIds(ids, this.beanClazz);
    }

    /**
     * 根据id获取
     * @param query
     * @return
     */
    @Override
    public List<T> listByQuery(Query<T> query){
        return dataAccess.selectList(query, this.beanClazz);
    }


    /**
     * 根据map查询返回
     * @param columnMap
     * @return
     */
    @Override
    public T getByMap(Map<String, Object> columnMap){
        return dataAccess.selectOneByMap(columnMap, this.beanClazz);
    }

    @Override
    public int countAll(){
        return dataAccess.selectCount(new Query<>(), this.beanClazz);
    }

    /**
     * search接口检查关联查询，防止前端随意关联
     * @param searchInfo
     */
    private void checkSearchRef(SearchInfo searchInfo){
        if(this.searchAnnotation == null){
            ExceptionUtils.raise("未配置search接口");
        }

        if(this.searchAnnotation.refs() == null || this.searchAnnotation.refs().length == 0){
            ExceptionUtils.raise("未配置search.refs接口");
        }

        SearchUtils.checkSearchRef(searchInfo, Arrays.asList(this.searchAnnotation.refs()));
    }

    private Object newBeanOject(){
        try {
            return this.beanClazz.newInstance();
        }catch (Exception e){
            return null;
        }
    }
}
