package com.efuture.msboot.service.impl;

import com.alibaba.fastjson.util.TypeUtils;
import com.efuture.msboot.base.context.SessionContext;
import com.efuture.msboot.base.context.SessionContextHolder;
import com.efuture.msboot.core.bean.EasyBeanWrapper;
import com.efuture.msboot.core.bean.SimpleBeanUtils;
import com.efuture.msboot.core.map.StringObjectMap;
import com.efuture.msboot.core.utils.ExceptionUtils;
import com.efuture.msboot.data.DataAccess;
import com.efuture.msboot.data.ViewAccess;
import com.efuture.msboot.data.bean.CURDEnum;
import com.efuture.msboot.data.bean.DataPage;
import com.efuture.msboot.data.bean.Query;
import com.efuture.msboot.data.bean.SlaveInfo;
import com.efuture.msboot.data.query.SqlBuilder;
import com.efuture.msboot.data.query.bean.SearchInfo;
import com.efuture.msboot.data.utils.EntityUtils;
import com.efuture.msboot.data.utils.SearchUtils;
import com.efuture.msboot.data.utils.SlaveUtils;
import com.efuture.msboot.service.BaseSheetService;
import com.efuture.msboot.service.annotation.Search;
import com.efuture.msboot.service.annotation.Sheet;
import com.efuture.msboot.service.sheet.SheetFlagDefine;
import com.efuture.msboot.service.sheet.SheetGlobalConfig;
import com.efuture.msboot.service.sheet.SheetPropertyDefine;
import com.efuture.msboot.service.sheet.SheetUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import java.io.Serializable;
import java.lang.reflect.ParameterizedType;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

/**
 * Created by wzm on 2019/2/25.
 */
@Slf4j
public class BaseSheetServiceImpl<T> implements BaseSheetService<T>{
    @Lazy
    @Autowired(required = false)
    protected DataAccess dataAccess;

    @Lazy
    @Autowired(required = false)
    protected ViewAccess viewAccess;

    @Autowired
    protected SqlBuilder sqlBuilder;

    @Autowired
    protected SheetGlobalConfig sheetGlobalConfig;

    /**
     * T 对应的 class
     */
    protected Class beanClazz;

    /**
     * 单据标签
     */
    protected Sheet sheetAnnotation;

    /**
     * search标签
     */
    protected Search searchAnnotation;

    /**
     * 单据类型
     */
    protected int sheetType;

    /**
     * 审核后执行exec
     */
    protected boolean execAfterCommit;

    /**
     * 异步执行
     */
    protected boolean asyncExecture = false;

    /**
     * 更新/删除 单据时，是否判断单据状态
     * @return
     */
    protected boolean checkSheetFlag = true;

    /**
     * 单据字段定义
     */
    protected SheetPropertyDefine sheetPropertyDefine;

    /**
     * flag字段定义
     */
    protected SheetFlagDefine sheetFlagDefine;

    public BaseSheetServiceImpl(){
        this(0);
    }

    public BaseSheetServiceImpl(int sheetType){
        this.sheetType = sheetType;
    }

    @PostConstruct
    public void init(){
        ParameterizedType type = (ParameterizedType) this.getClass().getGenericSuperclass();
        this.beanClazz = (Class<T>) type.getActualTypeArguments()[0];
        this.searchAnnotation = this.getClass().getAnnotation(Search.class);
        this.sheetAnnotation = this.getClass().getAnnotation(Sheet.class);

        if(sheetAnnotation != null){
            this.sheetType = sheetAnnotation.sheetType();
            this.execAfterCommit = sheetAnnotation.execAfterCommit();
            this.asyncExecture = sheetAnnotation.asyncExecture();
            this.checkSheetFlag = sheetAnnotation.checkSheetFlag();
        }

        if(this.sheetPropertyDefine == null) {
            this.sheetPropertyDefine = new SheetPropertyDefine();
            SimpleBeanUtils.copyProperties(sheetPropertyDefine, sheetGlobalConfig);
        }

        if(this.sheetFlagDefine == null) {
            this.sheetFlagDefine = new SheetFlagDefine();
            SimpleBeanUtils.copyProperties(sheetFlagDefine, sheetGlobalConfig);
        }

        //自定义配置可以在这里处理
        config();
    }

    @Override
    public T get(Serializable id) {
        Assert.notNull(id, "id 不能为空");

        return get(id, true);
    }

    @Override
    public T get(Serializable id, boolean returnSheetDetails) {
        return (T)viewAccess.selectById(id, beanClazz, returnSheetDetails);
    }

    @Override
    public T getBySheetId(String sheetId) {
        return getBySheetId(sheetId, true);
    }

    @Override
    public T getBySheetId(String sheetId, boolean returnSheetDetails) {
        List<T> resultList = dataAccess.selectByMap(StringObjectMap.create(sheetPropertyDefine.getSheetIdProperty(), sheetId), beanClazz);

        if(CollectionUtils.isEmpty(resultList)){
            return null;
        }

        T entity = (T)resultList.get(0);

        Serializable id = EntityUtils.getIdValue(entity);

        return get(id, returnSheetDetails);
    }

    @Override
    public Object getFlag(Serializable id) {
        Object entity = get(id, false);
        if(entity == null){
            ExceptionUtils.raise("单据id "+id+" 不存在");
        }

        Object value = EntityUtils.getPropertyValue(entity, sheetPropertyDefine.getFlagProperty());
        if(value == null){
            ExceptionUtils.raise("单据id "+id+" flag 不存在");
        }

        return value;
    }

    @Override
    public Object getFlag(String sheetId) {
        Object entity = getBySheetId(sheetId, false);
        if(entity == null){
            ExceptionUtils.raise("单据 "+sheetId+" 不存在");
        }

        Object value = EntityUtils.getPropertyValue(entity, sheetPropertyDefine.getFlagProperty());
        if(value == null){
            ExceptionUtils.raise("单据 "+sheetId+" flag 不存在");
        }

        return value;
    }

    @Override
    public boolean exist(Serializable id) {
        Object entity = get(id, false);
        return entity != null;
    }

    @Override
    public boolean exist(String sheetId) {
        Object entity = getBySheetId(sheetId, false);
        return entity != null;
    }

    @Transactional
    @Override
    public T add(T entity) {
        EntityUtils.initInsert(entity);

        setNewSheetId(entity);

        beforeAdd(entity);

        dataAccess.insert(entity);

        afterAdd(entity);

        return get(EntityUtils.getIdValue(entity));
    }

    @Transactional
    @Override
    public T update(T entity) {
        EntityUtils.initUpdate(entity);

        beforeUpdate(entity);

        //检查单据能否 更新/删除
        checkSheetFlag(EntityUtils.getIdValue(entity));

        dataAccess.updateById(entity);

        afterUpdate(entity);

        return get(EntityUtils.getIdValue(entity));
    }

    @Transactional
    @Override
    public void delete(Serializable id) {
        Assert.notNull(id, "id 不能为空");

        T entity = (T) get(id, false);
        if(entity == null){
            return;
        }

        //检查单据能否 更新/删除
        checkSheetFlag(id);

        beforeDelete((T)entity);

        dataAccess.deleteById(id, beanClazz, true);

        afterDelete((T)entity);
    }

    @Transactional
    @Override
    public T save(T entity, boolean queryDBExist){
        String saveType = CURDEnum.INSERT.toString();

        if(EntityUtils.hasIdValue(entity) && queryDBExist) {
            T sheet = get(EntityUtils.getIdValue(entity), false);
            if (sheet == null) {
                saveType = CURDEnum.INSERT.toString();
            }else{
                saveType = CURDEnum.UPDATE.toString();
            }
        }

        //新增 单据头
        if(CURDEnum.INSERT.toString().equals(saveType)){
            EntityUtils.initInsert(entity);

            setNewSheetId(entity);

            dataAccess.insert(entity);
        }
        //更新 单据头
        else{
            EntityUtils.initUpdate(entity);

            //检查单据能否 更新/删除
            checkSheetFlag(EntityUtils.getIdValue(entity));

            dataAccess.updateById(entity);
        }

        //保存明细
        saveItems(entity);

        return get(EntityUtils.getIdValue(entity));
    }

    @Transactional
    @Override
    public T save(T entity) {
        String saveType = EntityUtils.hasIdValue(entity)? CURDEnum.UPDATE.toString():CURDEnum.INSERT.toString();

        //新增 单据头
        if(CURDEnum.INSERT.toString().equals(saveType)){
            EntityUtils.initInsert(entity);

            setNewSheetId(entity);

            dataAccess.insert(entity);
        }
        //更新 单据头
        else{
            EntityUtils.initUpdate(entity);

            //检查单据能否 更新/删除
            checkSheetFlag(EntityUtils.getIdValue(entity));

            dataAccess.updateById(entity);
        }

        //保存明细
        saveItems(entity);

        return get(EntityUtils.getIdValue(entity));
    }

    @Transactional
    @Override
    public void batchDelete(List<Serializable> ids) {
        Assert.notEmpty(ids, "ids 不能为空");

        for(Serializable id:ids){
            delete(id);
        }
    }

    @Override
    public DataPage<T> search(SearchInfo searchInfo) {
        if(searchInfo == null){
            searchInfo = new SearchInfo();
        }
        searchInfo.setEntityClazz(this.beanClazz);

        String sqlTemplate = sqlBuilder.buildSqlTemplate(searchInfo);
        log.info(">>>> \n >> searchInfo: {}, \n >> sql: {}", searchInfo, sqlTemplate);

        return dataAccess.selectSqlPage(sqlTemplate, searchInfo.getParams(), searchInfo.getPageNo(), searchInfo.getPageSize(), beanClazz);
    }

    @Transactional
    @Override
    public void commit(Serializable id) {
        T entity = get(id);
        if(entity == null){
            ExceptionUtils.raise("单据"+id+" 不存在");
        }

        checkSheetFlag(id);

        beforeCommit(entity);

        EasyBeanWrapper beanWrapper = new EasyBeanWrapper(entity);

        //flag改为已审核
        beanWrapper.setPropertyValueIfExist(sheetPropertyDefine.getFlagProperty(), sheetFlagDefine.getSheet_checked_flag().get(0));

        //设置审核人
        SessionContext sessionContext = SessionContextHolder.get();
        if(sessionContext != null){
            beanWrapper.setPropertyValueIfExist(sheetPropertyDefine.getCheckerProperty(), sessionContext.getUserName());
            beanWrapper.setPropertyValueIfExist(sheetPropertyDefine.getCheckDateProperty(), new Date());
        }

        dataAccess.updateById(entity);

        afterCommit(entity);

        //执行单据
        if(execAfterCommit) {
            execture(id);
        }
    }

    @Override
    public void execture(Serializable id) {
        //同步执行
        if(!asyncExecture) {
            doExecture(id);
        }
        //异步执行
        else{
            log.info(">>异步执行单据[{}]", id);
        }
    }

    @Override
    public String getNewSheetId() {
        return SheetUtils.getNewSheetId(sheetType);
    }

    @Override
    public String getNewSheetId(String shopid) {
        return SheetUtils.getNewSheetId(sheetType, shopid);
    }

    @Override
    public void setNewSheetId(T entity) {
        EasyBeanWrapper easyBeanWrapper = new EasyBeanWrapper(entity);

        if(easyBeanWrapper.getPropertyValueIfExist(sheetPropertyDefine.getSheetIdProperty()) != null){
            return;
        }

        //如果有门店号，根据门店号生成单据号
        String shopid = (String)easyBeanWrapper.getPropertyValueIfExist(sheetPropertyDefine.getShopidProperty());

        String newSheetId = SheetUtils.getNewSheetId(sheetType, shopid);

        easyBeanWrapper.setPropertyValueIfExist(sheetPropertyDefine.getSheetIdProperty(), newSheetId);
    }

    @Override
    public int getSheetType() {
        return this.sheetType;
    }

    /**
     * init , 可以在这个方法内修改单据的配置
     */
    protected void config(){}

    /**
     * 新增前
     * @param entity
     */
    protected void beforeAdd(T entity){}

    /**
     * 新增后
     * @param entity
     */
    protected void afterAdd(T entity){}

    /**
     * 更新前
     * @param entity
     */
    protected void beforeUpdate(T entity){}

    /**
     * 更新后
     * @param entity
     */
    protected void afterUpdate(T entity){}


    /**
     * 删除前
     * @param entity
     */
    protected void beforeDelete(T entity){}

    /**
     * 删除后
     * @param entity
     */
    protected void afterDelete(T entity){}

    /**
     * 提交前
     */
    protected void beforeCommit(T entity){}


    /**
     * 提交前
     */
    protected void afterCommit(T entity){}

    /**
     * 执行审核
     * @param id
     */
    protected void doExecture(Serializable id){}

    /**
     * 保存子表
     * @param entity
     */
    protected void saveItems(T entity){
        List<SlaveInfo> slaveInfoList = SlaveUtils.getSlaveInfoList(this.beanClazz);

        //没有子表
        if(CollectionUtils.isEmpty(slaveInfoList)){
            return;
        }

        BeanWrapper masterWrapper = new BeanWrapperImpl(entity);
        String sheetid = (String)masterWrapper.getPropertyValue(sheetPropertyDefine.getSheetIdProperty());

        for(SlaveInfo slaveInfo:slaveInfoList){
            List slaveDataList = (List)masterWrapper.getPropertyValue(slaveInfo.getFieldName());

            //子表为空
            if(CollectionUtils.isEmpty(slaveDataList)){
                continue;
            }

            List<String> refFields = slaveInfo.getRefFields();
            Integer maxSerialId = 0;
            Boolean hasSerialid = null;

            //优先删除
            for(Object slaveData:slaveDataList){
                EasyBeanWrapper slaveWrapper = new EasyBeanWrapper(slaveData);
                if(!slaveWrapper.getPropertiesSet().contains("curd_flag")){
                    continue;
                }

                //更新标识，默认为更新
                String curd_flag = slaveWrapper.getPropertyValue("curd_flag")==null?"U":String.valueOf(slaveWrapper.getPropertyValue("curd_flag"));
                if(CURDEnum.DELETE.toString().equalsIgnoreCase(curd_flag)){
                    Serializable idValue = EntityUtils.getIdValue(slaveData);
                    dataAccess.deleteById(idValue, slaveInfo.getSlaveClazz());
                }
            }

            for(Object slaveData:slaveDataList){
                EasyBeanWrapper slaveWrapper = new EasyBeanWrapper(slaveData);

                //更新标识
                String curd_flag = EntityUtils.getCurdFlag(slaveData);

                //新增
                if(CURDEnum.INSERT.toString().equalsIgnoreCase(curd_flag)){
                    //子表中是否有 Serialid 字段
                    if(hasSerialid == null) {
                        hasSerialid = slaveWrapper.getPropertiesSet().contains(sheetPropertyDefine.getDetail_SerialId());

                        //获取明细表中序号最大值
                        if(hasSerialid){
                            maxSerialId = getMaxSerialId(sheetid, slaveInfo.getSlaveClazz());
                        }
                    }

                    //设置主从关联字段
                    if(!CollectionUtils.isEmpty(refFields)) {
                        for (String refField : refFields) {
                            slaveWrapper.setPropertyValueIfNull(refField, masterWrapper.getPropertyValue(refField));
                        }
                    }

                    if(hasSerialid) {
                        //设置序列号，如果有这个字段，并且为空时
                        slaveWrapper.setPropertyValueIfNull(sheetPropertyDefine.getDetail_SerialId(), ++maxSerialId);
                    }

                    EntityUtils.initInsert(slaveData);

                    dataAccess.insert(slaveData);
                }else if(CURDEnum.UPDATE.toString().equalsIgnoreCase(curd_flag)){
                    EntityUtils.initUpdate(slaveData);

                    dataAccess.updateById(slaveData);
                }else{
                    log.warn("无法处理标识 "+curd_flag);
                }
            }
        }
    }

    /**
     * 获取单据明细最大序列号
     * @param sheetId
     * @param sheetItemClazz
     * @return
     */
    protected int getMaxSerialId(String sheetId, Class sheetItemClazz){
        Query query = new Query();
        query.eq(sheetPropertyDefine.getSheetIdProperty(), sheetId);

        Object value = dataAccess.selectMax(sheetPropertyDefine.getDetail_SerialId(), query, sheetItemClazz);
        if(value == null){
            return 0;
        }
        return com.alibaba.fastjson.util.TypeUtils.castToInt(value);
    }

    /**
     * search接口检查关联查询，防止前端随意关联
     * @param searchInfo
     */
    private void checkSearchRef(SearchInfo searchInfo){
        if(this.searchAnnotation == null){
            ExceptionUtils.raise("未配置search接口");
        }

        if(this.searchAnnotation.refs() == null || this.searchAnnotation.refs().length == 0){
            ExceptionUtils.raise("未配置search.refs接口");
        }

        SearchUtils.checkSearchRef(searchInfo, Arrays.asList(this.searchAnnotation.refs()));
    }


    /**
     * 检查单据是否能 更新/删除
     * @param id
     * @return
     */
    protected void checkSheetFlag(Serializable id){
        if(id == null || !checkSheetFlag){
            return;
        }

        Object flag = getFlag(id);

        checkSheetFlag(flag);
    }

    /**
     * 检查单据是否能 更新/删除
     * @param flag
     * @return
     */
    public void checkSheetFlag(Object flag){
        if(flag == null){
            return;
        }

        String strFlag = TypeUtils.castToString(flag);

        //可能存在多级审核
        if(!CollectionUtils.isEmpty(sheetFlagDefine.getSheet_checked_flag())){
            for(Object value:sheetFlagDefine.getSheet_checked_flag()){
                if(TypeUtils.castToString(value).equals(strFlag)){
                    ExceptionUtils.raise("单据已审核");
                }
            }
        }

        if(TypeUtils.castToString(sheetFlagDefine.getSheet_cancel_flag()).equals(strFlag)){
            ExceptionUtils.raise("单据已取消");
        }

        if(TypeUtils.castToString(sheetFlagDefine.getSheet_complete_flag()).equals(strFlag)){
            ExceptionUtils.raise("单据已完成");
        }
    }
}
