package com.efuture.starter.config.boot;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

import javax.sql.DataSource;

import org.apache.commons.lang.StringUtils;
import org.apache.ibatis.plugin.Interceptor;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;

import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.druid.pool.DruidDataSourceFactory;

public abstract class DBConfigAbstract
{
    @Autowired
    protected Environment env;

    @Autowired
    protected ResourceLoader resourceLoader = new DefaultResourceLoader();

    protected String getValue(String key, String defaultVal)
    {
        String val = env.getProperty(key);

        if (val == null || val.isEmpty()) {
            return defaultVal;
        }
        else {
            return val;
        }
    }

    protected DataSource getDs(String dbkey)
    {
        Properties props = new Properties();
        String prestr = "efuture.dataSource." + dbkey;
        props.put("driverClassName", env.getProperty(prestr + ".driverClassName"));
        props.put("url", env.getProperty(prestr + ".url"));
        props.put("username", env.getProperty(prestr + ".username"));
        props.put("password", env.getProperty(prestr + ".password"));
        props.put("maxActive", getValue(prestr + ".maxActive", "50"));
        props.put("maxWait", getValue(prestr + ".maxWaitMillis", "60000"));
        props.put("maxIdle", getValue(prestr + ".maxIdle", "20"));
        props.put("minIdle", getValue(prestr + ".minIdle", "3"));
        props.put("removeAbandoned", getValue(prestr + ".removeAbandoned", "true"));
        props.put("removeAbandonedTimeout", getValue(prestr + ".removeAbandonedTimeout", "600"));
        props.put("validationQuery", env.getProperty(prestr + ".validationQuery"));

        try {
            DruidDataSource datasource = (DruidDataSource) DruidDataSourceFactory.createDataSource(props);
            datasource.setDbType(env.getProperty(prestr + ".dbtype"));
            return datasource;
        }
        catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    // *INDENT-OFF*
    protected void setSqlSessionResources(SqlSessionFactoryBean bean, String dbtype, DataSource db, String... mappers) throws IOException
    {
        bean.setDataSource(db);        
        setSqlSessionProperties(bean,dbtype,null);
        setSqlSessionAliases(bean,null);
        setSqlSessionPlugins(bean,dbtype,null);
        setSqlSessionMappers(bean,mappers);
    }

    protected void setSqlSessionProperties(SqlSessionFactoryBean bean, String dbtype, String filename) throws IOException 
    {
        // <properties resource="sqlfunc_mysql.properties"></properties>
        Properties prop = new Properties();
        InputStream in = resourceLoader.getClassLoader().getResourceAsStream(StringUtils.isEmpty(filename)?"sqlfunc_" + dbtype + ".properties":filename);
        prop.load(in);
        bean.setConfigurationProperties(prop);
    }

    protected void setSqlSessionAliases(SqlSessionFactoryBean bean, List<Class<?>> list) 
    {
        /*
        <typeAliases>
           <typeAlias type="com.efuture.omd.storage.mybatis.HashMapCase" alias="MapCase" />
        </typeAliases>
       */
       if (list == null) bean.setTypeAliases(new Class<?>[] { com.efuture.ocp.common.util.HashMapCase.class });
       else {
           list.add(com.efuture.ocp.common.util.HashMapCase.class);
           bean.setTypeAliases(list.toArray(new Class<?>[list.size()]));
       }
    }

    protected void setSqlSessionPlugins(SqlSessionFactoryBean bean, String dbtype, List<Interceptor> list) 
    {
        /*
        <plugins>
          <plugin interceptor="com.efuture.ocp.common.filter.ExecutorWrapper" >
              <property name="dbType" value="${dbtype}"/>
          </plugin>
          <plugin interceptor="com.efuture.ocp.common.filter.LogMybatisPlugin" >
          </plugin>
          <plugin interceptor="com.github.pagehelper.PageHelper">
              <property name="offsetAsPageNum" value="false"/>
              <property name="rowBoundsWithCount" value="true"/>
              <property name="pageSizeZero" value="false"/>
          </plugin>
        </plugins>
        */
        Interceptor[] plugins = new Interceptor[3];
        Properties p0 = new Properties();
        p0.setProperty("dbType", dbtype);
        plugins[0] = new com.efuture.ocp.common.filter.ExecutorWrapper();
        plugins[0].setProperties(p0);
        plugins[1] = new com.efuture.ocp.common.filter.LogMybatisPlugin();
        Properties p2 = new Properties();
        p2.setProperty("offsetAsPageNum", "false");
        p2.setProperty("rowBoundsWithCount", "true");
        p2.setProperty("pageSizeZero", "false");
        plugins[2] = new com.github.pagehelper.PageHelper();
        plugins[2].setProperties(p2);
        bean.setPlugins(plugins);
    }

    protected void setSqlSessionMappers(SqlSessionFactoryBean bean, String... mappers) throws IOException 
    {
        /*
        <mappers>
          <mapper resource="beanmapper/mybatis-crud.xml"/>
        </mappers>
        */
        // classpath*:beanmapper/**/*.xml
        ResourcePatternResolver resourceResolver = null;
        List<Resource> resources = new ArrayList<Resource>();
        resources.add(resourceLoader.getResource("classpath:mybatis/mybatis-crud.xml"));
        for (int i = 0; mappers != null && i < mappers.length; i++) {
            if (mappers[i].indexOf("*") > 0) {
                if (resourceResolver == null) {
                    resourceResolver = new PathMatchingResourcePatternResolver();
                }

                Resource[] res = resourceResolver.getResources(mappers[i].indexOf(":") > 0 ? mappers[i] : "classpath*:" + mappers[i]);
                resources.addAll(Arrays.asList(res));
            }
            else {
                resources.add(resourceLoader.getResource(mappers[i].indexOf(":") > 0 ? mappers[i] : "classpath:" + mappers[i]));
            }
        }
        bean.setMapperLocations(resources.toArray(new Resource[resources.size()]));
    }
}
