package com.efuture.ocp.common.Cache;

import java.util.Date;

import org.apache.log4j.Logger;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import com.alibaba.fastjson.JSONObject;
import com.efuture.ocp.common.autorollback.AutoRollbackAspect;
import com.efuture.ocp.common.exception.ServiceException;
import com.efuture.ocp.common.language.MessageSourceHelper;
import com.efuture.ocp.common.language.ResponseCode;
import com.efuture.ocp.common.util.CacheUtils;

@Aspect
@Component
public class CheckRepeatAspect {
	private static final Logger logger = Logger.getLogger(AutoRollbackAspect.class);
	
	/**
	 * 根据方法名和指定的参数序号和参数值来获取要判断重复的key
	 * @param point
	 * @param around
	 * @return
	 */
	private String getKey(ProceedingJoinPoint point, CheckRepeatAnnotation around) {
		String keyname = around.KeyName();
		int argindex = around.KeyArgIndex();
		String className = point.getTarget().getClass().getSimpleName();
		String methodName = point.getSignature().getName();
		Object[] args = point.getArgs();
		StringBuffer sb = new StringBuffer();
		if (argindex != -1 && args.length > argindex) {
			Object arg = args[argindex];
			JSONObject jsonarg;
			if (arg instanceof JSONObject) {
				jsonarg = (JSONObject) arg;

			} else {
				jsonarg = (JSONObject) JSONObject.toJSON(arg);
			}
			if (StringUtils.isEmpty(keyname)) {
				sb.append("_").append(JSONObject.toJSON(arg));
			} else {
				String[] keys = keyname.split(",");
				StringBuffer sbkey = new StringBuffer();
				for (String key : keys) {
					if (key != null) {
						Object value = jsonarg.get(key);
						if (value != null) {
							sbkey.append("-").append(key).append(":").append(value.toString());
						}
					}
				}
				if (sbkey.length() > 0) {
					sb.append("_").append(sbkey.toString());
				} else {
					sb.append("_").append(jsonarg.toJSONString());
				}
			}
		} else {
			for (Object arg : args) {
				if (arg != null) {
					sb.append("_").append(JSONObject.toJSON(arg));
				}
			}
		}

		return className.concat("_").concat(methodName).concat(sb.toString());
	}
	
	/**
	 * 阻止重复调用
	 * 1.使用的mem缓存来控制是否重复
	 * 2.流程
	 * call method ->获取要判断重复的key-->检查缓存中是否存在
	 * -->存在抛出异常阻止调用--->不存在，执行原本方法-->删除缓存
	 * @param point
	 * @param around
	 * @return
	 * @throws Throwable
	 */
	@Around("@annotation(around)")
	public Object round(ProceedingJoinPoint point, CheckRepeatAnnotation around) throws Throwable {

		String className = point.getTarget().getClass().getSimpleName();
		String methodName = point.getSignature().getName();

		if (logger.isDebugEnabled()) {
			logger.debug(MessageSourceHelper.formatMessage("开始检查重复调用:className->{0};methodName->{1}", className,
					methodName));
		}

		String cacheKey = getKey(point, around);
		// 用CAS put数据，如果成功，说明没有key存在，可以执行，
		// 否则则表示已经存在key了
		boolean exist = checkexist(cacheKey);
		if (exist) {
			throw new ServiceException(ResponseCode.CALLREPEAT, "重复调用");
		}
		Object result = null;
		try {
			result = point.proceed();
		} catch (Throwable e) {
			CacheUtils.getCacheUtils().deleteData(cacheKey);
			throw e;
		} finally {
			CacheUtils.getCacheUtils().deleteData(cacheKey);
		}
		return result;

	}
	
	/**
	 * 在缓存中检查key是否存在
	 * 改为使用add命令判断key是否存在
	 * @param key
	 * @return
	 */
	public boolean checkexist(String key) {
		boolean addSuccess = false;
		try {
			addSuccess = CacheUtils.getCacheUtils().addData(key, new Date(), 10*CacheUtils.CacheTimeOut.Min);
		} catch (Exception e) {
			addSuccess = true;
		}
		
		if(addSuccess){
			return false;
		}
		else
		{
			return true;
		}
//		Object rtn = CacheUtils.getCacheUtils().getData(key);
//		if (logger.isDebugEnabled()) {
//			logger.debug("***********************************************************************");
//			logger.debug("key-->" + key);
//			logger.debug("rtn-->" + rtn);
//		}
//		if (rtn != null) {
//			return true;
//		} else {
//			// Long value = UniqueID.getUniqueIDArray(10).get(4);
//			CacheUtils.getCacheUtils().putData(key, "S", CacheUtils.CacheTimeOut.HalfHour);
//			long cas = CacheUtils.getCacheUtils().getCasValue(key);
//			boolean ok = CacheUtils.getCacheUtils().putbyCas(key, cas, CacheUtils.CacheTimeOut.HalfHour, cas);
//			if (logger.isDebugEnabled()) {
//				logger.debug("cas-->" + cas);
//				logger.debug("putisok-->" + ok);
//			}
//			if (!ok) {
//				if (logger.isDebugEnabled()) {
//					logger.debug("***********************************************************************");
//				}
//				return false;
//			} else {
//				rtn = CacheUtils.getCacheUtils().getData(key);
//				if (logger.isDebugEnabled()) {
//					logger.debug("newrtn-->" + rtn);
//					logger.debug("***********************************************************************");
//				}
//
//				if (rtn.toString().equalsIgnoreCase(String.valueOf(cas))) {
//					return false;
//				} else {
//					return true;
//				}
//			}
//		}
	}
}
