package com.efuture.ocp.common.component;

import com.efuture.ocp.common.entity.AbstractEntityBean;
import com.efuture.ocp.common.entity.OperationLogBean;
import com.efuture.ocp.common.util.BatchInsService;
import com.efuture.ocp.common.util.UniqueID;
import com.efuture.omd.storage.FMybatisTemplate;
import com.efuture.omd.storage.Virtual;
import org.springframework.data.annotation.Transient;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;
import java.util.stream.Collectors;

public class OperationLogServiceImpl extends BasicComponentService<OperationLogBean> implements OperationLogService
{
    private String modulecol = "billmoduleid,moduleid";

    private static final ThreadLocal<Map<String, Object>> logList = new ThreadLocal<Map<String, Object>>();

    public static ThreadLocal<Map<String, Object>> getLogList()
    {
        return logList;
    }

    public OperationLogServiceImpl()
    {
        super();
        this.setStorageOperation_other("GlobalStorageOperation");
    }

    /**
     * 单表bean新增，记录新增bean业务key
     * 单表bean修改，记录修改字段，每个字段记录一行日志
     * 单表bean删除，记录删除bean的业务主键
     * 主从bean新增，记录新增主表bean业务key
     * 主从bean修改，记录主表的修改字段，每个字段记录一行日志，从表list，每一行bean的修改字段拼起来合并记录一行日志
     * 主从bean删除，记录新增主表bean业务key
     */

    @Override
    public void insOperLog(String oper, String usercode, String username, String moduleid, AbstractEntityBean originalbean, AbstractEntityBean newbean,
                           String type) throws Exception
    {
        // 记录更新日志
        //模块号优先取传进来，如传进来为空，则从bean中再取一次，模块号为空，则不记录日志
        List<OperationLogBean> list = new ArrayList<OperationLogBean>();
        boolean logflag = false;//默认不记录日志
        Class clazz = originalbean.getClass();
        Object islog = getBeanDefineKey(clazz, "ISLOG");

        if (StringUtils.isEmpty(islog) || !(Boolean) islog) {
            return;
        }

        if (StringUtils.isEmpty(moduleid)) {
            moduleid = getModuleId(originalbean);
        }

        if (StringUtils.isEmpty(moduleid)) {
            return;
        }

        if (StringUtils.isEmpty(type)) {
            type = BEANTYPE.MASTER;
        }

        //key按ent_id+单号
        String key = getBusinessKeyValue(originalbean);

        if (checkIsLog(key, oper, type)) {
            return;
        }

        if (OP.UPDATE.equalsIgnoreCase(oper)) {
            //仅记录发生变化的字段
            list = compareObject(originalbean, newbean, type);
        }
        else if (OP.INSERT.equalsIgnoreCase(oper) || OP.DELETE.equalsIgnoreCase(oper)) {
            //新增/删除仅记录业务主键
            OperationLogBean log = getMasterLog(oper, originalbean);
            list.add(log);
        }

        String masterkey = getMasterSlaveKey(clazz);
        String idkey = getIdKey(clazz);
        Object businesscode = getFieldValue(clazz, originalbean, masterkey);
        Object businesskey = getFieldValue(clazz, originalbean, idkey);

        if (!StringUtils.isEmpty(list) && list.size() > 0) {
            batchHandleLog(list, originalbean.getEnt_id(), moduleid, usercode, username, oper, businesscode, businesskey);
        }

        //执行都需要记录单对应的模块号，避免主表无修改，子表发生修改时找不到模块号,key=bean+ent_id+businesscode+oper
        Map<String, Object> map = new HashMap<>();

        if (!StringUtils.isEmpty(logList.get())) {
            map = logList.get();
        }

        map.put(originalbean.getEnt_id() + "-" + businesscode + "-" + oper, moduleid);
        logList.set(map);
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public void batchHandleLog(List<OperationLogBean> list, long ent_id, String billmoduleid, String usercode, String username, String oper, Object businesscode,
                               Object businesskey) throws Exception
    {
        list.forEach(e -> {
            if (!StringUtils.isEmpty(businesscode))
            {
                e.setBussinesscode(businesscode.toString());
            }
            if (!StringUtils.isEmpty(businesskey))
            {
                e.setBussinesskey(businesskey.toString());
            }

            e.setEnt_id(ent_id);
            e.setBillmoduleid(billmoduleid);
            e.setOperdate(new Date());
            e.setUsercode(usercode);
            e.setUsername(username);
            e.setPh_key(UniqueID.getUniqueID());
        });
        BatchInsService batchInsSrv = new BatchInsService();
        batchInsSrv.batchInsert(getGlobalStorageOperations(FMybatisTemplate.class), OperationLogBean.class, list);
    }

    public List<OperationLogBean> compareObject(Object oldBean, Object newBean, String type)
    {
        List<OperationLogBean> list = new ArrayList<OperationLogBean>();

        try {
            Class clazz = oldBean.getClass();
            Field[] fields = clazz.getDeclaredFields();
            String memo = "";

            for (Field field : fields) {
                if (!check(field, clazz)) {
                    continue;
                }

                Map<String, Object> map = getDiffField(clazz, field, oldBean, newBean);

                if (StringUtils.isEmpty(map)) {
                    continue;
                }

                if (BEANTYPE.MASTER.equals(type)) {
                    OperationLogBean log = new OperationLogBean();
                    log.setOpertype("U");
                    log.setCode(field.getName());
                    log.setOldvalue(StringUtils.isEmpty(map.get("oldvalue")) ? "" : map.get("oldvalue").toString());
                    log.setNewvalue(StringUtils.isEmpty(map.get("newvalue")) ? "" : map.get("newvalue").toString());
                    log.setMemo(map.get("memo").toString());
                    list.add(log);
                }
                else {
                    memo += map.get("memo") + ";";
                }
            }

            if (BEANTYPE.SLAVE.equals(type)) {
                OperationLogBean log = new OperationLogBean();
                log.setOpertype("U");
                log.setCode(AbstractEntityBean.fetchAnnotationTableName(clazz));
                log.setMemo(memo);
                list.add(log);
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }

        return list;
    }

    /**
     * 获取从表的修改项，明细行直接把所有修改拼接
     *
     * @param oldBean
     * @param newBean
     * @return
     * @throws Exception
     */
    public String getDiff(Object oldBean, Object newBean) throws Exception
    {
        Class clazz = oldBean.getClass();
        Field[] fields = clazz.getDeclaredFields();
        String str = "";

        for (Field field : fields) {
            if (!check(field, clazz)) {
                continue;
            }

            Map<String, Object> map = getDiffField(clazz, field, oldBean, newBean);

            if (StringUtils.isEmpty(map)) {
                continue;
            }

            str += map.get("memo") + ";";
        }

        return str;
    }

    /**
     * 获取某个字段的修改日志
     *
     * @param clazz
     * @param field
     * @param oldBean
     * @param newBean
     * @return
     * @throws Exception
     */
    public Map<String, Object> getDiffField(Class clazz, Field field, Object oldBean, Object newBean) throws Exception
    {
        Object o1 = getFieldValue(clazz, oldBean, field.getName());
        Object o2 = getFieldValue(clazz, newBean, field.getName());

        if (o1 == null && o2 == null) {
            return null;
        }

        if (StringUtils.isEmpty(o1) && StringUtils.isEmpty(o2)) {
            return null;
        }

        if (field.getGenericType().toString().equals("class java.util.Date")) {
            Date d1 = null;
            Date d2 = null;

            if (o1 != null) {
                d1 = (Date) o1;
            }

            if (o2 != null) {
                d2 = (Date) o2;
            }

            if (!StringUtils.isEmpty(d1) && !StringUtils.isEmpty(d2) && d1.compareTo(d2) == 0) {
                return null;
            }
        }
        else {
            if (!StringUtils.isEmpty(o1) && (!StringUtils.isEmpty(o2)) && o1.equals(o2)) {
                return null;
            }
        }

        String memo = "字段名称" + field.getName() + ",旧值:" + o1 + ",新值:" + o2;
        Map<String, Object> map = new HashMap<String, Object>();
        map.put("oldvalue", o1);
        map.put("newvalue", o2);
        map.put("memo", memo);
        return map;
    }

    /**
     * 校验字段是否需要记录日志
     *
     * @param field
     * @param cl
     * @return
     * @throws Exception
     */
    public boolean check(Field field, Class<?> cl) throws Exception
    {
        Object ignorecols = getBeanDefineKey(cl, "IGNORECOLS");

        if (field.toString().contains("static final")) {
            return false;
        }

        if (!StringUtils.isEmpty(ignorecols) && ignorecols.toString().contains(field.getName())) {
            return false;
        }

        if (!StringUtils.isEmpty(field.getAnnotation(Transient.class))) {
            return false;
        }

        if (!StringUtils.isEmpty(field.getAnnotation(Virtual.class))) {
            return false;
        }

        return true;
    }

    /**
     * 当接口中未传入模块号时，取bean中的模块号,bean没有取缓存
     *
     * @param bean
     * @return
     * @throws Exception
     */
    public String getModuleId(Object bean) throws Exception
    {
        AbstractEntityBean entity = (AbstractEntityBean) bean;
        Class clazz = entity.getClass();
        String[] cols = modulecol.split(",");

        for (String col : cols) {
            try {
                Object object = getFieldValue(clazz, bean, col);

                if (!StringUtils.isEmpty(object)) {
                    return object.toString();
                }
            }
            catch (Exception e) {
            }
        }

        Map<String, Object> map = logList.get();

        if (!StringUtils.isEmpty(map)) {
            String filters = getBusinessKeyValue(entity);
            map = map.entrySet().stream()
                  .filter((e) -> e.getKey().indexOf(filters) >= 0)
                  .collect(Collectors.toMap(
                               (e) -> (String) e.getKey(),
                               (e) -> e.getValue()
                           ));
        }

        if (!StringUtils.isEmpty(map) && map.size() > 0) {
            String mapkey = map.keySet().iterator().next();
            return map.get(mapkey).toString();
        }
        else {
            return null;
        }
    }

    /**
     * 获取bean单个字段的值
     *
     * @param clazz
     * @param bean
     * @param col
     * @return
     * @throws Exception
     */
    public Object getFieldValue(Class clazz, Object bean, String col) throws Exception
    {
        PropertyDescriptor pd = new PropertyDescriptor(col, clazz);
        Method getMethod = pd.getReadMethod();
        Object o1 = getMethod.invoke(bean);
        return o1;
    }

    /**
     * 获取唯一键的拼接
     *
     * @param bean
     * @return
     * @throws Exception
     */
    public String getUniqueKeyValue(Object bean) throws Exception
    {
        Class clazz = bean.getClass();
        String[] uks = getUniqueKeys(clazz);
        String memo = "";

        for (String uk : uks) {
            String[] k = uk.split(",");

            for (String s : k) {
                Object o1 = getFieldValue(clazz, bean, s);
                memo += s + ":" + o1 + ";";
            }
        }

        return memo;
    }

    /**
     * 主表新增删除时，仅记录唯一键的值
     *
     * @param oper
     * @param bean
     * @return
     * @throws Exception
     */
    public OperationLogBean getMasterLog(String oper, Object bean) throws Exception
    {
        //获取主表bean配置的UNIQUE_KEYS
        Class clazz = bean.getClass();
        OperationLogBean log = new OperationLogBean();
        log.setOpertype(oper);
        String businesscode = getMasterSlaveKey(clazz);
        log.setCode(businesscode);

        if (OP.INSERT.equalsIgnoreCase(oper)) {
            log.setNewvalue(getFieldValue(clazz, bean, businesscode).toString());
        }
        else if (OP.DELETE.equalsIgnoreCase(oper)) {
            log.setOldvalue(getFieldValue(clazz, bean, businesscode).toString());
        }

        String memo = getUniqueKeyValue(bean);
        log.setMemo(memo);
        return log;
    }

    /**
     * 检查主表是否已经记录新增或者删除日志
     *
     * @return
     */
    public boolean checkIsLog(String key, String oper, String type)
    {
        if (BEANTYPE.SLAVE.equals(type) && (OP.INSERT.equals(oper) || OP.DELETE.equals(oper))) {
            String filters = key + "-" + oper;
            Map<String, Object> map = logList.get();

            if (!StringUtils.isEmpty(map)) {
                map = map.entrySet().stream()
                      .filter((e) -> e.getKey().equals(filters))
                      .collect(Collectors.toMap(
                                   (e) -> (String) e.getKey(),
                                   (e) -> e.getValue()
                               ));

                if (!StringUtils.isEmpty(map) && map.size() > 0) {
                    return true;
                }
            }
        }

        return false;
    }

    public String getBusinessKeyValue(AbstractEntityBean originalbean) throws Exception
    {
        String businesscode = getMasterSlaveKey(originalbean.getClass());
        return originalbean.getEnt_id() + "-" + getFieldValue(originalbean.getClass(), originalbean, businesscode).toString();
    }
}
