/**
 *
 */
package com.efuture.ocp.common.util;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.Period;
import java.util.*;

/**
 * @author wudong
 * @description 日期工具类
 */
public class DateUtils extends org.apache.commons.lang3.time.DateUtils
{

    static SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

    public static String formatDateByFormat(Date date, String format)
    {
        SimpleDateFormat sdfhms = new SimpleDateFormat(format);
        return sdfhms.format(date);
    }

    /**
     * 得到当前日期 不含时间
     *
     * @return date
     */
    public static Date gettoday()
    {
        return trunctoday(new Date());
    }

    /**
     * 截断日期到天
     *
     * @param d
     * @return Date
     */
    public static Date trunctoday(Date d)
    {
        return truncate(d, Calendar.DAY_OF_MONTH);
    }

    /**
     * 得到两个日期之间的天数（包含最后一天）
     *
     * @param startDate
     * @param endDate
     * @return 天数
     */
    public static long getDaysBetween(Date startDate, Date endDate)
    {
        return (trunctoday(startDate).getTime() - trunctoday(endDate).getTime())
               / (1000 * 60 * 60 * 24) + 1;
    }

    /**
     * 返回日期是星期几
     *
     * @param date
     * @return
     */
    public static long getDayOfWeek(Date date)
    {
        return toCalendar(date).get(Calendar.DAY_OF_WEEK);
    }

    /**
     * 返回当月的第一天
     *
     * @param date
     * @return 日期
     */
    public static Date getFirstDayOfMonth(Date date)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH), 1);
        return calendar.getTime();
    }

    /**
     * 获取指定日期当月的最后一天
     * @param date
     * @return
     */
    public static Date getLastDayOfMonth(Date date)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.MONTH, 1);
        calendar.set(Calendar.DAY_OF_MONTH, 0);
        return calendar.getTime();
    }

    /**
     * 返回当年的第一天
     *
     * @param date
     * @return 日期
     */
    public static Date getFirstDayOfYear(Date date)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(calendar.get(Calendar.YEAR), 0, 1);
        return calendar.getTime();
    }

    /**
     * 获取指定日期当年的最后一天
     * @param date
     * @return
     */
    public static Date getLastDayOfYear(Date date)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.YEAR, 1);
        calendar.set(Calendar.DAY_OF_YEAR, 0);
        return calendar.getTime();
    }

    public static String getWeek(Date date)
    {
        return String.valueOf(getDayOfWeek(date) - 1);
    }

    /**
     * 得到指定日期对应周的周一
     *
     * @param date
     * @return
     */
    public static Date getThisWeekMonday(Date date)
    {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        // 获得当前日期是一个星期的第几天
        int dayWeek = cal.get(Calendar.DAY_OF_WEEK);

        if (1 == dayWeek) {
            cal.add(Calendar.DAY_OF_MONTH, -1);
        }

        // 设置一个星期的第一天，按中国的习惯一个星期的第一天是星期一
        cal.setFirstDayOfWeek(Calendar.MONDAY);
        // 获得当前日期是一个星期的第几天
        int day = cal.get(Calendar.DAY_OF_WEEK);
        // 根据日历的规则，给当前日期减去星期几与一个星期第一天的差值
        cal.add(Calendar.DATE, cal.getFirstDayOfWeek() - day);
        return cal.getTime();
    }

    /**
     * 得到指定日期对应下周的周一
     *
     * @param date
     * @return
     */
    public static Date getNextWeekMonday(Date date)
    {
        Calendar cal = Calendar.getInstance();
        cal.setTime(getThisWeekMonday(date));
        cal.add(Calendar.DATE, 7);
        return cal.getTime();
    }

    public static int getQuarterFromMonth(int month)
    {
        int quarter = 0;

        if (month <= 2) {
            quarter = 1;
        }
        else if (month >= 3 && month <= 5) {
            quarter = 2;
        }
        else if (month >= 6 && month <= 8) {
            quarter = 3;
        }
        else if (month >= 9 && month <= 11) {
            quarter = 4;
        }
        else {
            quarter = 0;
        }

        return quarter;
    }

    public static int getStartMonthFromQuarter(int quarter)
    {
        int startMonth = 0;//从1月开始，1月为0

        switch (quarter) {
            case 1:
                startMonth = 0;
                break;

            case 2:
                startMonth = 3;
                break;

            case 3:
                startMonth = 6;
                break;

            case 4:
                startMonth = 9;
                break;

            default:
                break;
        }

        return startMonth;
    }

    public static Date addQuarterStartDay(final Date date, final int amount)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH);
        int quarter = getQuarterFromMonth(month);
        int nextQuarter = quarter + amount;
        int y = nextQuarter / 4;
        int i = nextQuarter % 4;

        //        System.out.println("nextQuarter："+nextQuarter);
        //        System.out.println("y："+y);
        //        System.out.println("i："+i);
        if (nextQuarter == 0) {
            nextQuarter = 4;
            y = -1;
            year = year + y;
        }

        if (y >= 1 && nextQuarter > 4) {
            nextQuarter = i;
            year = year + y;
        }

        month = getStartMonthFromQuarter(nextQuarter);
        calendar.set(year, month, 1);
        return calendar.getTime();
    }

    public static String getTodayStr()
    {
        return sdf.format(gettoday());
    }

    public static String getDateTimeStr(Date date)
    {
        SimpleDateFormat sdfhms = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return sdfhms.format(date);
    }

    public static String getNowStr()
    {
        return getDateTimeStr(new Date());
    }

    public static Date getFirstDay()
    {
        Calendar calendar = Calendar.getInstance();
        calendar.set(1900, 0, 1, 0, 0, 0);
        return calendar.getTime();
    }

    /**
     * 获取日期属于当月的第几天
     * */
    public static int getDayCurrentMonth(Date date)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        int days = calendar.get(Calendar.DATE);
        return days;
    }

    /**
     * 判断日期是否月末
     * */
    public static boolean isLastDayOfMonth(Date date)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.DATE, (calendar.get(Calendar.DATE) + 1));

        if (calendar.get(Calendar.DAY_OF_MONTH) == 1) {
            return true;
        }

        return false;
    }

    public static int[] getDiff(Date start, Date end)
    {
        //Calendar.MONTH 初始值为0，所以我们用它来表示日历月份时需要加1
        SimpleDateFormat sdfmd = new SimpleDateFormat("MMdd");
        SimpleDateFormat sdfy = new SimpleDateFormat("yyyy");

        //2022-01-29-2022-02-28 为1个月
        //2022-02-01-2022-02-28 为1个月
        //2020-01-29-2020-02-29 为1个月
        //2020-02-29-2021-02-28 三年
        if ("0228".equals(sdfmd.format(end)) && PrecisionUtils.mod(Integer.parseInt(sdfy.format(end)), 4) != 0) {
            if ("0229".equals(sdfmd.format(start))) {
                int[] diff = new int[2];
                diff[0] = (Integer.parseInt(sdfy.format(end)) - Integer.parseInt(sdfy.format(start))) * 12;
                diff[1] = 0;
                return diff;
            }
            else if ("0129".equals(sdfmd.format(start))) {
                int[] diff = new int[2];
                diff[0] = (Integer.parseInt(sdfy.format(end)) - Integer.parseInt(sdfy.format(start))) * 12 + 1;
                diff[1] = 0;
                return diff;
            }
            else if ("0201".equals(sdfmd.format(start))) {
                int[] diff = new int[2];
                diff[0] = (Integer.parseInt(sdfy.format(end)) - Integer.parseInt(sdfy.format(start))) * 12 + 1;
                diff[1] = 0;
                return diff;
            }
            else {
                return getDiffBaisc(start, end);
            }
        }
        else {
            return getDiffBaisc(start, end);
        }
    }
    public static int[] getDiffBaisc(Date start, Date end)
    {
        end = addDays(end, 1);
        Calendar ca = Calendar.getInstance();
        ca.setTime(start);
        int year_start = ca.get(Calendar.YEAR);
        int month_start = ca.get(Calendar.MONTH) + 1;
        int day_start = ca.get(Calendar.DAY_OF_MONTH);
        ca.setTime(end);
        int year_end = ca.get(Calendar.YEAR);
        int month_end = ca.get(Calendar.MONTH) + 1;
        int day_end = ca.get(Calendar.DAY_OF_MONTH);
        LocalDate startDate = LocalDate.of(year_start, month_start, day_start);
        LocalDate endDate = LocalDate.of(year_end, month_end, day_end);
        int[] diff = getDiff(startDate, endDate);
        return diff;
    }
    public static int[] getDiff(LocalDate start, LocalDate end)
    {
        if (!start.isBefore(end)) {
            throw new IllegalArgumentException("Start must not be before end.");
        }

        Period period = Period.between(start, end);
        int years = period.getYears();
        int months = period.getMonths();
        int days = period.getDays();
        return new int[] {years * 12 + months, days};
    }
    // 计算两个日期相差的月份数
    public static double betweenMonths(Date endDate, Date startDate)
    {
        endDate = addDays(endDate, 1);
        Calendar cal = Calendar.getInstance();
        cal.setTime(startDate);
        int startDayOfMonth = cal.get(Calendar.DAY_OF_MONTH);
        int startMonth = cal.get(Calendar.MONTH);
        int startYear = cal.get(Calendar.YEAR);
        cal.setTime(endDate);
        int endDayOfMonth = cal.get(Calendar.DAY_OF_MONTH);
        int endMonth = cal.get(Calendar.MONTH);
        int endYear = cal.get(Calendar.YEAR);
        int diffMonths = endMonth - startMonth;
        int diffYears = endYear - startYear;
        int diffDays = endDayOfMonth - startDayOfMonth;
        return (diffYears * 12) + diffMonths + diffDays / 31.0;
    }

    /**
     * 获取两个日期之间的所有日期
     *
     * @param startTime 开始日期
     * @param endTime   结束日期
     * @return
     */
    public static List<String> getDays(String startTime, String endTime)
    {
        // 返回的日期集合
        List<String> days = new ArrayList<String>();

        try {
            Date start = sdf.parse(startTime);
            Date end = sdf.parse(endTime);
            Calendar tempStart = Calendar.getInstance();
            tempStart.setTime(start);
            Calendar tempEnd = Calendar.getInstance();
            tempEnd.setTime(end);
            tempEnd.add(Calendar.DATE, +1);// 日期加1(包含结束)

            while (tempStart.before(tempEnd)) {
                days.add(sdf.format(tempStart.getTime()));
                tempStart.add(Calendar.DAY_OF_YEAR, 1);
            }
        }
        catch (ParseException e) {
            e.printStackTrace();
        }

        return days;
    }

    /**
     * 获取指定日期的天数
     * @param date
     * @param day
     * @return
     */
    public static Date getDaysByDate(Date date, int day)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DATE, day);
        return calendar.getTime();
    }

    //判断两个日期是否相等,只判断年月日
    public static boolean isSameDate(Date date1, Date date2)
    {
        try {
            Calendar cal1 = Calendar.getInstance();
            cal1.setTime(date1);
            Calendar cal2 = Calendar.getInstance();
            cal2.setTime(date2);
            boolean isSameYear = cal1.get(Calendar.YEAR) == cal2
                                 .get(Calendar.YEAR);
            boolean isSameMonth = isSameYear
                                  && cal1.get(Calendar.MONTH) == cal2.get(Calendar.MONTH);
            boolean isSameDate = isSameMonth
                                 && cal1.get(Calendar.DAY_OF_MONTH) == cal2
                                 .get(Calendar.DAY_OF_MONTH);
            return isSameDate;
        }
        catch (Exception e) {
            e.printStackTrace();
        }

        return false;
    }

    // 获得本周一与当前日期相差的天数
    private static int getMondayPlus()
    {
        Calendar cd = Calendar.getInstance();
        int dayOfWeek = cd.get(Calendar.DAY_OF_WEEK);

        if (dayOfWeek == 1) {
            return -6;
        }
        else {
            return 2 - dayOfWeek;
        }
    }

    // 获得当前周 周一的日期
    public static String getCurrentMonday()
    {
        int mondayPlus = getMondayPlus();
        GregorianCalendar currentDate = new GregorianCalendar();
        currentDate.add(GregorianCalendar.DATE, mondayPlus);
        Date monday = currentDate.getTime();
        String format = sdf.format(monday);
        return format;
    }


    // 获得当前周 周日  的日期
    public static String getPreviousSunday()
    {
        int mondayPlus = getMondayPlus();
        GregorianCalendar currentDate = new GregorianCalendar();
        currentDate.add(GregorianCalendar.DATE, mondayPlus + 6);
        Date monday = currentDate.getTime();
        String preMonday = sdf.format(monday);
        return preMonday;
    }

    public static Date min(Date d1, Date d2)
    {
        if (d2 == null) {
            return d1;
        }
        else {
            if (d1.compareTo(d2) > 0) {
                return d2;
            }
            else {
                return d1;
            }
        }
    }

    // 获得当前月--开始日期
    public static String getMinMonthDate(String date)
    {
        Calendar calendar = Calendar.getInstance();

        try {
            calendar.setTime(sdf.parse(date));
            calendar.set(Calendar.DAY_OF_MONTH,
                         calendar.getActualMinimum(Calendar.DAY_OF_MONTH));
            return sdf.format(calendar.getTime());
        }
        catch (java.text.ParseException e) {
            e.printStackTrace();
        }

        return null;
    }

    // 获得当前月--结束日期
    public static String getMaxMonthDate(String date)
    {
        Calendar calendar = Calendar.getInstance();

        try {
            calendar.setTime(sdf.parse(date));
            calendar.set(Calendar.DAY_OF_MONTH,
                         calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
            return sdf.format(calendar.getTime());
        }
        catch (java.text.ParseException e) {
            e.printStackTrace();
        }

        return null;
    }

    /**
     * oracle add_month
     * @param d
     * @param month
     * @return
     * @author 肖志文
     * 2020年8月14日 下午4:46:52
     */
    public static Date add_month(Date d, int month)
    {
        Calendar c = Calendar.getInstance();
        c.setTime(d);
        c.add(Calendar.DAY_OF_MONTH, 1);
        c.add(Calendar.MONTH, month);
        c.add(Calendar.DAY_OF_MONTH, -1);
        return c.getTime();
    }

    public static Date date(Object v) throws Exception
    {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        if (v instanceof Date) {
            return sdf.parse(sdf.format((Date) v)); // 确保是yyyy-MM-dd格式的日期
        }
        else if (v instanceof Timestamp) {
            return sdf.parse(sdf.format((Timestamp) v));
        }

        return sdf.parse(v.toString());
    }

    public static  int getDaysOfYear(Date date)
    {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        int year = c.get(Calendar.YEAR);
        return getDaysOfYear(year);
    }

    public static int getDaysOfYear(int year)
    {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.YEAR, year);
        return cal.getActualMaximum(Calendar.DAY_OF_YEAR);
    }
}
