/**
 * Copyright (C), 2007-2014, eFuture 北京富基融通科技有限公司
 * FileName:	CacheUtilsMemcached.java
 * Author:		亮
 * Date:		2014-9-16 下午4:50:31
 * Description:
 * History:
 * <author>		<time>			<version>		<description>
 *
 */
package com.efuture.ocp.common.cache;

import java.io.*;
import java.util.Properties;
import java.util.concurrent.TimeoutException;

import com.efuture.ocp.common.cache.CacheConfig;
import com.efuture.ocp.common.cache.CacheUtils;
import com.efuture.ocp.common.util.WebPathUtils;
import net.rubyeye.xmemcached.GetsResponse;
import net.rubyeye.xmemcached.MemcachedClient;
import net.rubyeye.xmemcached.MemcachedClientBuilder;
import net.rubyeye.xmemcached.XMemcachedClientBuilder;
import net.rubyeye.xmemcached.command.BinaryCommandFactory;
import net.rubyeye.xmemcached.exception.MemcachedException;
import net.rubyeye.xmemcached.utils.AddrUtil;

import org.apache.log4j.Logger;
import org.springframework.util.StringUtils;

import com.efuture.ocp.common.exception.ServiceException;
import com.google.code.yanf4j.core.impl.StandardSocketOption;

/**
 * @author 亮
 * @description
 *
 */
public class CacheUtilsMemcached extends CacheUtils implements Serializable
{
    private MemcachedClientBuilder builder = null;
    private MemcachedClient memcachedClient = null;
    private boolean reBuild = false;
    private String cfgFile = null;
    private int cachetime = -1;

    public Logger logger = Logger.getLogger(CacheUtilsMemcached.class);
    public CacheUtilsMemcached()
    {
        this(null);
    }
    public CacheUtilsMemcached(String config)
    {
        this(config, -1);
    }
    public CacheUtilsMemcached(String config, int cachetime)
    {
        this.cfgFile = config;
        this.cachetime = cachetime;
    }


    private MemcachedClient getClient() throws FileNotFoundException, IOException
    {
        if (memcachedClient == null) {
            CreateBuilder();
        }

        return memcachedClient;
    }
    /**
     * 往缓存赋值
     *
     * @param key
     *            键值
     * @param object
     *            值对象
     * @param timeout
     *            超时：毫秒
     * @return
     */
    public String putData(String key, Object object, int timeout)
    {
        if (memcachedClient == null) {
            try {
                CreateBuilder();
            }
            catch (FileNotFoundException e) {
                e.printStackTrace();
                return null;
            }
            catch (IOException e) {
                e.printStackTrace();
                return null;
            }
        }

        // set
        try {
            if (timeout < 0) {
                timeout = (cachetime > 0 ? cachetime : 0);
            }

            memcachedClient.set(key, timeout, object);
            return key;
        }
        catch (TimeoutException e) {
            e.printStackTrace();

            if (reBuild) {
                logger.warn("memcached Client ReBuild!!!!!!!!!!!");
                builder = null;
                memcachedClient = null;
            }

            return null;
        }
        catch (InterruptedException e) {
            e.printStackTrace();
            return null;
        }
        catch (MemcachedException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 缓存取值
     *
     * @param key
     *            键值
     * @return
     */
    public Object getData(String key)
    {
        if (memcachedClient == null) {
            try {
                CreateBuilder();
            }
            catch (FileNotFoundException e) {
                e.printStackTrace();
                return null;
            }
            catch (IOException e) {
                e.printStackTrace();
                return null;
            }
        }

        // get
        Object result = null;

        try {
            result = memcachedClient.get(key);
        }
        catch (TimeoutException e) {
            e.printStackTrace();

            if (reBuild) {
                logger.warn("memcached Client ReBuild!!!!!!!!!!!");
                builder = null;
                memcachedClient = null;
            }

            return null;
        }
        catch (InterruptedException e) {
            e.printStackTrace();
            return null;
        }
        catch (MemcachedException e) {
            e.printStackTrace();
            return null;
        }

        return result;
    }

    @Override
    public Object getCacheData(String key)
    {
        return getData(key);
    }
    /**
     * 缓存删除
     *
     * @param key
     *            键值
     */
    public boolean deleteData(String key)
    {
        if (memcachedClient == null) {
            try {
                CreateBuilder();
            }
            catch (FileNotFoundException e) {
                e.printStackTrace();
                return false;
            }
            catch (IOException e) {
                e.printStackTrace();
                return false;
            }
        }

        // delete
        try {
            memcachedClient.delete(key);
        }
        catch (TimeoutException e) {
            e.printStackTrace();

            if (reBuild) {
                logger.warn("memcached Client ReBuild!!!!!!!!!!!");
                builder = null;
                memcachedClient = null;
            }

            return false;
        }
        catch (InterruptedException e) {
            e.printStackTrace();
            return false;
        }
        catch (MemcachedException e) {
            e.printStackTrace();
            return false;
        }

        return true;
    }

    /**
     * 创建Memcached缓存客户端
     *
     * @return Memcached客户端对象
     * @throws InvalidFileFormatException
     * @throws FileNotFoundException
     * @throws IOException
     */
    public synchronized MemcachedClient CreateBuilder() throws FileNotFoundException, IOException
    {
        if (memcachedClient != null) {
            return memcachedClient;
        }

        // 取memcached.ini中的MemCached服务器配置
        String ini = null, ini0 = null;
        boolean useconfig = false;

        if (config != null && StringUtils.isEmpty(cfgFile)) {
            useconfig = true;
        }
        else if (StringUtils.isEmpty(cfgFile)) {
            String p = WebPathUtils.getWebRootClassPath();
            ini = p + "/memcached.ini"; // （原来的）
            ini0 = p + "/conf/memcached.ini";
            // *****************解决有的配置文件存储在/WEB-INF/classes/conf 下***********//

        }
        else {
            if (cfgFile.startsWith("\\") || cfgFile.startsWith("/")) {
                String p = WebPathUtils.getWebRootClassPath();
                ini = p + cfgFile;
                ini0 = p + "/conf" + cfgFile;
            }
            else {
                ini = cfgFile;
            }
        }

        String serverList = "localhost:11211";
        int iPoolSize = 10;
        int iTimeOut = 5000;
        int iMerge = 150;
        boolean bHeart = true;
        int iRcvBuf = 32;
        int iSndBuf = 16;
        File input = null;

        if (!StringUtils.isEmpty(ini)) {
            input = new File(ini);

            // 如果classes目录下没有查找classes/conf目录
            if (ini0 != null && !input.exists()) {
                input = new File(ini0);
            }
        }


        logger.info("memcached.ini path:" + ini);

        if (!useconfig && input != null && input.exists()) {
            InputStream is = new FileInputStream(input);
            Properties p = new Properties();
            p.load(is);
            is.close();

            String sServer = p.getProperty("server");

            if (sServer != null) {
                sServer = sServer.replaceAll(";", " ");
                serverList = sServer.trim();
            }

            String sPoolSize = p.getProperty("poolsize");

            if (sPoolSize != null) {
                iPoolSize = Integer.parseInt(sPoolSize);
            }

            String sRcvBuf = p.getProperty("rcvbuf");

            if (sRcvBuf != null) {
                iRcvBuf = Integer.parseInt(sRcvBuf);
            }

            String sSndBuf = p.getProperty("sndbuf");

            if (sSndBuf != null) {
                iSndBuf = Integer.parseInt(sSndBuf);
            }

            String sTimeOut = p.getProperty("timeout");

            if (sTimeOut != null) {
                iTimeOut = Integer.parseInt(sTimeOut);
            }

            String sMerge = p.getProperty("merge");

            if (sMerge != null) {
                iMerge = Integer.parseInt(sMerge);
            }

            String sHeart = p.getProperty("heart");

            if (sHeart != null) {
                bHeart = ((sHeart != null && sHeart.equalsIgnoreCase("false")) ? false : true);
            }

            String re = p.getProperty("rebuild");

            if (re != null) {
                reBuild = ((re != null && re.equalsIgnoreCase("true")) ? true : false);
            }

            logger.info("memcached server:" + serverList);

        }
        else if (useconfig) {


            serverList = this.config.getServerList().trim();
            serverList = serverList.replaceAll(";", " ");

            iPoolSize = config.getPoolsize();
            iRcvBuf = config.getRcvbuf();
            iSndBuf = config.getSndbuf();
            iTimeOut = config.getTimeout();
            iMerge = config.getMerge();

            bHeart = config.isHeart();
            reBuild = config.isRebuild();
        }
        else {
            logger.warn("memcached.ini not exits!");
            serverList = "localhost:11211";
            iPoolSize = 2;
            System.out.println("memcached server-地址:" + "localhost:11211");
        }

        builder = new XMemcachedClientBuilder(AddrUtil.getAddresses(serverList));

        // use binary protocol
        builder.setCommandFactory(new BinaryCommandFactory());
        builder.setConnectionPoolSize(iPoolSize);


        builder.setSocketOption(StandardSocketOption.SO_RCVBUF, iRcvBuf * 1024); // 设置接收缓存区为32K，默认16K
        builder.setSocketOption(StandardSocketOption.SO_SNDBUF, iSndBuf * 1024); // 设置发送缓冲区为16K，默认为8K
        builder.setSocketOption(StandardSocketOption.TCP_NODELAY, false); // 启用nagle算法，提高吞吐量，默认关闭
        memcachedClient = builder.build();
        memcachedClient.setOpTimeout(iTimeOut);

        if (iMerge <= 0) {
            memcachedClient.setOptimizeMergeBuffer(false);    // 关闭合并buffer的优化
        }

        if (iMerge > 0) {
            memcachedClient.setMergeFactor(iMerge);    // 默认是150，缩小到50
        }

        memcachedClient.setEnableHeartBeat(bHeart);
        return memcachedClient;
    }

    @Override
    public long getCasValue(String key)
    {
        if (memcachedClient == null) {
            try {
                CreateBuilder();
            }
            catch (FileNotFoundException e) {
                e.printStackTrace();
                return -1;
            }
            catch (IOException e) {
                e.printStackTrace();
                return -1;
            }
        }

        try {
            GetsResponse res = memcachedClient.gets(key);

            if (res == null) {
                return 0;
            }
            else {
                return res.getCas();
            }
        }
        catch (TimeoutException e) {
            e.printStackTrace();

            if (reBuild) {
                logger.warn("memcached Client ReBuild!!!!!!!!!!!");
                builder = null;
                memcachedClient = null;
            }

            return -1;
        }
        catch (InterruptedException e) {
            e.printStackTrace();
            return -1;
        }
        catch (MemcachedException e) {
            e.printStackTrace();
            return -1;
        }
    }

    @Override
    public boolean putbyCas(String key, Object object, int timeout, long casvalue)
    {
        if (memcachedClient == null) {
            try {
                CreateBuilder();
            }
            catch (FileNotFoundException e) {
                e.printStackTrace();
                return false;
            }
            catch (IOException e) {
                e.printStackTrace();
                return false;
            }
        }

        try {
            if (timeout < 0) {
                timeout = (cachetime > 0 ? cachetime : 0);
            }

            return memcachedClient.cas(key, timeout, object, casvalue);
        }
        catch (TimeoutException e) {
            e.printStackTrace();

            if (reBuild) {
                logger.warn("memcached Client ReBuild!!!!!!!!!!!");
                builder = null;
                memcachedClient = null;
            }

            return false;
        }
        catch (InterruptedException e) {
            e.printStackTrace();
            return false;
        }
        catch (MemcachedException e) {
            e.printStackTrace();
            return false;
        }
    }

    @Override
    public GetsResponse getCasData(String key)
    {
        if (memcachedClient == null) {
            try {
                CreateBuilder();
            }
            catch (FileNotFoundException e) {
                e.printStackTrace();
                return null;
            }
            catch (IOException e) {
                e.printStackTrace();
                return null;
            }
        }

        try {
            GetsResponse res = memcachedClient.gets(key);
            return res;
        }
        catch (TimeoutException e) {
            e.printStackTrace();

            if (reBuild) {
                logger.warn("memcached Client ReBuild!!!!!!!!!!!");
                builder = null;
                memcachedClient = null;
            }

            return null;
        }
        catch (InterruptedException e) {
            e.printStackTrace();
            return null;
        }
        catch (MemcachedException e) {
            e.printStackTrace();
            return null;
        }
    }


    /**
     * 往缓存加入指定的key,如果不存在，则加入，返回 true
     *  如果存在则加入不成功 返回false
     *  如果mem
     * @param key
     * @param object
     * @param timeout
     * @return
     * @throws MemcachedException
     * @throws InterruptedException
     * @throws TimeoutException
     * @throws IOException
     * @throws FileNotFoundException
     */
    @Override
    public boolean AddData(String key, Object object, int timeout)
    {
        MemcachedClient client;

        try {
            client = getClient();
        }
        catch (IOException e) {
            throw new ServiceException("30001", "Memcached创建错误!");
        }

        boolean ibadd = true;

        try {
            ibadd = client.add(key, timeout, object);
        }
        catch (TimeoutException | InterruptedException | MemcachedException e) {
            throw new ServiceException("30002", "执行Mem_AddData错误!");
        }

        return ibadd;

    }
}
