package com.efuture.ocp.common.annotation;

import com.efuture.ocp.common.entity.ServiceSession;
import com.efuture.ocp.common.exception.ServiceException;
import com.efuture.ocp.common.language.MessageSourceHelper;
import com.efuture.ocp.common.language.ResponseCode;
import com.efuture.ocp.common.proxy.EnterpriseProxy;
import com.efuture.ocp.common.util.CacheUtils;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.RateLimiter;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.stereotype.Component;

import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;
import java.util.Map;

/**
 * @title: AutoAccesslimitAspect
 * @description: TODO
 * @author: wangf
 * @date: 2020/04/18
 */

@Aspect
@Component
public class AutoAccesslimitAspect
{

    private static Logger logger = LogManager.getLogger(AutoAccesslimitAspect.class);
    private Map<String, RateLimiter> limitMap = Maps.newConcurrentMap();

    //获取企业Session
    private ServiceSession getSessionKey(ProceedingJoinPoint point, AutoAccesslimit around)
    {
        Object[] args = point.getArgs();

        for (Object arg : args) {
            if (arg instanceof ServiceSession) {
                return (ServiceSession) arg;
            }
        }

        return null;
    }

    private static String getHostIP()
    {
        try {
            Enumeration<NetworkInterface> netInterfaces = NetworkInterface.getNetworkInterfaces();

            while (netInterfaces.hasMoreElements()) {
                NetworkInterface nif = netInterfaces.nextElement();
                Enumeration<InetAddress> iparray = nif.getInetAddresses();

                while (iparray.hasMoreElements()) {
                    InetAddress ip = iparray.nextElement();

                    if (ip != null) {
                        //System.out.println("IP: "+ ip.getHostAddress());
                        if (ip instanceof Inet4Address) {
                            String localip = ip.getHostAddress();

                            if (!"127.0.0.1".equals(localip)) {
                                return ip.getHostAddress();
                            }
                        }
                    }
                }
            }
        }
        catch (SocketException e) {
            e.printStackTrace();
        }

        return null;
    }

    @Around("@annotation(around)")
    public Object round(ProceedingJoinPoint point, AutoAccesslimit around) throws Throwable
    {
        String className = point.getTarget().getClass().getSimpleName();
        String methodName = point.getSignature().getName();
        String service = around.Service();
        int time = around.time();
        int maxLimit = around.maxLimit();
        ServiceSession session = getSessionKey(point, around);
        String cacheKey;

        if (session != null) {
            String accessMaxList = EnterpriseProxy.getInstance().getEnterpriseConfig(session, "accessMaxLimit", false, "");

            if (!StringUtils.isEmpty(accessMaxList)) {
                maxLimit = Integer.parseInt(accessMaxList);
            }

            cacheKey = String.format("%d-%s-%s-%s-%s", session.getEnt_id(), service, className, methodName, getHostIP());
        }
        else {
            cacheKey = String.format("%s-%s-%s-%s", service, className, methodName, getHostIP());
        }

        Object result = null;

        //mode 0 - (最大吞吐量限流) 1 - (速率限流-漏桶限流)
        if (around.mode() == 0) {
            Integer requestNum = (Integer) CacheUtils.getCacheUtils().getData(cacheKey);

            if (requestNum == null) {
                requestNum = 0;
            }

            Integer nowRequestNum = requestNum + 1;

            if (nowRequestNum >= maxLimit) {
                throw new ServiceException(ResponseCode.FAILURE, "服务繁忙，稍后再试");
            }

            CacheUtils.getCacheUtils().putData(cacheKey, nowRequestNum, time);

            try {
                result = point.proceed();
            }
            catch (Throwable e) {
                throw e;
            }
            finally {
                requestNum = (Integer) CacheUtils.getCacheUtils().getData(cacheKey);

                if (requestNum == null) {
                    CacheUtils.getCacheUtils().putData(cacheKey, 0, time);
                }
                else {
                    Integer newRequestNum = 0;

                    if (requestNum > 0) {
                        newRequestNum = requestNum - 1;
                    }

                    CacheUtils.getCacheUtils().putData(cacheKey, newRequestNum, time);
                }
            }

            // 1 - (速率限流-漏桶限流)
        }
        else if (around.mode() == 1) {
            RateLimiter limiter = null;

            if (!limitMap.containsKey(cacheKey)) {
                // 创建令牌桶
                limiter = RateLimiter.create(maxLimit, around.time(), around.timeunit());
                limitMap.put(cacheKey, limiter);
                logger.info( MessageSourceHelper.formatMessage("请求{0},创建令牌桶,容量{1}", cacheKey, maxLimit));
            }

            limiter = limitMap.get(cacheKey);
            // 获取令牌
            boolean acquire = limiter.tryAcquire();

            if (!acquire) {
                throw new ServiceException(ResponseCode.FAILURE, "服务繁忙，稍后再试");
            }

            result = point.proceed();
        }

        return result;
    }
}
