package com.efuture.ocp.common.cache.redis;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import redis.clients.jedis.*;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;


/**
 * jedis管理类
 *
 * @author zwt
 */
public class JedisManager {

    private static final Log log = LogFactory.getLog(JedisManager.class);
    private static ShardedJedisPool pool;
    private static JedisPool jedisPool;
    private static JedisPoolConfig poolConfig = new JedisPoolConfig();

    static {
        log.info("JedisManager static code block loader...");
        //最大活动的对象个数
        poolConfig.setMaxTotal(500);
        //对象最大空闲时间
        poolConfig.setMaxIdle(60 * 1000);
        //获取对象时最大等待时间
        poolConfig.setMaxWaitMillis(1000L);
        //当调用borrow Object方法时，是否进行有效性检查
        poolConfig.setTestOnBorrow(true);
        String dbIndex  = RedisConfig.getString("efuture.cache.redisDatabase");
        String[] strArray = RedisConfig.getRedisArray();
        System.out.println(strArray[0]);
        String password = RedisConfig.getStringArray("efuture.cache.redisPassword")[0];//RedisConfig.getRedisPassword().split(",")[0];
        System.out.println(password);
        List<JedisShardInfo> shardJedis = new ArrayList<JedisShardInfo>();
        for (int i = 0; i < strArray.length; i++) {
        	int port = Integer.parseInt(strArray[i].split(":")[1]);
        	String host = strArray[i].split(":")[0];
        	if(dbIndex != null && dbIndex.matches("^\\d+$")){
        		String url = "http://"+host+":"+port+"/"+dbIndex;
        		JedisShardInfo jedisShardInfo;
				try {
					jedisShardInfo = new JedisShardInfo(new URI(url));
					if (password != null && !password.equals(""))
	                    jedisShardInfo.setPassword(password);
	                shardJedis.add(jedisShardInfo);
				} catch (URISyntaxException e) {
					e.printStackTrace();
				}

        		 if(i == 0){
        			 jedisPool = new JedisPool(poolConfig, host, port, 5000, password, Integer.parseInt(dbIndex));
                 }
        	}else{
        		  JedisShardInfo jedisShardInfo = new JedisShardInfo(host, port, 5000);
                  if (password != null && !password.equals(""))
                      jedisShardInfo.setPassword(password);
                  shardJedis.add(jedisShardInfo);

                  if(i == 0){
                  	jedisPool = new JedisPool(poolConfig, host, port, 5000, password);
                  }
        	}




        }
        pool = new ShardedJedisPool(poolConfig, shardJedis);
    }

    public static Set<String> keys(String pattern) {
        Jedis jedis = jedisPool.getResource();
        Set<String> set = null;
        try {
        	set = jedis.keys(pattern);
        } catch (Exception e) {
            log.error(e);
        } finally {
        	jedis.close();
        }
        return set;
    }

    public static String setString(String key, String str, int liveSeconds) {
        if ((key == null) || (str == null))
            return null;
        ShardedJedis jedis = null;
        String ret = null;
        try {
            jedis = pool.getResource();
            if (liveSeconds <= 0)
                //永久保存
                ret = jedis.set(encode(key), str);
            else
                //设置有效期
                ret = jedis.setex(encode(key), liveSeconds, str);
        } catch (Exception e) {
            log.error(e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return ret;
    }

    public static String getString(String key) {
        ShardedJedis jedis = null;
        String ret = null;
        try {
            jedis = pool.getResource();
            ret = jedis.get(encode(key));
        } catch (Exception e) {
            log.error(e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }

        }
        return ret;
    }

    public static long delString(String key) {
        return del(key);
    }

    public static long delStringNew(String key) {
        return del(encode(key));
    }

    public static long delObject(String key) {
        return del(key);
    }

    public static Object getObject(String key) {
        if (key == null)
            return null;
        byte[] ret = getBytes(key.getBytes());
        if (ret == null)
            return null;
        return ObjectBytesExchange.toObject(ret);
    }

    public static String setObject(String key, int seconds, Object obj) {
        if ((key == null) || (obj == null))
            return null;
        byte[] byteObj = ObjectBytesExchange.toByteArray(obj);
        if (null == byteObj)
            return null;
        return setBytes(key.getBytes(), seconds, byteObj);
    }

    public static byte[] getBytes(byte[] key) {
        if (key == null)
            return null;
        ShardedJedis jedis = null;
        byte[] ret = null;
        try {
            jedis = (ShardedJedis) pool.getResource();
            ret = jedis.get(key);
        } catch (Exception e) {
            log.error(e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static String setBytes(byte[] key, int seconds, byte[] bytes) {
        if ((key == null) || (bytes == null))
            return null;
        ShardedJedis jedis = null;
        String ret = null;
        try {
            jedis = (ShardedJedis) pool.getResource();
            if (seconds <= 0)
                ret = jedis.set(key, bytes);
            else
                ret = jedis.setex(key, seconds, bytes);
        } catch (Exception e) {
            log.error(e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static String encode(String str) {
        String ret = null;
        if (str != null) {
            try {
                ret = URLEncoder.encode(str, "UTF-8");
            } catch (UnsupportedEncodingException e) {
                log.error(e);
            }
        }
        return ret;
    }


    public long del(byte[] key) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.del(key).longValue();
        } catch (Exception e) {
            log.error(e);
            ret = -1L;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    private static long del(String key) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.del(key).longValue();
        } catch (Exception e) {
            log.error(e);
            ret = -1L;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static long expire(String key, int seconds) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.expire(key, seconds);
        } catch (Exception e) {
            log.error(e);
            ret = -1L;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static boolean lock(String key, int seconds) {
        if (seconds <= 0) {
            seconds = 21600;
        }
        boolean result = false;
        if (key == null)
            return result;
        ShardedJedis jedis = null;
        long currentTime = System.currentTimeMillis();
        long expireTime = currentTime + seconds * 1000;
        try {
            jedis = pool.getResource();
            if (jedis.setnx(encode(key), String.valueOf(expireTime)).longValue() == 1L) {
                result = true;
                jedis.expire(encode(key), seconds);
            } else {
                Long oldExpireTime = Long.valueOf(getString(encode(key)));
                if ((oldExpireTime != null) && (currentTime > oldExpireTime.longValue() + 300000L)) {
                    del(encode(key));
                    if (jedis.setnx(encode(key), String.valueOf(expireTime)).longValue() == 1L) {
                        result = true;
                        jedis.expire(encode(key), seconds);
                    }
                }
            }
        } catch (Exception e) {
            log.error(e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return result;
    }

    public static ShardedJedisPool getPool() {
        return pool;
    }

    private static void destroy() {
        if (pool != null) {
            try {
                pool.destroy();
            } catch (Exception ex) {
                log.warn("Cannot properly close ShardedJedisPool ", ex);
            }
            pool = null;
        }
    }

    public static long incrByStr(String key) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.incr(key).longValue();
        } catch (Exception e) {
            log.error(e);
            ret = -1L;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }


    public static long decrByStr(String key,long integer) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.decrBy(key, integer);
        } catch (Exception e) {
            log.error(e);
            ret = -1L;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static double incrByFloat(String key,double integer) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        double ret = 0;
        try {
            jedis = pool.getResource();
            ret = jedis.incrByFloat(key, integer);
        } catch (Exception e) {
        	e.printStackTrace();
            log.error(e);
            ret = -1;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }



    public static long decrByStr(String key) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.decr(key);
        } catch (Exception e) {
            log.error(e);
            ret = -1L;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static long incrByByte(byte[] key) {
        if ((key == null) || (key.length == 0)) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.incr(key).longValue();
        } catch (Exception e) {
            log.error(e);
            ret = -1L;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static long rpush(String key, String value) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.rpush(key, value);
        } catch (Exception e) {
            log.error(e);
            ret = -1L;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static long llen(String key) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.llen(key);
        } catch (Exception e) {
            log.error(e);
            ret = -1L;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static List<String> lrange(String key, int start, int end) {

        ShardedJedis jedis = null;
        List<String> ret = new ArrayList<String>();
        try {
            jedis = pool.getResource();
            ret = jedis.lrange(key, start, end);
        } catch (Exception e) {
            log.error(e);
            ret = null;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static String lpop(String key) {
        if (key == null) {
            return null;
        }
        ShardedJedis jedis = null;
        String ret = null;
        try {
            jedis = pool.getResource();
            ret = jedis.lpop(key);
        } catch (Exception e) {
            log.error(e);
            ret = null;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    /**
     * 查看剩余时间数量
     *
     * @param key
     * @return
     * @throws
     * @Description:
     */
    public static long ttl(String key) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.ttl(key);
        } catch (Exception e) {
            log.error(e);
            ret = -1L;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static long lpush(String key, String... value) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.lpush(key, value);
        } catch (Exception e) {
            log.error(e);
            ret = -1L;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static List<String> lrange(String key, long start, long end) {
        ShardedJedis jedis = null;
        List<String> ret = new ArrayList<String>();
        try {
            jedis = pool.getResource();
            ret = jedis.lrange(key, start, end);
        } catch (Exception e) {
            log.error(e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static String rpop(String key) {
        if (key == null) {
            return null;
        }
        ShardedJedis jedis = null;
        String ret = null;
        try {
            jedis = pool.getResource();
            ret = jedis.rpop(key);
        } catch (Exception e) {
            log.error(e);
            ret = null;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }


    public static long lrem(String key, long count, String value) {
        if (key == null) {
            return -1;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.lrem(key, count, value);
        } catch (Exception e) {
            log.error(e);
            ret = -1;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static String ltrim(String key, long start, long end) {
        if (key == null) {
            return null;
        }
        ShardedJedis jedis = null;
        String ret = null;
        try {
            jedis = pool.getResource();
            ret = jedis.ltrim(key, start, end);
        } catch (Exception e) {
            log.error(e);
            ret = null;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static  Set<String> spop(String key,long count) {
        if (key == null) {
            return null;
        }
        ShardedJedis jedis = null;
        Set<String> ret = null;
        try {
            jedis = pool.getResource();
            ret = jedis.spop(key,count);
        } catch (Exception e) {
        	e.printStackTrace();
            log.error(e);
            ret = null;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }
    public static  Set<String> smembers(String key) {
        if (key == null) {
            return null;
        }
        ShardedJedis jedis = null;
        Set<String> ret = null;
        try {
            jedis = pool.getResource();
            ret = jedis.smembers(key);
        } catch (Exception e) {
        	e.printStackTrace();
            log.error(e);
            ret = null;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }
    //随机移除
    public static String spop(String key) {
        if (key == null) {
            return null;
        }
        ShardedJedis jedis = null;
        String ret = null;
        try {
            jedis = pool.getResource();
            ret = jedis.spop(key);
        } catch (Exception e) {
            log.error(e);
            ret = null;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    //增加 重复的将被覆盖
    public static long sadd(String key, String... members) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.sadd(key,members);
        } catch (Exception e) {
        	e.printStackTrace();
            log.error(e);
            ret = -1;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }


    public static long zadd(String key,double score,String member) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.zadd(key, score, member);
        } catch (Exception e) {
        	e.printStackTrace();
            log.error(e);
            ret = -1;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static long zrem(String key,String member) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.zrem(key,member);
        } catch (Exception e) {
        	e.printStackTrace();
            log.error(e);
            ret = -1;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static double zscore(String key,String member) {
        ShardedJedis jedis = null;
        double ret = 0;
        try {
            jedis = pool.getResource();
            ret = jedis.zscore(key, member);
        } catch (Exception e) {
        	e.printStackTrace();
            log.error(e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static Set<String> zrange(String key,long start,long end) {
        ShardedJedis jedis = null;
        Set<String> ret = null;
        try {
            jedis = pool.getResource();
            ret = jedis.zrange(key, start, end);
        } catch (Exception e) {
        	e.printStackTrace();
            log.error(e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static long zcard(String key) {
        if (key == null) {
            return 0L;
        }
        ShardedJedis jedis = null;
        long ret = 0L;
        try {
            jedis = pool.getResource();
            ret = jedis.zcard(key);
        } catch (Exception e) {
        	e.printStackTrace();
            log.error(e);
            ret = -1;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }

    public static String get(String key) {
        ShardedJedis jedis = null;
        String ret = null;
        try {
            jedis = pool.getResource();
            ret = jedis.get(key);
        } catch (Exception e) {
        	e.printStackTrace();
            log.error(e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return ret;
    }
}
