package com.efuture.ocp.common.annotation;

import com.efuture.ocp.common.component.GlobalConfig;
import com.efuture.ocp.common.entity.AccessLimitBean;
import com.efuture.ocp.common.entity.ServiceSession;
import com.efuture.ocp.common.exception.ServiceException;
import com.efuture.ocp.common.language.MessageSourceHelper;
import com.efuture.ocp.common.language.ResponseCode;
import com.efuture.ocp.common.proxy.EnterpriseProxy;
import com.efuture.ocp.common.util.CacheUtils;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.RateLimiter;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.stereotype.Component;

import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @title: AutoAccesslimitAspect
 * @description: TODO
 * @author: wangf
 * @date: 2020/04/18
 */

@Aspect
@Component
public class AutoAccesslimitAspect
{

    private static Logger logger = LogManager.getLogger(AutoAccesslimitAspect.class);
    private Map<String, RateLimiter> limitMap = Maps.newConcurrentMap();

    //获取企业Session
    private ServiceSession getSessionKey(ProceedingJoinPoint point, AutoAccesslimit around)
    {
        Object[] args = point.getArgs();

        for (Object arg : args) {
            if (arg instanceof ServiceSession) {
                return (ServiceSession) arg;
            }
        }

        return new ServiceSession(0);
    }

    private static String getHostIP()
    {
        try {
            Enumeration<NetworkInterface> netInterfaces = NetworkInterface.getNetworkInterfaces();

            while (netInterfaces.hasMoreElements()) {
                NetworkInterface nif = netInterfaces.nextElement();
                Enumeration<InetAddress> iparray = nif.getInetAddresses();

                while (iparray.hasMoreElements()) {
                    InetAddress ip = iparray.nextElement();

                    if (ip != null) {
                        //System.out.println("IP: "+ ip.getHostAddress());
                        if (ip instanceof Inet4Address) {
                            String localip = ip.getHostAddress();

                            if (!"127.0.0.1".equals(localip)) {
                                return ip.getHostAddress();
                            }
                        }
                    }
                }
            }
        }
        catch (SocketException e) {
            e.printStackTrace();
        }

        return null;
    }

    @Around("@annotation(around)")
    public Object round(ProceedingJoinPoint point, AutoAccesslimit around) throws Throwable
    {
        String className = point.getTarget().getClass().getSimpleName();
        String methodName = point.getSignature().getName();
        String service = around.Service();
        int mode = around.mode();
        int time = around.time();
        int maxLimit = around.maxLimit();

        ServiceSession session = getSessionKey(point, around);
        String cacheKey = String.format("%s-%s-%s", service, className, methodName);
        AccessLimitBean limit = GlobalConfig.getInstance().getAndSetAccessLimit( session.getEnt_id(), cacheKey, mode, time, maxLimit );

        if (limit != null) {
            mode = limit.getLimitMode();
            maxLimit = limit.getPermitsPerSecond();
            time = limit.getWarmUpPeriod();
            // 生成不同的对象
            cacheKey = cacheKey.concat( "-" ).concat( String.valueOf(session.getEnt_id()) ).concat( "-" ).concat( String.valueOf(maxLimit) );
        }
        else {
            cacheKey = cacheKey.concat( "-" ).concat( String.valueOf(session.getEnt_id()) );
        }

        // 设置为0则不限流
        if (maxLimit <= 0) {
            return point.proceed();
        }

        Object result = null;

        //mode 0 - (最大吞吐量限流) 1 - (速率限流-漏桶限流)
        if (mode == 0) {

            if (time <= 0) {
                time = CacheUtils.CacheTimeOut.Min;
            }

            Integer requestNum = (Integer) CacheUtils.getInMemUtils().getData(cacheKey);

            if (requestNum == null) {
                requestNum = 0;
            }

            Integer nowRequestNum = requestNum + 1;

            if (nowRequestNum >= maxLimit) {
                throw new ServiceException(ResponseCode.FAILURE, "服务繁忙，稍后再试");
            }

            CacheUtils.getInMemUtils().putData(cacheKey, nowRequestNum, time);

            try {
                result = point.proceed();
            }
            catch (Throwable e) {
                throw e;
            }
            finally {
                requestNum = (Integer) CacheUtils.getInMemUtils().getData(cacheKey);

                if (requestNum == null) {
                    CacheUtils.getInMemUtils().putData(cacheKey, 0, time);
                }
                else {
                    Integer newRequestNum = 0;

                    if (requestNum > 0) {
                        newRequestNum = requestNum - 1;
                    }

                    CacheUtils.getInMemUtils().putData(cacheKey, newRequestNum, time);
                }
            }
        }
        else if (mode == 1) {
            // 1 - (速率限流-漏桶限流)
            RateLimiter limiter = null;

            if (!limitMap.containsKey(cacheKey)) {
                if (time > 0) {
                    limiter = RateLimiter.create(maxLimit, time, around.timeunit());
                }
                else {
                    limiter = RateLimiter.create(maxLimit );
                }

                limitMap.put(cacheKey, limiter);
            }

            limiter = limitMap.get(cacheKey);
            // 获取令牌
            boolean acquire = limiter.tryAcquire(1, 10, TimeUnit.MICROSECONDS);

            if (!acquire) {
                throw new ServiceException(ResponseCode.FAILURE, "服务繁忙，稍后再试");
            }

            result = point.proceed();
        }

        return result;
    }
}
