package com.efuture.ocp.common.util;

import java.util.HashMap;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.efuture.ocp.common.component.GlobRuleSrv;

/**
 * 单一RULEID的序号生成器(带缓存),一个RULEID请保持一个实例
 * 
 * @author wudong
 * 
 */

public class KeyGen {

	private Map<Long, Key> mapkey = new HashMap<Long, Key>();;// 按企业ID缓存的KEY
	// 批量获取序列时 放进程中的KEY,与其他进程隔离
	private static ThreadLocal<Key> ThreadLocalKey = new ThreadLocal<Key>();

	private String ruleid;// 规则编号 globrule表的测试
	private long defaultcachenum = 20;// 缓存序号数量

	private GlobRuleSrv globrulesrv;

	public KeyGen(String ruleid, long defaultcachenum, GlobRuleSrv globrulesrv) {
		super();
		this.ruleid = ruleid;
		this.defaultcachenum = defaultcachenum;
		this.globrulesrv = globrulesrv;
	}

	public void setGlobrulesrv(GlobRuleSrv globrulesrv) {
		this.globrulesrv = globrulesrv;
	}

	public void setDefaultcachenum(long defaultcachenum) {
		this.defaultcachenum = defaultcachenum;
	}

	/**
	 * 内部使用 提供缓存的key
	 * 
	 * @author wudong
	 * 
	 */
	private class Key {
		private long cachenum = 0; // 缓存数量
		private long cur = 0;
		private GlobRuleSrv globrulesrv;
		boolean iblocal;
		private long defaultcachenum;
		private long ent_id;

		public Key(long defaultcachenum, GlobRuleSrv globrulesrv,
				boolean iblocal, long ent_id) {
			super();
			this.defaultcachenum = defaultcachenum;
			this.globrulesrv = globrulesrv;
			this.iblocal = iblocal;
			this.ent_id = ent_id;
		}

		public long getCachenum() {
			return cachenum;
		}

		public void setCachenum(long cachenum) {
			this.cachenum = cachenum;
		}

		public synchronized void setCur(long cur) {
			this.cur = cur;
		}

		public synchronized long getNext() {
			if (cachenum <= 0) {
				if (!iblocal || cur == 0) {
					cur = getdblastno(ent_id, defaultcachenum);
					cachenum = defaultcachenum;
				} else
					return 0;
			}
			cur = cur + 1;
			cachenum = cachenum - 1;
			return cur;
		}

		/**
		 * 到数据库取序号
		 * 
		 * @param ent_id
		 *            企业ID
		 * @param cachenum
		 *            缓存数量
		 * @return
		 */

		private long getdblastno(long ent_id, long cachenum) {
			return globrulesrv.getlast(ent_id, ruleid, "#", "#", cachenum);
		}
	}

	public String getRuleid() {
		return ruleid;
	}

	public void setRuleid(String ruleid) {
		this.ruleid = ruleid;
	}

	/**
	 * 得到下一个序号 如果当前进程中有缓存的，如果没有则取公共的
	 * 
	 * @param ent_id
	 *            企业ID
	 * @return
	 */
	public long getNext(long ent_id) {
		long next = getThreadLocalNext(ent_id);
		if (next == 0L) {
			next = getpubnext(ent_id);
		}
		return next;
	}

	/**
	 * 批量得到序号，放当前进程中，取序号时一样
	 * 
	 * @param ent_id
	 *            企业ID
	 * @param cachenum
	 *            需要的序号数
	 */
	public void getbatch(long ent_id, long cachenum) {

		ThreadLocalKey.set(getnewkey(ent_id, cachenum, true));

	}

	/**
	 * 到数据库取序号
	 * 
	 * @param ent_id
	 *            企业ID
	 * @param cachenum
	 *            缓存数量
	 * @return
	 */

	private long getdblastno(long ent_id, long cachenum) {
		return globrulesrv.getlast(ent_id, ruleid, "#", "#", cachenum);
	}

	/**
	 * 得到公共缓存的下一个序号,同步读取
	 * 
	 * @param ent_id
	 *            企业ID
	 * @return
	 */
	private long getpubnext(long ent_id) {
		return getpubkey(ent_id).getNext();
	}

	/**
	 * 得到当前进程缓存的下一个序号，如果没有或者没有当前进程的缓存，返回0，不需要同步讀
	 * 
	 * @param ent_id
	 *            企业ID
	 * @return
	 */
	private long getThreadLocalNext(long ent_id) {
		Key thiskey = getThreadLocalKey(ent_id);
		if (null == thiskey) {
			return 0L;
		} else
			return thiskey.getNext();
	}

	/**
	 * 得到一个KEY的实例
	 * 
	 * @param ent_id
	 *            企业ID
	 * @param cachenum
	 *            缓存数量
	 * @return
	 */
	private Key getnewkey(long ent_id, long cachenum, boolean iblocal) {
		return new Key(cachenum, globrulesrv, iblocal, ent_id);

	}

	/**
	 * 得到当前进程缓存的KEY实例,当前进程的不存在企业ID
	 * 
	 * @param ent_id
	 * @return
	 */
	private Key getThreadLocalKey(long ent_id) {

		Key localkey = ThreadLocalKey.get();
		if (null == localkey) {
			return null;
		}
		return localkey;
	}

	/**
	 * 得到公共的缓存的KEY对象
	 * 
	 * @param ent_id
	 *            企业ID
	 * @return
	 */
	private Key getpubkey(long ent_id) {
		Key thiskey = mapkey.get(ent_id);
		if (null == thiskey) {
			thiskey = putkeytomap(ent_id);
		}
		return thiskey;
	}

	private synchronized Key putkeytomap(long ent_id) {
		Key thiskey = mapkey.get(ent_id);
		if (null == thiskey) {
			thiskey = getnewkey(ent_id, defaultcachenum, false);
			mapkey.put(ent_id, thiskey);
		}
		return thiskey;
	}

}
