package com.efuture.ocp.common.language;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.*;

import com.efuture.ocp.common.util.DataUtils;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PropertiesLoaderUtils;

/**
 * 国际化语言包处理
 * 
 * 增加语言支持的处理, 后端服务的 异常信息,提示信息统一通过资源配置文件处理
 * 
 * 规则: 
 * 1. 模块定义 : 每个模块下建立的src/main/resources/i18n,
 *              然后建立对应的语言包信息文件 如ocm-work => ocm-work_zh_CN.properties, 目前只要中文即可 
 * 2. 使用规则 : springboot中在application.properties中增加配置信息
 *              efuture.messages.basename=i18n/ocm-work
 *              也可以直接指定目录
 *              efuture.messages.basename=i18n/ocm-work,file:/d:/opt/efuture/i18n
 *              目录支持递归定义
 *              不同包项目相同目录下相同资源文件,配置最后的启作用. 所有资源文件中相同key,排在前面的启作用
 *              
 * 3. 信息定义 ： 语言包中的信息项定义统一以模块开头：
 *              如ocm.work.save.ok=保存成功
 * 4. 使用规则 ：MessageUtils.getString("ocm.work.save.ok");
 * 
 * 注意:这里并不需要依赖spring bean的注入管理, 完全可以在spring启动之前可以使用
 * 
 * 
 * @author 肖志文 2020年6月3日 下午4:55:46
 *
 */
public class MessageUtils {

	private static MessageUtils instance = null;
	private ReloadableResourceBundleMessageSource handler = null;
	// private ResourceBundleMessageSource handler = null;
	private static Properties springEnv = new Properties();

	protected static final ThreadLocal<Locale> threadLocales = new ThreadLocal<Locale>();

	private static Locale defaultLocale = Locale.SIMPLIFIED_CHINESE;
	private static Locale failoverLocale = Locale.US;

	public static final String[] localeCodes = { "en_US", "zh_CN" };
	public static final String[] localeDescr = { "English (US)", "简体中文" };

	private static final String[] defaultProject = { "i18n/ocp-common", "i18n/ocp-boot", "i18n/ocp-job",
			"i18n/ocp-rocketmq", "i18n/ocp-taskcore", "i18n/ocp-taskflow" };

	// private static Locale locale = LocaleContextHolder.getLocale();
	public static final Map<String, String> simpleLocaleCode = new HashMap<String, String>()
	{{
		put("CN", "zh_CN");
		put("EN", "en_US");
		put("TR", "zh_TR");
	}};


	static {
		getInstance();
	}

	private MessageUtils() {
		init();
	}

	public static MessageUtils getInstance() {
		if (instance == null) {
			instance = new MessageUtils();
		}
		return instance;
	}

	private void init() {
		loadSpringEnv();
		handler = new ReloadableResourceBundleMessageSource();
		// handler = new ResourceBundleMessageSource();
		// ClassPathResource cpr = new
		// ClassPathResource("i18n/messages_zh_CN.properties");
		// handler.setResourceLoader(cpr);
		handler.setCacheSeconds(Integer
				.parseInt(getEnv("efuture.messages.cache-seconds", getEnv("spring.messages.cache-seconds", "-1"))));
		handler.setDefaultEncoding(getEnv("efuture.messages.encoding", getEnv("spring.messages.encoding", "UTF-8")));

		String[] paths = getBasenames(
				getEnv("efuture.messages.basename", getEnv("spring.messages.basename", "classpath:i18n/messages")));
		handler.setBasenames(paths);
		String codeAsDefault = getEnv("efuture.messages.use-code-as-default-message",
				getEnv("spring.messages.use-code-as-default-message", "1"));
		if ("true".equalsIgnoreCase(codeAsDefault) || "1".equalsIgnoreCase(codeAsDefault)) {
			handler.setUseCodeAsDefaultMessage(true);
		}
		// System.out.println(getEnv("efuture.messages.basename",getEnv("spring.messages.basename",
		// "classpath:i18n/messages")));
		String locale = getEnv("efuture.messages.default-locale", getEnv("spring.messages.default-locale", null));
		if (locale != null)
			defaultLocale = createLocale(locale);
		locale = getEnv("efuture.messages.failover-locale", getEnv("spring.messages.failover-locale", null));
		if (locale != null)
			failoverLocale = createLocale(locale);
		setLocale(defaultLocale);

	}

	public static Locale getLocale() {
		Locale rtn = threadLocales.get();
		if (rtn != null) {
			return rtn;
		}
		setLocale(defaultLocale);
		return defaultLocale;
	}

	public static void setLocale(Locale newLocale) {
		threadLocales.set(newLocale);
	}
	
	public static  void setLocale(String locale) {
		Locale newLocale = createLocale(locale);
		threadLocales.set(newLocale);
	}
	public static void setSimpleLocale(String locale)
	{
		setLocale( createLocale( DataUtils.nvl(simpleLocaleCode.get(locale), "zh_CN")));
	}

	protected static String getLocaleString(String locString) {
		if (locString != null && locString.length() == 5 && locString.charAt(2) == '_') {
			locString = locString.substring(0, 2).toLowerCase() + "_" + locString.substring(3).toUpperCase();
		}
		return locString;
	}

	public static Locale createLocale(String localeCode) {
		Locale resultLocale = null;
		if (localeCode != null) {
			StringTokenizer parser = new StringTokenizer(localeCode, "_"); //$NON-NLS-1$
			if (parser.countTokens() == 2) {
				resultLocale = new Locale(parser.nextToken(), parser.nextToken());
			} else {
				resultLocale = new Locale(localeCode);
			}
		}
		return resultLocale;
	}

	private static String getEnv(String key, String defaultVal) {
		String val = System.getenv(key);
		if (!isEmpty(val)) {
			return val;
		}
		val = System.getProperty(key);
		if (!isEmpty(val)) {
			return val;
		}
		val = springEnv.getProperty(key);
		if (!isEmpty(val)) {
			return val;
		}
		return defaultVal;
	}

	/**
	 * spring配置信息, 没有实际使用spring的环境类, 实际上不依赖spring的启动
	 * 
	 * @author 肖志文 2020年6月3日 下午4:48:33
	 */
	private static void loadSpringEnv() {
		String pro = System.getenv("spring.active.profile");
		String cfg = "/application";
		String sec = null;
		if (!isEmpty(pro)) {
			sec = "/application-" + pro;
		} else {
			pro = System.getProperty("spring.active.profile");
			if (!isEmpty(pro)) {
				sec = "/application-" + pro;
			}
		}
		Properties m1 = null, m2 = null;
		try {
			Resource res = new ClassPathResource(cfg + ".properties");
			m1 = PropertiesLoaderUtils.loadProperties(res);
		} catch (Exception x) {
			try {
				Resource res = new ClassPathResource(cfg + ".yml");
				m1 = PropertiesLoaderUtils.loadProperties(res);
			} catch (Exception x1) {
			}
		}
		try {
			Resource res = new ClassPathResource(sec + ".properties");
			m2 = PropertiesLoaderUtils.loadProperties(res);
		} catch (Exception x) {
			try {
				Resource res = new ClassPathResource(sec + ".yml");
				m2 = PropertiesLoaderUtils.loadProperties(res);
			} catch (Exception x1) {
			}
		}
		if (m1 != null)
			springEnv.putAll(m1);
		if (m2 != null)
			springEnv.putAll(m2);
	}

	private static String[] split(String cfg) {
		if (cfg == null)
			return null;
		else
			return cfg.split("\\,|\\|");
	}

	/**
	 * 通过配置信息来获取所有匹配的国际化配置文件
	 * 
	 * @param cfg
	 * @return
	 * @author 肖志文 2020年6月3日 下午4:47:58
	 */
	public static String[] getBasenames(String cfg) {
		String[] names = split(cfg);
		List<String> patterns = new ArrayList<String>();
		// Map<String, String> patterns = new HashMap<String, String>();
		if (names != null) {
			for (String path : names) {
				File fp;
				try {
					// Try to parse the location as a URL...
					URL url = new URL(path);
					fp = new File(url.getFile());
				} catch (MalformedURLException ex) {
					fp = new File(path);
				}
				getPaths(path, fp, patterns);

			}
		}
		for (String p : defaultProject) {
			boolean isInc = false;
			for (String s : patterns) {
				if (s.indexOf(p) != -1) {
					isInc = true;
					break;
				}
			}
			if (!isInc) {
				patterns.add(p);
			}
		}
		return (String[]) patterns.toArray(new String[patterns.size()]);
	}

	/**
	 * 国际化语言包配置在目录时,可以递归目录来处理
	 * 
	 * @param src
	 * @param dir
	 * @param paths
	 * @author 肖志文 2020年6月3日 下午4:49:24
	 */
	private static void getPaths(String src, File dir, List<String> paths) {
		String prefix = src;
		if (!src.endsWith("\\") && !src.endsWith("/")) {
			prefix = prefix + "/";
		}
		if (dir.isDirectory()) {
			File[] fps = dir.listFiles();
			for (File fp : fps) {
				if (fp.isFile() && fp.canRead()) {
					String fname = fp.getName();
					int idx = fname.lastIndexOf('.');
					if (idx != -1) {
						fname = fname.substring(0, idx);
					}
					for (String lang : localeCodes) {
						if (fname.endsWith("_" + lang)) {
							fname = fname.substring(0, fname.lastIndexOf("_" + lang));
							break;
						}
					}
					if (!paths.contains(prefix + fname)) {
						paths.add(prefix + fname);
					}
				} else if (fp.isDirectory()) {
					getPaths(prefix + fp.getName(), fp, paths);
				}
			}
		} else {
			paths.add(src);
		}
	}

	private static boolean isEmpty(String s) {
		if (s == null || s.length() == 0)
			return true;
		return false;
	}

	protected ReloadableResourceBundleMessageSource getHandler() {
		return handler;
	}

	protected static ReloadableResourceBundleMessageSource getInstanceHandler() {
		return getInstance().getHandler();
	}

	public static String getString(String key) {
		String string = null;
		try {
			string = getInstanceHandler().getMessage(key, new String[] {}, getLocale()); //defaultLocale
		} catch (Exception x) {
		}
		if (string != null)
			return string;
		try {
			string = getInstanceHandler().getMessage(key, new String[] {}, failoverLocale);
		} catch (Exception x) {
		}
		if (string != null)
			return string;
		return key;
	}

	public static String getString(String key, String... parameters) {
		String string = null;
		try {
			string = getInstanceHandler().getMessage(key, parameters, getLocale()); //defaultLocale
		} catch (Exception x) {
		}
		if (string != null)
			return string;
		try {
			string = getInstanceHandler().getMessage(key, parameters, failoverLocale);
		} catch (Exception x) {
		}
		if (string != null)
			return string;
		return key;
	}

	public static String getString(String key, Object... parameters) {
		String[] strings = new String[parameters.length];
		for (int i = 0; i < strings.length; i++) {
			strings[i] = parameters[i] != null ? parameters[i].toString() : "";
		}
		String string = null;
		try {
			string = getInstanceHandler().getMessage(key, strings, getLocale()); //defaultLocale
		} catch (Exception x) {
		}
		if (string != null)
			return string;
		try {
			string = getInstanceHandler().getMessage(key, strings, failoverLocale);
		} catch (Exception x) {
		}
		if (string != null)
			return string;
		return key;
	}

	public static final String CR = System.getProperty("line.separator");

	public static final String getClassTrace(Throwable e) {
		StringWriter stringWriter = new StringWriter();
		PrintWriter printWriter = new PrintWriter(stringWriter);
		e.printStackTrace(printWriter);
		String string = stringWriter.toString();
		try {
			stringWriter.close();
		} catch (IOException ioe) {
		} // is this really required?
		return string;
	}

	public static String getCustomTrace(Throwable aThrowable) {
		final StringBuilder result = new StringBuilder();
		String errorMessage = aThrowable.toString();
		result.append(errorMessage);
		if (!errorMessage.contains(CR)) {
			result.append(CR);
		}

		// add each element of the stack trace
		//
		for (StackTraceElement element : aThrowable.getStackTrace()) {
			result.append(element);
			result.append(CR);
		}
		return result.toString();
	}

	public static ClassLoader getDefaultClassLoader() {
		ClassLoader cl = null;
		try {
			cl = Thread.currentThread().getContextClassLoader();
		} catch (Throwable ex) {
			// Cannot access thread context ClassLoader - falling back...
		}
		if (cl == null) {
			// No thread context class loader -> use class loader of this class.
			cl = MessageUtils.class.getClassLoader();
			if (cl == null) {
				// getClassLoader() returning null indicates the bootstrap ClassLoader
				try {
					cl = ClassLoader.getSystemClassLoader();
				} catch (Throwable ex) {
					// Cannot access system ClassLoader - oh well, maybe the caller can live with
					// null...
				}
			}
		}
		return cl;
	}
}
