package com.efuture.ocp.common.distributedLock;

import java.util.HashMap;

import com.efuture.ocp.common.cache.CacheUtils;
import com.efuture.ocp.common.rest.ServiceLogs;
import com.efuture.ocp.common.util.UniqueID;

public abstract class AbstractDistributedLockHandle implements DistributedLockHandle {

	private final ThreadLocal<HashMap<String, Long>> LocalTransId = new ThreadLocal<HashMap<String, Long>>();

	protected Long newTransId(String key)
	{
		if (LocalTransId == null || LocalTransId.get() == null || !LocalTransId.get().containsKey( key ))
		{
			return UniqueID.getUniqueID();
		}
		else
		{
			return LocalTransId.get().get( key );
		}
	}

	protected long getThisTransId(String key) {
		HashMap<String, Long> trans = LocalTransId.get();
		if (trans == null) {
			return 0;
		}

		return trans.get(key);
	}

	protected void setThisTransId(String key, long id) {
		HashMap<String, Long> trans = LocalTransId.get();
		if (trans == null) {
			trans = new HashMap<String, Long>();
			LocalTransId.set(trans);
		}
		trans.put(key, id);
	}

	/**
	 * 1.调用子类的缓存实现把key和随机生成的key放入缓存 2.如果成功则放入线程对象，等待解锁时判断
	 */
	@Override
	public boolean tryLock(String key, int lockExpire, int waitMillSeconds) {
		long id = newTransId(key);
		boolean ok = doNaiveCachePut(key, id, lockExpire, waitMillSeconds);
		if (!ok) {
			return false;
		}
		setThisTransId(key, id);
		return ok;
	}

	public abstract boolean doNaiveCachePut(String key, long id, int lockExpire, int waitMillSeconds);

	/**
	 * 1.先检查当前进程有没有缓存成功的数据 2.如果有，调用子类的缓存实现，获取缓存的ID 3.如果分布式缓存ID=进程缓存的ID，则表示配对成功
	 * 4.配对成功，则调用子类的缓存实现，删除缓存 5.配对不成功，则应该只有如下这一种情况，这种情况下，线程A的处理 都已经完成了，可以不做任何处理
	 * 
	 * -- 1.线程A加锁成功 -- 2.线程A的锁到期,没有自动延期机制,缓存到期清除 -- 3.线程B加锁成功 --
	 * 4.线程A执行完，开始解锁，发现缓存的ID不是开始加锁的ID
	 * 
	 */
	@Override
	public boolean unlock(String key, int lockExpire) {
		long id = getThisTransId(key);
		try {
			long value = doNaiveCacheGet(key);
			if (id == value && value != 0 ) {
				doNaiveCacheDelete(key);
				return true;
			} else {
				ServiceLogs.debuglog("DistributedLock", "解锁时key["+key+"]的缓存ID不匹配,不做处理,[" + id + "]-[" + value + "]", 0);
				return false;
			}
		} catch (Exception e) {
			ServiceLogs.errLog("DistributedLock", e, "解锁是失败,key[{0}]", key);
		} finally {
			clearThisTransId(key);
		}
		return false;
	}

	@Override
	public boolean unlock(String key)
	{
		return unlock( key, CacheUtils.CacheTimeOut.Min );
	}
	/**
	 * 强制解锁xzw
	 */
	@Override
	public boolean forceUnlock(String key) {
		try {
			return doNaiveCacheDelete(key);
		} catch(Exception x) {
			ServiceLogs.errLog("DistributedLock", x, "强制解锁时失败,key[{0}]", key);
		}
		return false;
	}

	protected void clearThisTransId(String key) {
		HashMap<String, Long> trans = LocalTransId.get();
		if (null != trans) {
			trans.remove(key);
			if (trans.isEmpty()) {
				LocalTransId.remove();
			}
		}
	}

	public abstract boolean doNaiveCacheDelete(String key);

	public abstract long doNaiveCacheGet(String key);

}
