package com.efuture.ocp.taskcore.rocketmq;

import com.alibaba.fastjson.JSONObject;
import com.efuture.ocp.common.exception.ServiceException;
import com.efuture.ocp.common.rest.ServiceLogs;
import com.efuture.ocp.common.util.SpringBeanFactory;
import com.efuture.ocp.taskcore.consumer.ConsumerNode;
import com.efuture.ocp.taskcore.consumer.IConsumerDupHandle;
import com.efuture.ocp.taskcore.consumer.IConsumerHandle;
import com.efuture.ocp.taskcore.message.IMessageHandle;
import com.efuture.ocp.taskcore.message.Message;
import com.efuture.ocp.taskcore.message.MessageStatus;
import com.efuture.ocp.taskcore.service.IMessageDupHandle;
import org.apache.log4j.Logger;
import org.apache.rocketmq.common.message.MessageExt;
import org.apache.rocketmq.spring.starter.annotation.RocketMQMessageListener;
import org.apache.rocketmq.spring.starter.core.RocketMQListener;
import org.springframework.util.StringUtils;

import java.util.Date;

public class DefaultFutRocketMqConsumerListener implements RocketMQListener<MessageExt> {

    private static Logger logger = Logger.getLogger(DefaultFutRocketMqConsumerListener.class);
    private IMessageDupHandle messagedup;
    private IMessageHandle messagehandle;
    private ConsumerNode nodeconfig;
    private IConsumerHandle consumerHandler;
    private String consumerobj;
    private String consumerGroup;

    public DefaultFutRocketMqConsumerListener() {
        super();
        setConsumerGroupbyCalss();
    }

    public IMessageDupHandle getMessagedup() {
        if (messagedup == null) {
            messagedup = SpringBeanFactory.getBean("MessageDupService", IMessageDupHandle.class);
        }
        return messagedup;
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    public void setConsumerGroupbyCalss() {
        Class cls = this.getClass();
        if (cls.isAnnotationPresent(RocketMQMessageListener.class)) {
            RocketMQMessageListener s = (RocketMQMessageListener) cls.getAnnotation(RocketMQMessageListener.class);
            this.setConsumerGroup(s.consumerGroup());
        }

    }

    public void lockMessage(Message msg) {
        if (getMessagehandle().lockMessage(nodeconfig, msg) < 0) {
            msg.setStatus(MessageStatus.FAIL.name());
            msg.setMsg("锁定消息失败");
            consumeCallbackMessage(msg);
            throw new ServiceException("80001", "锁定消息[{0}]失败,等重试", msg.getKeyvalue());
        }
    }

    @Override
    public void onMessage(MessageExt msgExt) {

        JSONObject jobj = (JSONObject) JSONObject.parse(msgExt.getBody());
        Message msg = JSONObject.toJavaObject(jobj, Message.class);
        msg.setExecdate(new Date());
        msg.setExecnode(this.getConsumerGroup());
        // 锁定消息，锁定失败抛异常
        lockMessage(msg);

        long stime = System.currentTimeMillis();
        ServiceLogs.logConsumeMqStart(msg.getTopic(), msg.getKeyvalue(), msgExt.getMsgId());
        if (getMessagedup() != null && getMessagedup().checkMessageDup(msg) < 1) {
            // message is dup
            msg.setStatus(MessageStatus.DUP.name());
            msg.setMsg("消息重复,不执行");
            IConsumerHandle ch = getConsumerHandler();
            if (ch instanceof IConsumerDupHandle) {
                ((IConsumerDupHandle) ch).isDup(nodeconfig, msg);
            }
            ServiceLogs.logConsumeMqSuccess(msg.getTopic(), msg.getKeyvalue(), msg.getStatus(), stime);
            consumeCallbackMessage(msg);
            return;
        }
        try {
            doConsumer(msg);
            if (!msg.getStatus().equalsIgnoreCase(MessageStatus.NORUN.name())) {
                msg.setStatus(MessageStatus.SUCCESS.name());
            }
            if (messagedup != null) {
                getMessagedup().onSuccess(msg);
            }
            ServiceLogs.logConsumeMqSuccess(msg.getTopic(), msg.getKeyvalue(), msg.getStatus(), stime);
        } catch (Exception e) {
            if (messagedup != null) {
                getMessagedup().onFail(msg);
            }
            logger.error("", e);
            msg.setStatus(MessageStatus.FAIL.name());
            String msgstr = e.getMessage();
            if (!StringUtils.isEmpty(msgstr)) {
                if (msgstr.length() > 1900) {
                    msgstr = msgstr.substring(0, 1900);
                }
                msg.setMsg(msgstr);
            } else {
                msg.setMsg("Unknown Exception");
            }
            ServiceLogs.logConsumeMqError(msg.getTopic(), msg.getKeyvalue(), msg.getStatus(), msg.getMsg(), stime);
            throw new ServiceException("7000", e.getMessage());
        } finally {
            consumeCallbackMessage(msg);
        }

    }

    public IConsumerHandle getConsumerHandler() {
        if (consumerHandler == null) {
            consumerHandler = SpringBeanFactory.getBean(consumerobj, IConsumerHandle.class);
        }
        return consumerHandler;
    }

    public void setConsumerHandler(IConsumerHandle consumerHandler) {
        this.consumerHandler = consumerHandler;
    }

    public IMessageHandle getMessagehandle() {
        if (messagehandle == null) {
            messagehandle = new RocketMqMessageHandle();
        }
        return messagehandle;
    }

    private void doConsumer(Message msg) throws Exception {
        if (nodeconfig == null) {
            nodeconfig = new ConsumerNode();
        }
        getConsumerHandler().consume(nodeconfig, msg);

    }

    private void consumeCallbackMessage(Message msg) {
        try {
            getMessagehandle().consumeCallbackMessage(nodeconfig, msg);
        } catch (Exception e) {
            logger.error("[" + msg.getKeyvalue() + "]调用consumeCallbackMessage发生错误:", e);
        }
        getMessagehandle().unLockMessage(nodeconfig, msg);
    }

    public String getConsumerobj() {
        return consumerobj;
    }

    public void setConsumerobj(String consumerobj) {
        this.consumerobj = consumerobj;
    }

    public String getConsumerGroup() {
        return consumerGroup;
    }

    public void setConsumerGroup(String consumerGroup) {
        this.consumerGroup = consumerGroup;
    }

}
