package com.efuture.ocp.taskcore.rocketmq;

import com.efuture.ocp.common.rest.ServiceLogs;
import com.efuture.ocp.common.util.SpringBeanFactory;
import com.efuture.ocp.common.util.UniqueID;
import com.efuture.ocp.taskcore.consumer.ConsumerNode;
import com.efuture.ocp.taskcore.message.IMessageHandle;
import com.efuture.ocp.taskcore.message.Message;
import com.efuture.ocp.taskcore.message.MessageStatus;
import com.efuture.omd.storage.FStorageOperations;
import org.apache.log4j.Logger;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.common.message.MessageConst;
import org.apache.rocketmq.spring.starter.core.RocketMQTemplate;
import org.springframework.messaging.support.MessageBuilder;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;

public class RocketMqMessageHandle implements IMessageHandle {

	private static Logger logger = Logger.getLogger(RocketMqMessageHandle.class);
	public final static String StorageOperation = "StorageOperation_task";
	public final static String message_tablename = "task_message";
	public final static String message_success_tablename = "task_message_success";
	public final static String message_error_tablename = "task_message_error";

	@Resource
	private RocketMQTemplate rocketMQTemplate;

	public FStorageOperations getStorageOperations() {
		return SpringBeanFactory.getBean(StorageOperation, FStorageOperations.class);
	}
	
	@Transactional(propagation = Propagation.REQUIRES_NEW ,value = "tasktransactionManager")
	public int produce(Message msg) {
		// 使用rocketmq的接口发送消息，如果发送错误，则写到数据库中
		// 如果数据库写失败，则记录到日志中
		// 如果是重新发消息，则不记录到数据库中，因为重发消息都是从数据库中取出来再重发的。
		boolean ibRetry = false;
		if(MessageStatus.RETRY.name().equals(msg.getStatus())){
			ibRetry = true;
		}
		msg.setId(UniqueID.getUniqueID());
		msg.setCreatedate(new Date());
		msg.setStatus(MessageStatus.NEW.name());
		msg.setKeyvalue(msg.getTopic() + "-" + msg.getKeyvalue());
		long stime = System.currentTimeMillis();
		try {
			// logger.debug("-----------rocketmq------开始发送--" +
			// msg.getKeyvalue());
			ServiceLogs.logSendMqStart(msg.getTopic(), msg.getKeyvalue(), "");
			SendResult sr = rocketMQTemplate.syncSend(msg.getTopic() + ":" + msg.getEvent(),
					MessageBuilder.withPayload(msg).setHeader(MessageConst.PROPERTY_KEYS, msg.getKeyvalue()).build());
			// sr.getSendStatus()
			ServiceLogs.logSendMqSuccess(msg.getTopic(), msg.getKeyvalue(), sr.toString(), "", stime);
			return IMessageHandle.msg_produce_rtncode.SUCCESS;
		} catch (Exception e) {
			ServiceLogs.logSendMqError(msg.getTopic(), msg.getKeyvalue(), "1.1", "发送失败，后续保存到数据库-->"+e.getMessage(), "", stime);
			msg.setMsg(e.getMessage());
			// 如果是重新发消息，则不记录到数据库中，因为重发消息都是从数据库中取出来再重发的。
			if(!ibRetry){
				insdb(msg, stime);
			}
			
			return IMessageHandle.msg_produce_rtncode.COMM_ERR;
		}

	}
	
	@Transactional(propagation = Propagation.REQUIRES_NEW ,value = "tasktransactionManager")
	public void insdb(Message msg, long stime) {
		try {
			getStorageOperations().insert(msg, message_tablename);
		} catch (Exception e) {
			msg.setMsg(e.getMessage());
			// 日志记录时 不在记录详细消息信息，只记录key,这样在日志上好检查，业务上需要保证KEY可追溯
			msg.setData("");
			ServiceLogs.logSendMqError(msg.getTopic(), msg.getKeyvalue(), "1.2", "保存数据库失败,请手工发送",
					msg.getMsg(), stime);
		}

	}

	public List<Message> pullNewMessage(ConsumerNode node) {
		logger.warn("rocketmq 不支持 pullNewMessage");
		return null;
	}

//	public int lockMessage(ConsumerNode node, Message msg) {
//		logger.warn("rocketmq 不支持 lockMessage");
//		return 0;
//	}
	
	@Transactional(propagation = Propagation.REQUIRES_NEW ,value = "tasktransactionManager")
	public void consumeCallbackMessage(ConsumerNode node, Message msg) {
		if (msg == null) {
			return;
		}
		try {
			msg.setData("");
			if (!msg.getStatus().equalsIgnoreCase(MessageStatus.FAIL.name())) {
				getStorageOperations().insert(msg, message_success_tablename);
			} else {
				getStorageOperations().insert(msg, message_error_tablename);

			}

		} catch (Exception e) {
			msg.setMsg(e.getMessage());
			ServiceLogs.logConsumeMqError(msg.getTopic(), msg.getKeyvalue(), "2.1", "消息回调,保存到数据库失败:" + e.getMessage(),
					msg.getExecdate().getTime());
		}
	}

}
