package com.efuture.taskflow.repository;


import com.alibaba.fastjson.JSONObject;
import com.efuture.ocp.common.entity.AbstractEntityBean;
import com.efuture.ocp.common.exception.ServiceException;
import com.efuture.ocp.common.rest.ServiceLogs;
import com.efuture.ocp.common.util.*;
import com.efuture.omd.storage.FMybatisTemplate;
import com.efuture.omd.storage.FStorageOperations;
import com.efuture.taskflow.TaskConstant;
import com.efuture.taskflow.entity.SubTaskStatusInfo;
import com.efuture.taskflow.entity.Task;
import com.efuture.taskflow.entity.TaskErrorInfo;
import com.efuture.taskflow.entity.TaskExecCompleteLog;
import com.efuture.taskflow.exception.TaskExceptionCode;
import com.efuture.taskflow.utils.DataBaseConfigUtils;
import com.efuture.taskflow.utils.Utils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.annotation.Transient;
import org.springframework.data.domain.Sort;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

@Component("taskRepositoryBydb")
public class TaskRepositoryDbImpl implements TaskRepository {


    @Autowired
    BatchInsService batchIns;
    List<String> ALL_TASK_TABLE = null;
    List<String> notCompletedTaskTables = null;

    private String getTableName(String OriTableName) {
        return DataBaseConfigUtils.getTableName(OriTableName);
    }

    @Override
    @Transactional
    public Task save(Task task) {
        if (task.getPh_key() == 0) {
            task.setPh_key(UniqueID.getUniqueID());
        } else {
            ServiceLogs.debuglog("Task-SAVE", "ph_key不为0,不保存数据库", 0);
            return task;
        }
        task.setTask_status(TaskConstant.TASK_STATUS.SAVE);
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();

//        Criteria criteria = Criteria.where("ent_id").is(task.getEnt_id());
//        criteria.and("billno").is(task.getBillno());
//        Query query = new Query(criteria);
//        query.limit(1);
        //有可能已经存在，但是数据已经变化 比如ph_key已经变了，这时直接返回就有可能会出现bug，by zhouwd  2020-6-28
        //先查询，如果存在直接返回查询的这个task，
        Task extTask = checkTaskExist(task);
        if (extTask == null) {
            fso.insert(task, getTableName(DataBaseConfigUtils.TASK_UNFINISHED_TABLE_NAME));
        } else {
            extTask.setData(task.getData());
            task = extTask;
        }
//        if (fso.count(query, getTableName(DataBaseConfigUtils.TASK_UNFINISHED_TABLE_NAME)) <= 0) {
//            fso.insert(task, getTableName(DataBaseConfigUtils.TASK_UNFINISHED_TABLE_NAME));
//        }

        return task;

    }

    private Map<String, Object> taskToColMap(Object task) {
        Map<String, Object> rtn = new HashMap<String, Object>();
        StringBuffer sb = new StringBuffer();
        AbstractEntityBean.fetchAllDeclaredField(task.getClass(), sb);
        String[] flds = sb.toString().split(",");
        Set<BeanField> beanFields = new HashSet<BeanField>();

        for (String col : flds) {
            java.lang.reflect.Field fld = AbstractEntityBean.fetchDeclaredField(Task.class, col);
            // Transient表示存储时忽略的列
            if (fld != null && fld.getAnnotation(Transient.class) == null) {
                try {
                    rtn.put(col, fld.get(task));
                } catch (IllegalArgumentException e) {
                    TaskExceptionCode.GET_OBJ_COLDATA.throwThisException(task.toString(), col, e.getMessage());
                } catch (IllegalAccessException e) {
                    TaskExceptionCode.GET_OBJ_COLDATA.throwThisException(task.toString(), col, e.getMessage());
                }
            }
        }
        // 添加扩展字段
        //String[] colArray = TaskParam.TASK_PUBLIC.TASK_EXT_COLS.getVal(task.getEnt_id(), "").split(",");
        //Map<String, Object> extInfo = task.getExtInfo();
//        if (colArray.length > 0 && extInfo != null) {
//            for (int i = 0; i < colArray.length; i++) {
//                String col = colArray[i];
//                if (extInfo.containsKey(col)) {
//                    rtn.put(col, extInfo.get(col));
//                }
//            }
//        }
        return rtn;
    }

    @Override
    @Transactional
    public void batchSave(List<Task> taskList) {
        if (taskList == null) {
            return;
        }
        for (Task task : taskList) {
            save(task);
        }

    }

    @Override
    public Task checkTaskExist(Task task) {
        return findTaskByBillnoByTables(task.getEnt_id(), task.getBillno(), getNotCompletedTables());
    }

    @Override
    public Task findTaskByBillno(long entId, String billNO) {
        return findTaskByBillnoByTables(entId, billNO, getAllTableList());
    }

    public Task findTaskByBillnoByTables(long entId, String billNO, List<String> tables) {
        Criteria criteria = Criteria.where("ent_id").is(entId);
        criteria.and("billno").is(billNO);
        Query query = new Query(criteria);
        query.limit(1);
        return (Task) findOneByTableList(query, tables, Utils.getTaskClass(entId));
    }

    private List<String> getAllTableList() {
        if (ALL_TASK_TABLE == null) {
            ALL_TASK_TABLE = new ArrayList<String>();
            ALL_TASK_TABLE.add(getTableName(DataBaseConfigUtils.TASK_UNFINISHED_TABLE_NAME));
            ALL_TASK_TABLE.add(getTableName(DataBaseConfigUtils.TASK_WAIT_SUB_TABLE_NAME));
            ALL_TASK_TABLE.add(getTableName(DataBaseConfigUtils.TASK_SUCCESS_TABLE_NAME));
            ALL_TASK_TABLE.add(getTableName(DataBaseConfigUtils.TASK_ERRORLIST_TABLE_NAME));
        }

        return ALL_TASK_TABLE;
    }

    private <T> T findOneByTableList(Query query, List<String> tables, Class<T> cls) {
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        for (String table : tables) {
            T task = fso.selectOne(query, cls, table);
            if (task != null) {
                ((Task) task).setSrc_table(table);

                //System.out.println(JSONObject.toJSON(task));
                return task;
            }
        }

        return null;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void logError(Task task, String message) {
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        Date now = new Date();
        task.setLast_exec_date(now);
        task.setLast_exec_error(message);
        if (task.getPh_timestamp() == null) {
            task.setPh_timestamp(now);
        }
        fso.insert(task, getTableName(DataBaseConfigUtils.TASK_ERRORLOG_TABLE_NAME));
    }

    @Override
    @Transactional
    public void updateTask(Task task, String uptCols) {
        Criteria criteria = Criteria.where("ph_key").is(task.getPh_key()).and("task_status").lte(task.getTask_status());
        Query queryByKey = new Query(criteria);
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        Update update = null;
        Set<String> keys = new HashSet<String>(Arrays.asList(uptCols.split(",")));
        try {
            update = StorageUtils.createUpdateFormBean(task, keys);
        } catch (IllegalArgumentException | IllegalAccessException e) {
            throw new ServiceException(TaskExceptionCode.GET_UPT_INFO_ERROR.getCode(),
                    TaskExceptionCode.GET_UPT_INFO_ERROR.getMsgTemplate(), e.getMessage());
        }
        String tableName = getTableName(DataBaseConfigUtils.TASK_UNFINISHED_TABLE_NAME);

        if (task.getTask_status() == TaskConstant.TASK_STATUS.WAIT_SUB_COMPLETE) {
            tableName = getTableName(DataBaseConfigUtils.TASK_WAIT_SUB_TABLE_NAME);
        }
        if (task.getSrc_table() != null && !task.getSrc_table().equalsIgnoreCase("MQ")) {
            tableName = task.getSrc_table();
        }
        int rtn = fso.update(queryByKey, update, tableName);
        if (rtn == 0) {
            ServiceLogs.truedebuglog("TaskRepository", "更新任务[{0}]状态到[{1}]时更新行数为0", 0, task.getBillno(),
                    task.getTask_status());
        }
    }

    @Override
    @Transactional
    public Task findTaskById(long ent_id, long ph_key) {
        Criteria criteria = Criteria.where("ph_key").is(ph_key).and("ent_id").is(ent_id);
        Query queryByKey = new Query(criteria);
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        return (Task) fso.selectOne(queryByKey, Utils.getTaskClass(ent_id), getTableName(DataBaseConfigUtils.TASK_UNFINISHED_TABLE_NAME));
    }

    @Override
    @Transactional
    public void moveTaskToSuccess(Task task) {
        Criteria criteria = Criteria.where("ph_key").is(task.getPh_key());
        Query queryByKey = new Query(criteria);
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        task.setPh_timestamp(new Date());
        if (fso.count(queryByKey, getTableName(DataBaseConfigUtils.TASK_SUCCESS_TABLE_NAME)) <= 0) {
            fso.insert(task, getTableName(DataBaseConfigUtils.TASK_SUCCESS_TABLE_NAME));
        }
        String table = task.getSrc_table();
        if (table.equalsIgnoreCase("MQ")) {
            table = getTableName(DataBaseConfigUtils.TASK_UNFINISHED_TABLE_NAME);
        }
        fso.delete(queryByKey, table);
        //by zhouwd 2020-7-31 转移到成功任务表中时，永远要清理下错误日志表
        clearTaskErrorLog(task);
    }

    @Override
    @Transactional
    public void moveTaskToWaitSub(Task task) {
        Criteria criteria = Criteria.where("ph_key").is(task.getPh_key());
        Query queryByKey = new Query(criteria);
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        task.setPh_timestamp(new Date());
        task.setExec_count(0);
        fso.insert(task, getTableName(DataBaseConfigUtils.TASK_WAIT_SUB_TABLE_NAME));
        fso.delete(queryByKey, getTableName(DataBaseConfigUtils.TASK_UNFINISHED_TABLE_NAME));
        //by zhouwd 2020-7-31 转移到等待任务表中时，永远要清理下错误日志表
        clearTaskErrorLog(task);
    }

    @Override
    @Transactional
    public void moveTaskToErrorList(Task task) {
        Criteria criteria = Criteria.where("ph_key").is(task.getPh_key());
        Query queryByKey = new Query(criteria);
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        task.setPh_timestamp(new Date());
        //task.setExec_count(0);
        fso.insert(task, getTableName(DataBaseConfigUtils.TASK_ERRORLIST_TABLE_NAME));
        fso.delete(queryByKey, getTableName(DataBaseConfigUtils.TASK_UNFINISHED_TABLE_NAME));
        fso.delete(queryByKey, getTableName(DataBaseConfigUtils.TASK_WAIT_SUB_TABLE_NAME));
        //by zhouwd 2020-7-31 转移到错误任务表中时，永远要清理下错误日志表
        clearTaskErrorLog(task);
    }

    @Override
    @Transactional
    public void clearTaskErrorLog(Task task) {
        Criteria criteria = Criteria.where("ph_key").is(task.getPh_key());
        Query queryByKey = new Query(criteria);
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        List<Task> errlog = fso.select(queryByKey, Utils.getTaskClass(task.getEnt_id()), getTableName(DataBaseConfigUtils.TASK_ERRORLOG_TABLE_NAME));
        fso.delete(queryByKey, getTableName(DataBaseConfigUtils.TASK_ERRORLOG_TABLE_NAME));
        fso.insert(errlog, getTableName(DataBaseConfigUtils.HIS_TASK_ERRORLOG_TABLE_NAME));
        // for(Task errtask:errlog){
        //
        // }
        // try {
        // batchIns.batchInsert((FMybatisTemplate) fso, errlog,
        // getTableName(HIS_TASK_ERRORLOG_TABLE_NAME), 1000);
        // } catch (Exception e) {
        // throw new
        // ServiceException(TaskExceptionCode.BATCHSAVE_ERROR.getCode(),
        // TaskExceptionCode.BATCHSAVE_ERROR.getMsgTemplate(), e.getMessage());
        // }
    }

    @Override
    @Transactional
    public void insExecCompleteLog(Task task) {
        TaskExecCompleteLog log = new TaskExecCompleteLog(task);
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        fso.insert(log, getTableName(DataBaseConfigUtils.TASK_EXEC_COMPLETE_LOG));
    }

    @Override
    @Transactional
    public void delExecCompleteLog(Task task) {
        Criteria criteria = Criteria.where("ent_id").is(task.getEnt_id()).and("billno").is(task.getBillno());
        Query query = new Query(criteria);
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        fso.delete(query, getTableName(DataBaseConfigUtils.TASK_EXEC_COMPLETE_LOG));
    }

    @Override
    public SubTaskStatusInfo querySubTaskStatus(long ent_id, String billno) {
        Criteria criteria = Criteria.where("ent_id").is(ent_id).and("parent_billno").is(billno);
        Query query = new Query(criteria);
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        long successNum = fso.count(query, getTableName(DataBaseConfigUtils.TASK_SUCCESS_TABLE_NAME));

        long errNum = fso.count(query, getTableName(DataBaseConfigUtils.TASK_ERRORLIST_TABLE_NAME));

        long excingNum = fso.count(query, getTableName(DataBaseConfigUtils.TASK_UNFINISHED_TABLE_NAME));

        SubTaskStatusInfo rtn = new SubTaskStatusInfo(successNum, errNum, excingNum);
        return rtn;
    }

    @Override
    public boolean checkTaskIsExecComplete(Task task) {
        Criteria criteria = Criteria.where("ent_id").is(task.getEnt_id()).and("billno").is(task.getBillno());
        Query query = new Query(criteria);
        query.limit(1);
        query.fields().include("billno");
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        Map<String, Object> rtn = fso.selectOne(query, getTableName(DataBaseConfigUtils.TASK_EXEC_COMPLETE_LOG));
        if (rtn == null || rtn.isEmpty()) {
            return false;
        }
        return rtn.size() > 0;
    }

//    /**
//     * 查询未完成的任务列表
//     */
//    @Override
//    public List<Task> queryNotCompletedTaskInfo(int overMins) {
//        Date ltedate = DateUtils.addMinutes(new Date(), -1 * overMins);
//        Criteria criteria = Criteria.where("ph_timestamp").lte(ltedate);
//        Query query = new Query(criteria);
//        query.limit(1000);
//        List<String> tables = getNotCompletedTables();
//        List<Task> taskList = queryFromTables(query, tables);
//        return taskList;
//    }

    @Override
    public TaskErrorInfo queryTaskErrorInfo(long ph_key) {
        Criteria criteria = Criteria.where("ph_key").is(ph_key);
        Query queryByKey = new Query(criteria);
        queryByKey.fields().include("last_exec_date").include("last_exec_error");
        queryByKey.with(new Sort(Direction.DESC, "last_exec_date"));
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        List<Map<String, Object>> errLog = fso.select(queryByKey, getTableName(DataBaseConfigUtils.TASK_ERRORLOG_TABLE_NAME));
        int count = errLog.size();
        long lastDate = 0;
        String LastMsg = null;
        if (count > 0) {
            lastDate = Long.valueOf(errLog.get(0).get("last_exec_date").toString());
            LastMsg = (String) errLog.get(0).get("last_exec_error");

        }
        TaskErrorInfo rtn = new TaskErrorInfo(count, lastDate, LastMsg);
        return rtn;
    }

    @Override
    public void updateSubTaskStatus(String billNo, int status) {
        Criteria criteria = Criteria.where("billno").is(billNo);
        criteria.orOperator(Criteria.where("parent_billno").is(billNo));
        Query query = new Query(criteria);
        Update update = new Update();
        update.set("task_status", status);
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        List<String> tables = getNotCompletedTables();
        for (String table : tables) {
            fso.update(query, update, table);
        }
    }

    @Override
    public List<Task> querySubTaskNotLast(String billNo) {
        Criteria criteria = Criteria.where("isMainTask").ne("L");
        criteria.andOperator(Criteria.where("billno").is(billNo).orOperator(Criteria.where("parent_billno").is(billNo)));
        Query query = new Query(criteria);
        query.limit(1000);
        List<String> tables = getNotCompletedTables();
        List<Task> taskList = queryFromTables(query, tables);
        return taskList;
    }

    @Override
    public List<Task> queryUnfinishedTaskInfo(Date startDate, Date endDate, long lastPhkey) {
        Criteria criteria = null;

        if (startDate != null) {
            criteria = Criteria.where("ph_timestamp").lte(endDate).gte(startDate);
        } else {
            criteria = Criteria.where("ph_timestamp").lte(endDate);
        }
        criteria.and("ph_key").gt(lastPhkey);
        Query query = new Query(criteria);
        query.limit(1000);
        query.with(new Sort(Direction.ASC, "ph_timestamp", "ph_key"));
        //query.with(new Sort(Direction.ASC, ));
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        String table = getTableName(DataBaseConfigUtils.TASK_UNFINISHED_TABLE_NAME);
        List<Task> taskList = fso.select(query, Utils.getTaskClass(0), table);
        for (Task task : taskList) {
            task.setSrc_table(table);
        }
        return taskList;
    }

    @Override
    public List<Task> queryWaitSubTaskInfo(Date startDate, Date endDate, long lastPhkey) {
        Criteria criteria = null;

        if (startDate != null) {
            criteria = Criteria.where("ph_timestamp").lte(endDate).gte(startDate);
        } else {
            criteria = Criteria.where("ph_timestamp").lte(endDate);
        }
        criteria.and("ph_key").gt(lastPhkey);
        Query query = new Query(criteria);
        query.limit(1000);
        query.with(new Sort(Direction.ASC, "ph_timestamp", "ph_key"));
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        String table = getTableName(DataBaseConfigUtils.TASK_WAIT_SUB_TABLE_NAME);
        List<Task> taskList = fso.select(query, Task.class, table);
        for (Task task : taskList) {
            task.setSrc_table(table);
        }
        return taskList;
    }

    @Override
    public List<Task> queryTaskList(JSONObject condition) {
        String status = condition.getString("task_status");
        condition.remove("task_status");
        Query query = StorageUtils.createQueryFormJson(condition, Task.class);
        List<String> tables = new ArrayList<>();
        if (status.equalsIgnoreCase(QueryTaskCondition.TASK_STATUS.FAILED)) {
            tables.add(DataBaseConfigUtils.getTableName(DataBaseConfigUtils.TASK_ERRORLIST_TABLE_NAME));
        } else if (status.equalsIgnoreCase(QueryTaskCondition.TASK_STATUS.SUCCEED)) {
            tables.add(DataBaseConfigUtils.getTableName(DataBaseConfigUtils.TASK_SUCCESS_TABLE_NAME));
        } else {
            tables.addAll(getNotCompletedTables());
        }
        List<Task> taskList = queryFromTables(query, tables);
        return taskList;
    }

    /**
     * 查询错误分组的错误任务条数
     *
     * @param startDay
     * @param endDay
     * @return
     */
    @Override
    public Map<String, Long> queryFailedTaskErrorGroup(Date startDay, Date endDay) {
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        FMybatisTemplate ft = (FMybatisTemplate) fso;
        String table = getTableName(DataBaseConfigUtils.TASK_ERRORLIST_TABLE_NAME);
        String sDay = DateFormatUtils.format(startDay, "yyyy-MM-dd");
        String eDay = DateFormatUtils.format(endDay, "yyyy-MM-dd");
        String sql = "select last_exec_errorGroup errorGrp,count(*) grpCount from " + table + " ";
        sql = sql + "where ph_timestamp >='" + sDay + "' and ph_timestamp < '" + eDay + "' ";
        sql = sql + "group by last_exec_errorGroup";
        List<Map<String, Object>> rtn = ft.getSqlSessionTemplate().selectList("select", sql);
        Map<String, Long> rtnMap = new HashMap<>();
        for (Map row : rtn) {
            rtnMap.put(MapUtils.getString(row, "errorGrp"), MapUtils.getLongValue(row, "grpCount", 0));
        }
        return rtnMap;
    }

    @Override
    public long querySucceedTaskCount(Date startDay, Date endDay) {
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        String table = getTableName(DataBaseConfigUtils.TASK_SUCCESS_TABLE_NAME);
        Criteria criteria = Criteria.where("ph_timestamp").lt(endDay).gte(startDay);
        Query query = new Query(criteria);
        return fso.count(query, table);
    }

    private List<Task> queryFromTables(Query query, List<String> tables) {
        List<Task> taskList = new ArrayList<Task>();
        FStorageOperations fso = DataBaseConfigUtils.getStorageOperations();
        for (String table : tables) {
            List<Task> qtask = fso.select(query, Task.class, table);
            for (Task task : qtask) {
                task.setSrc_table(table);
            }
            taskList.addAll(qtask);
        }
        return taskList;
    }

    private List<String> getNotCompletedTables() {
        if (notCompletedTaskTables == null) {
            notCompletedTaskTables = new ArrayList<String>();
            notCompletedTaskTables.add(getTableName(DataBaseConfigUtils.TASK_UNFINISHED_TABLE_NAME));
            notCompletedTaskTables.add(getTableName(DataBaseConfigUtils.TASK_WAIT_SUB_TABLE_NAME));
        }

        return notCompletedTaskTables;
    }

}
