package com.efuture.taskflow.taskmanager;

import com.efuture.ocp.common.distributedLock.DLock;
import com.efuture.ocp.common.exception.ServiceException;
import com.efuture.ocp.common.rest.ServiceLogs;
import com.efuture.ocp.common.util.CacheUtils;
import com.efuture.taskflow.TaskComponentFactory;
import com.efuture.taskflow.TaskConstant;
import com.efuture.taskflow.entity.InterTaskStatus;
import com.efuture.taskflow.entity.SubTaskStatusInfo;
import com.efuture.taskflow.entity.Task;
import com.efuture.taskflow.entity.TaskErrorInfo;
import com.efuture.taskflow.exception.TaskExceptionCode;
import com.efuture.taskflow.param.TaskParam;
import com.efuture.taskflow.repository.TaskRepository;
import com.efuture.taskflow.taskdata.TaskDataQuery;
import com.efuture.taskflow.work.TaskWorkerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.LinkedBlockingQueue;

@Component("defaultTaskStatusManager")
public class TaskStatusManagerImpl implements TaskStatusManager {//, ApplicationListener<ContextClosedEvent> {
    public static String logtype = "TaskStatusManager";

    // LinkedBlockingQueue<TaskStatusChangeEvent> subTaskCompletedEvents = new
    // LinkedBlockingQueue<TaskStatusChangeEvent>(
    // 50000);
    LinkedBlockingQueue<TaskStatusChangeEvent> taskStatusChangedEvents = new LinkedBlockingQueue<TaskStatusChangeEvent>(
            50000);
    @Autowired
    TaskRepository taskRepo;
    boolean ibExit = false;
    Thread taskStatusChangedHandleThread = null;

    // private void notificationParent(TaskStatusChangeEvent event) {
    // if (event.getTask().isThisComplete() && event.getTask().hasParent()) {
    // subTaskCompletedEvents.add(event);
    // }
    //
    // }

    public TaskStatusManagerImpl() {
        super();
        start();
    }

    //TaskWorkerFactory taskWorkerFactory;

    //TaskDataQuery taskDataQuery;

    public TaskRepository getTaskRepo() {
        return taskRepo;
    }

    @Override
    public void taskStatusChanged(Task task, int newStatus) {
        // 先更新缓存，再放入队列，然后判断是否需要立即刷新，如果需要立即刷新的，则直接存盘
        // 统一管理状态
        int oldStatus = task.getTask_status();
        if (newStatus == TaskConstant.TASK_STATUS.WAIT_SUB_COMPLETE && task.getTot_subtask_num() == 0) {
            newStatus = TaskConstant.TASK_STATUS.ALL_COMPLETE;
        }
        task.setTask_status(newStatus);
        // 设置缓存
        setValToCached(task.getStatusKey(), task.getStatusValue());
        // 放入缓冲
        TaskStatusChangeEvent event = new TaskStatusChangeEvent(task, oldStatus);
        publishEvent(event);

        // 通知上一级
        // notificationParent(event);

    }

    private TaskStatusManager getTaskStatusManager() {
        return TaskComponentFactory.getTaskStatusManager();
    }

    private TaskWorkerFactory getTaskWorkerFactory() {
        return TaskComponentFactory.getTaskWorkerFactory();
    }

    private TaskDataQuery getTaskDataQuery() {
        return TaskComponentFactory.getTaskDataQuery();
    }

    /**
     * 1.如果是强制保存的，就直接保存，否则放入队列中等待后续同步 2.如果保存状态失败，则先放入缓存，还是放入MQ？
     *
     * @param event
     */
    public void publishEvent(TaskStatusChangeEvent event) {
        if (event.isSaveImmediately()) {
            try {
                getTaskStatusManager().updateTask(event.getTask(), event.getOldStatus());
            } catch (Exception e) {
                ServiceLogs.errLog(logtype, e, "更新task[{0}]状态时错误,放入缓冲队列", event.getTask().getBillno());
                taskStatusChangedEvents.add(event);
            }
        } else {
            taskStatusChangedEvents.add(event);
        }

    }
    // Thread subTaskCompletedHandleThread = null;

    private void start() {

        taskStatusChangedHandleThread = new Thread(new Runnable() {
            public void run() {
                try {
                    doHandleTaskStatusChanged();
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
        }, "taskStatusChangedHandle");
        //Runtime.getRuntime().addShutdownHook(taskStatusChangedHandleThread);
        taskStatusChangedHandleThread.start();
        //
        // subTaskCompletedHandleThread = new Thread(new Runnable() {
        // public void run() {
        // try {
        // doHandleSubTaskCompleted();
        // } catch (InterruptedException e) {
        // e.printStackTrace();
        // }
        // }
        // }, "subTaskCompleted");
        // subTaskCompletedHandleThread.start();

    }

    // 刷新TASK状态
    @Override
    @DLock(key = "#args[0].getLockKey()")
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void refreshTaskStatus(Task task) {
        /**
         * 1.同步状态 2.分3种状态处理 1) 还未开始 2) 执行中 3) 等待子任务完成
         */

        //公共检查
        task = syncTaskStatus(task);
        if (task.isNotStart()) {
            handleNotStartTaskStatus(task);
        } else if (task.isExecing()) {
            handleExecingTaskStatus(task);
        } else if (task.isWaitSubComplete()) {
            handleWaitSubCompleteTaskStatus(task);
        } else if (task.isStoped()) {
            handleStopTaskStatus(task);
        } else {
            // 未处理的状态 都是执行完毕的状态，可能执行完了，也可能执行错误
        }
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void refreshTaskError(Task task, String errMsg) {
        int needCheckCount = TaskParam.TASK_PUBLIC.MAX_CHECK_COUNT.getIntVal(task.getEnt_id());
        task.setExec_count(task.getExec_count() + 1);
        task.setLast_exec_error(errMsg);
        task.setLast_exec_date(System.currentTimeMillis() / 1000);
        // 增加错误CODE和GROUP

        if (task.getExec_count() > needCheckCount) {
            onOverMaxCheckTimes(task);
        } else {
            String uptCols = "last_exec_date,exec_count,last_exec_error";
            getTaskRepo().updateTask(task, uptCols);
        }
    }

    private void handleStopTaskStatus(Task task) {
        onAllComplete(task);
    }

    /**
     * 自己执行完成等待子任务完成 1.查询子任务成功数和失败数
     * 2.检查超时时间，如果时间达到超时时间，说明子任务的失败还未处理，移动到需要人工处理列表中
     * 3.未到超时时间则更新最后检查事件，任务成功数，失败数到数据库中
     *
     * @param task
     */
    private void handleWaitSubCompleteTaskStatus(Task task) {
        // 如果是等待子任务执行状态
        if (task.isWaitSubComplete()) {
            checkSubTaskStatus(task);
        }

    }

    // 转移错误人工处理表中
    private void doMoveToErrorList(Task task) {
        getTaskRepo().moveTaskToErrorList(task);
    }

    private void doUpdateSubtaskStatus(Task task) {
        String uptCols = "success_subtask_num,error_subtask_num,exec_count";
        getTaskRepo().updateTask(task, uptCols);
    }

    private void checkSubTaskStatus(Task task) {
        int checkCount = task.getExec_count();
        int needCheckCount = TaskParam.TASK_PUBLIC.MAX_CHECK_COUNT.getIntVal(task.getEnt_id());

        SubTaskStatusInfo subTaskInfo = getTaskRepo().querySubTaskStatus(task.getEnt_id(), task.getBillno());
        task.setSubTaskStatusInfo(subTaskInfo);
        // 全部完成了
        if (task.isAllSubCompleted()) {
            task.setTask_status(TaskConstant.TASK_STATUS.ALL_COMPLETE);
            onAllComplete(task);
        } else {
            // 存在错误或还在执行
            // 1.检查次数已经到了指定的次数
            if (checkCount >= needCheckCount) {
                onOverMaxCheckTimes(task);
            } else {
                // 更新子任务状态和检查次数
                task.setExec_count(checkCount + 1);
                doUpdateSubtaskStatus(task);
            }
        }
    }

    private void handleExecingTaskStatus(Task task) {
        // 如果有子任务，检查子任务
        if (task.hasSub()) {
            checkSubTaskStatus(task);
        } else
        // 没有子任务的，检查自己的错误次数
        {
            checkThisTaskStatus(task);
        }
    }

    private void onOvertime(Task task) {
        task.setTask_status(TaskConstant.TASK_STATUS.OVER_TIME);
        doMoveToErrorList(task);
    }

    private void onOverMaxCheckTimes(Task task) {
        task.setTask_status(TaskConstant.TASK_STATUS.MAX_EXEC_NUM);
        doMoveToErrorList(task);
    }
    // 检查本任务错误次数和状态
    // 理论上本任务执行的状态应该在它自己执行完毕后，进行处理，这里是为了防止异常，重新检查一次

    /**
     * 1.查询执行成功的log表，是否有执行成功的信息,如果有，执行完成任务 2.查询错误表，是否有错误信息，如果有，则更新
     * 重试次数信息，最后一次执行时间和最后一次错误到任务信息表中
     *
     * @param task
     */
    private void checkThisTaskStatus(Task task) {
        if (getTaskRepo().checkTaskIsExecComplete(task)) {
            task.setTask_status(TaskConstant.TASK_STATUS.ALL_COMPLETE);
            onAllComplete(task);
        } else {
            TaskErrorInfo errorInfo = getTaskRepo().queryTaskErrorInfo(task.getPh_key());
            //判断超时 判断最后执行时间是否小于当前指定时间
            long thisTime = new Date().getTime() - TaskParam.TASK_PUBLIC.TASK_TIMEOUT_HOURS.getIntVal(task.getEnt_id()) * 60 * 60 * 1000;
            if (thisTime >= errorInfo.getLastExecDate()) {
                onOvertime(task);
                return;
            }
            //判断重试次数
            if (errorInfo.getErrorCount() == 0) {
                task.setExec_count(task.getExec_count() + 1);
            } else {
                task.setExec_count(errorInfo.getErrorCount());
            }
            task.setLast_exec_date(errorInfo.getLastExecDate());
            task.setLast_exec_error(errorInfo.getLastExecError());
            String uptCols = "exec_count,last_exec_date,last_exec_error";
            int checkCount = task.getExec_count();
            int needCheckCount = TaskParam.TASK_PUBLIC.MAX_CHECK_COUNT.getIntVal(task.getEnt_id());
            if (checkCount >= needCheckCount) {
                onOverMaxCheckTimes(task);
            } else {
                getTaskRepo().updateTask(task, uptCols);
            }

        }

    }


    private void handleNotStartTaskStatus(Task task) {
        int checkCount = task.getExec_count();
        int needCheckCount = TaskParam.TASK_PUBLIC.MAX_CHECK_COUNT.getIntVal(task.getEnt_id());
        try {
            Date startDate = task.getPh_timestamp();
            Date now = new Date();
            int startTimeout = TaskParam.TASK_PUBLIC.START_TIMEOUT_MINS.getIntVal(task.getEnt_id());
            long thisMins = (now.getTime() - startDate.getTime()) / 1000 / 60;


            if (thisMins >= startTimeout) {
                // 超过时长，需要查询task内容，
                try {
                    Object data = getTaskDataQuery().getTaskData(task);
                    if (data != null) {
                        task.setData(data);
                    } else {
//                        ServiceLogs.debuglog(logtype, "未能获取[{0}]的任务数据", 0, task.getBillno());
//                        updateCheckCount(task, "未能获取任务数据");
                        TaskExceptionCode.GET_TASKDATA_ERROR.throwThisException(task.getBillno(), "未知原因");
                        return;
                    }

                } catch (ServiceException e) {
                    throw e;
                } catch (Exception e) {
//                    ServiceLogs.errLog(logtype, e, "获取[{0}]的任务数据时错误,不能自动发起重试,错误信息[{1}]", task.getBillno(), e.getMessage());
//                    updateCheckCount(task, "未能获取任务数据");
                    TaskExceptionCode.GET_TASKDATA_ERROR.throwThisException(task.getBillno(), e.getMessage());
                    return;
                }
                if (task.getData() != null) {
                    getTaskWorkerFactory().getWorker(task.getTask_group(), task.getTask_type()).receiveTask(task);
                }

            }

        } catch (ServiceException e) {
            throw e;
        } catch (Exception e) {
            ServiceLogs.errLog(logtype, e, "重启启动[{0}]任务时错误,错误信息[{1}]", task.getBillno(), e.getMessage());
            TaskExceptionCode.GET_TASKDATA_ERROR.throwThisException(task.getBillno(), e.getMessage());
        }
    }

    @Override
    public Task syncTaskStatus(Task task) {
        //如果是强制重试，则不同步状态，状态从外部传入或者手工改了数据库
        if (task.getRun_mode() == TaskConstant.RUN_MODE.FORCE_RETRY) {
            return task;
        }
        String statusVal = getValFromCached(task.getStatusKey());
        String parentStatusVal = getValFromCached(task.getParentStatusKey());
        if (statusVal != null) {
            // 如果父任务已经停止，则直接需要本任务也停止
            if (parentStatusVal != null) {
                InterTaskStatus PStatus = task.convertToTaskStatus(parentStatusVal);
                if (PStatus.getTask_status() == TaskConstant.TASK_STATUS.STOP) {
                    taskStatusChanged(task, TaskConstant.TASK_STATUS.STOP);
                    return task;
                }
            }
            int oldStatus = task.getTask_status();
            boolean ibStatuChanged = task.setStatusValue(statusVal);
            // 说明缓存比数据库的状态新，需要出发一次从缓存往数据库同步的命令
            // 有一中可能是，集群中其他机器改了缓存，但是还未同步到数据库，这里已经得到了最新的？
            if (ibStatuChanged) {
                int newStatus = task.getTask_status();
                task.setTask_status(oldStatus);
                taskStatusChanged(task, newStatus);
            }
        } else {
            Task exTask = taskRepo.findTaskById(task.getEnt_id(), task.getPh_key());
            if (exTask != null) {
                task.setStatusValue(exTask.getStatusValue());
                setValToCached(task.getStatusKey(), task.getStatusValue());
            }
        }
        return task;
    }

    // 10秒执行一次
    protected void doHandleTaskStatusChanged() throws InterruptedException {
        do {
            try {
                List<TaskStatusChangeEvent> events = getTaskStatusChangedEvents();
                changeStatus(events);
            } catch (Exception e) {
                e.printStackTrace();
            }
            Thread.sleep(10 * 1000);
        } while (!ibExit);

    }

    private void changeStatus(List<TaskStatusChangeEvent> events) {
        long time = System.currentTimeMillis();
        ServiceLogs.truedebuglog(logtype, "获取到[{0}]条状态变更", 0, events.size());
        HashMap<String, TaskStatusChangeEvent> billMap = new HashMap<String, TaskStatusChangeEvent>();
        for (TaskStatusChangeEvent event : events) {
            String key = event.getTask().getBillno();
            TaskStatusChangeEvent oldEvent = billMap.get(key);
            if (oldEvent == null || oldEvent.getTimestamp() < event.getTimestamp()) {
                billMap.put(key, event);
            }
        }
        ServiceLogs.truedebuglog(logtype, "合并后还有[{0}]条状态变更", time, billMap.size());
        int errorNum = 0;
        for (TaskStatusChangeEvent event : billMap.values()) {
            try {
                getTaskStatusManager().updateTask(event.getTask(), event.getOldStatus());
            } catch (Exception e) {
                errorNum++;
            }
        }
        ServiceLogs.truedebuglog(logtype, "处理[{0}]条状态变更,失败[{1}]条", time, billMap.size(), errorNum);
    }

    /**
     * 当所有子任务都完成时
     *
     * @param task
     */
    @Transactional
    public void onAllComplete(Task task) {
        //
        getTaskRepo().moveTaskToSuccess(task);
//        getTaskRepo().clearTaskErrorLog(task);
        // 执行回调task的回调
        TaskComponentFactory.getTaskWorkerFactory().getWorker(task.getTask_group(), task.getTask_type())
                .onComplete(task);
    }

    /**
     * 状态为变更为等待子任务完成
     *
     * @param task
     */
    @Transactional
    public void onWaitSubComplete(Task task) {
        //
        getTaskRepo().moveTaskToWaitSub(task);
//        getTaskRepo().clearTaskErrorLog(task);
    }

    @Transactional
    public void updateToDb(Task task) {
        String uptCols = "task_status";
        if (task.getExecReturn() != null && task.getExecReturn().getNeedUpdateCols() != null) {
            uptCols = uptCols + "," + task.getExecReturn().getNeedUpdateCols();
        }
        getTaskRepo().updateTask(task, uptCols);

    }

    private List<TaskStatusChangeEvent> getTaskStatusChangedEvents() {
        List<TaskStatusChangeEvent> events = new ArrayList<TaskStatusChangeEvent>(10000);
        taskStatusChangedEvents.drainTo(events, 10000);
        return events;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void updateTask(Task task, int oldStatus) {
        try {
            if (task.getTask_status() == TaskConstant.TASK_STATUS.WAIT_SUB_COMPLETE) {
                onWaitSubComplete(task);
            } else if (task.getTask_status() == TaskConstant.TASK_STATUS.ALL_COMPLETE) {
                onAllComplete(task);
            } else {
                updateToDb(task);
            }
        } catch (Exception e) {
            ServiceLogs.errLog(logtype, e, "更新任务[{0}]状态失败", task.getBillno());
            throw e;
        }

    }

    private String getValFromCached(String key) {
        Object obj = CacheUtils.getCacheUtils().getData(key);
        if (obj != null) {
            return obj.toString();
        } else {
            return null;
        }
    }

    private void setValToCached(String key, String val, int mins) {
        CacheUtils.getCacheUtils().putData(key, val, mins * CacheUtils.CacheTimeOut.Min);
    }

    private void setValToCached(String key, String val) {
        setValToCached(key, val, 24 * 60);
    }

    @Override
    public void faildown(Task task, int errStatus) {
        task.setTask_status(errStatus);
        getTaskRepo().moveTaskToErrorList(task);
    }

    @Override
    public void clearCachedStatus(Task task) {
        CacheUtils.getCacheUtils().deleteData(task.getStatusKey());
    }

    @Override
    public void stop(Task task) {
        //update 任务，子任务表状态
        getTaskRepo().updateSubTaskStatus(task.getBillno(), TaskConstant.TASK_STATUS.STOP);
        //查询本任务和不是最末级的子任务，设置缓存状态停止
        List<Task> listSubTask = getTaskRepo().querySubTaskNotLast(task.getBillno());
        for (Task subTask : listSubTask) {
            subTask.setTask_status(TaskConstant.TASK_STATUS.STOP);
            // 设置缓存
            setValToCached(subTask.getStatusKey(), subTask.getStatusValue());
        }
        //把所有未完成的任务放入到已完成的表中 等待后面定时任务或下一次任务执行时自动执行
    }

}
