package com.efuture.taskflow.job;

import com.efuture.ocp.common.distributedLock.exception.DistributedLockException;
import com.efuture.ocp.common.rest.ServiceLogs;
import com.efuture.ocp.common.util.CacheUtils;
import com.efuture.ocp.common.util.DateUtils;
import com.efuture.taskflow.TaskComponentFactory;
import com.efuture.taskflow.entity.Task;
import com.efuture.taskflow.exception.TaskExceptionCode;
import com.efuture.taskflow.monitoring.UnfinishedTaskSummary;
import com.efuture.taskflow.taskmanager.TaskStatusManager;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;

/**
 * 定时检查任务
 *
 * @author zhouwd
 */
@Component("taskJob")
public class TaskJob {

    String logtype = "TaskJob";

    @Scheduled(initialDelay = 1000 * 60 * 10, fixedDelay = 1000 * 60 * 10)
    public void checkTaskStatus() {
        UnfinishedTaskSummary taskSummary = new UnfinishedTaskSummary();
        //检查等待子任务完成的任务
        ServiceLogs.setCurLogKey(logtype + "-" + (new Date()).getTime());
        try {
            checkTaskStatus(CHECKTYPE.waitSub, taskSummary);
        } catch (Exception e) {
            ServiceLogs.errLog(logtype, e, "刷新[等待任务]的状态失败,错误信息为[{0}]", e.getMessage());
        }

        //检查未完成的任务
        try {
            checkTaskStatus(CHECKTYPE.unfinish, taskSummary);
        } catch (Exception e) {
            ServiceLogs.errLog(logtype, e, "刷新[未完成任务]的状态失败,错误信息为[{0}]", e.getMessage());
        }
        taskSummary.save();
    }

    public checkRtn checkTaskStatus(long entId, String billno) {
        Task task = TaskComponentFactory.getTaskRepository().findTaskByBillno(entId, billno);
        return checkTaskStatus(TaskComponentFactory.getTaskStatusManager(), task, true);
    }

    public checkRtn checkTaskStatus(TaskStatusManager tsm, Task task, boolean ibForce) {
        if (!ibForce) {
            if (hasChecked(task)) {
                ServiceLogs.debuglog(logtype, "任务[{0}]已经在本时段刷新过,等下一时段再刷新", 0, task.getBillno());
                return new checkRtn(99, "已经在本时段刷新过,等下一时段再刷新");
            }
        }

        ServiceLogs.debuglog(logtype, "刷新任务[{0}]的状态", 0, task.getBillno());
        try {
            tsm.refreshTaskStatus(task);
            return new checkRtn(0, "刷新成功");
        } catch (DistributedLockException e) {
            ServiceLogs.errLog(logtype, e, "刷新任务[{0}]的状态失败,获取锁错误,错误信息为[{1}]", task.getBillno(), e.getMessage());
            return new checkRtn(-1, "刷新任务状态失败,获取锁错误,错误信息为:" + e.getMessage());
        } catch (Exception e) {
            ServiceLogs.errLog(logtype, e, "刷新任务[{0}]的状态失败,错误信息为[{1}]", task.getBillno(), e.getMessage());
            tsm.refreshTaskError(task, "刷新任务状态:" + e.getMessage());
            return new checkRtn(-1, "刷新任务状态失败,错误信息为:" + e.getMessage());
        }
    }

    private void checkTaskStatus(int checktype, UnfinishedTaskSummary taskSummary) {
        // UnfinishedTaskSummary taskSummary = new UnfinishedTaskSummary();
        TaskStatusManager tsm = TaskComponentFactory.getTaskStatusManager();
        int i = 0;
        long start = System.currentTimeMillis();
        ServiceLogs.debuglog(logtype, "开始检查任务状态-[{0}]", 0, checktype);
        Date lastDate = null;
        long lastPhkey = 0;
        Date endDate = DateUtils.addMinutes(new Date(), -10);
        do {
            List<Task> taskList = getNeedCheckTaskList(checktype, lastDate, endDate, lastPhkey);
            if (taskList == null || taskList.size() < 1) {
                break;
            }
            i = i + taskList.size();
            for (Task task : taskList) {
                lastDate = task.getPh_timestamp();
                lastPhkey = task.getPh_key();
                checkRtn rtn = checkTaskStatus(tsm, task, false);
                taskSummary.possessTask(task);
                if (rtn.getCode() == 99) {
                    continue;
                }
                setLastCheckTime(task);

            }

        } while (i < 100000);
        ServiceLogs.debuglog(logtype, "检查任务状态完成,刷新[{1}]类任务[{0}]条", start, i, checktype);
    }

    private String getCheckKey(Task task) {
        return "check:" + task.getLockKey();
    }

    private long getLastCheckTime(Task task) {
        String key = getCheckKey(task);
        Object obj = CacheUtils.getCacheUtils().getData(key);
        if (obj != null) {
            return (long) obj;
        } else {
            return 0;
        }
    }

    private void setLastCheckTime(Task task) {
        String key = getCheckKey(task);
        long time = System.currentTimeMillis();
        CacheUtils.getCacheUtils().putData(key, time, 10 * CacheUtils.CacheTimeOut.Min);
    }

    private boolean hasChecked(Task task) {
        //检查缓存中的信息，如果在10分钟内已经刷新过，就不再刷新
        long time = System.currentTimeMillis();
        long lastCheckTime = getLastCheckTime(task);
        return time - lastCheckTime < 10 * 60 * 1000;
    }

    private List<Task> getNeedCheckTaskList(int checktype, Date startDate, Date endDate, long lastPhkey) {
//        Date startDate = lastDate;
//        Date endDate = DateUtils.addMinutes(new Date(),-10);
        if (checktype == CHECKTYPE.unfinish) {
            return TaskComponentFactory.getTaskRepository().queryUnfinishedTaskInfo(startDate, endDate, lastPhkey);
        } else if (checktype == CHECKTYPE.waitSub) {
            return TaskComponentFactory.getTaskRepository().queryWaitSubTaskInfo(startDate, endDate, lastPhkey);
        } else {
            TaskExceptionCode.WRONG_CHECK_TYPE.throwThisException(checktype);
            return null;
        }

    }

    private interface CHECKTYPE {
        int unfinish = 0;
        int waitSub = 1;
    }

    public class checkRtn {
        int code;
        String msg;

        public checkRtn(int code, String msg) {
            this.code = code;
            this.msg = msg;
        }

        public int getCode() {
            return code;
        }

        public void setCode(int code) {
            this.code = code;
        }

        public String getMsg() {
            return msg;
        }

        public void setMsg(String msg) {
            this.msg = msg;
        }
    }

}
