package com.efuture.taskflow.taskmanager;

import com.efuture.ocp.common.exception.ServiceException;
import com.efuture.ocp.common.rest.ServiceLogs;
import com.efuture.taskflow.TaskComponentFactory;
import com.efuture.taskflow.TaskConstant;
import com.efuture.taskflow.TaskManager;
import com.efuture.taskflow.entity.Task;
import com.efuture.taskflow.exception.TaskExceptionCode;
import com.efuture.taskflow.param.TaskParam;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.util.Date;
import java.util.List;

/**
 * taskManager实现
 */
@Component("defaultTaskManager")
public class TaskManagerImpl implements TaskManager
{

    // TaskRepository taskRepo;

    // TaskQueueService tqs;

    TaskStatusManager taskStatusManager;

    private TaskStatusManager getTaskStatusManager()
    {
        if (taskStatusManager == null) {
            taskStatusManager = TaskComponentFactory.getTaskStatusManager();
        }

        return taskStatusManager;
    }

    /**
     * 错误处理原则 1.错误记录到数据库,然后往上抛出 2.最外层要判断执行次数，如果执行次数到达一定，需要把数据放入错误表中
     *
     * @throws Exception
     */
    @Override
    public void handleError(Task task, Exception e)
    {
        String msg = "taskgroup:[{0}],tasktype:[{1}],taskbillno:[{2}]";
        ServiceLogs.errLog("TaskManager", e, msg, task.getTask_group(), task.getTask_type(), task.getBillno());
        String errorGroup = ServiceException.ERR_GROUP.UNKNOWN;
        String errorCode = TaskExceptionCode.NOT_HANDLE_ERROR.getCode();
        String errorMsg = e.getMessage();

        if (e instanceof ServiceException) {
            ServiceException se = (ServiceException) e;
            errorGroup = se.getErrGroup();
            errorCode = se.getErrorCode();
        }

        task.setLast_exec_errorCode(errorGroup, errorCode);
        task.setLast_exec_date(new Date());
        task.setLast_exec_error(errorMsg);

        //有一类异常是可以忽略的，只记录错误，然后不抛出
        if (Long.valueOf(errorCode) >= 100000 || errorGroup.equalsIgnoreCase(ServiceException.ERR_GROUP.IGNORE)) {
            ServiceLogs.errLog("TaskManager", e, "忽略异常:" + msg, task.getTask_group(), task.getTask_type(),
                               task.getBillno());
            task.setTask_status(TaskConstant.TASK_STATUS.IGNORE);
            //TaskComponentFactory.getTaskRepository().logError(task, e.getMessage());
            TaskComponentFactory.getTaskRepository().moveTaskToErrorList(task);
            return;
        }

        if (StringUtils.isEmpty(errorMsg)) {
            StackTraceElement[] stackTrace = e.getStackTrace();
            StringBuilder sb = new StringBuilder();

            for (int i = 0; i < stackTrace.length; i++) {
                sb.append(stackTrace[i]);

                if (i >= 3) {
                    break;
                }
            }

            errorMsg = sb.toString();
        }

        // 检查是否超过重试次数
        if (isMaxRetry(task)) {
            ServiceLogs.errLog("TaskManager", e, "超过重试次数:" + msg, task.getTask_group(), task.getTask_type(),
                               task.getBillno());
            task.setTask_status(TaskConstant.TASK_STATUS.MAX_EXEC_NUM);
            TaskComponentFactory.getTaskRepository().moveTaskToErrorList(task);
            //保存任务数据
            TaskComponentFactory.getTaskDataRepository().save(task);
        }
        else {
            TaskComponentFactory.getTaskRepository().logError(task, errorMsg);

            if (e instanceof ServiceException) {
                throw (ServiceException) e;
            }
            else {
                throw ServiceException.newGroupServiceException(errorGroup, TaskExceptionCode.NOT_HANDLE_ERROR.getCode(),
                        TaskExceptionCode.NOT_HANDLE_ERROR.getMsgTemplate(), errorMsg);
            }
        }
    }

    private boolean isMaxRetry(Task task)
    {
        int max = TaskParam.TASK_PUBLIC.MAX_RETRY_COUNT.getIntVal(task.getEnt_id());
        return task.getExec_count() + 1 >= max;
    }

    /**
     * 1.同步状态 1.1如果有缓存，获取缓存中的状态，如果没有缓存，则查询数据库，放入缓存，然后返回状态 2.同步数据
     * 2.1如果存在数据管理器，则使用数据管理器获取任务数据，否则不处理数据，交给任务处理程序自己处理，
     * 任务数据默认是不可变数据，数据会提交到mq，不应该存在同步数据的问题
     *
     * @param task
     */
    @Override
    public void syncTaskInfo(Task task)
    {
        // 同步任务状态
        getTaskStatusManager().syncTaskStatus(task);
        // TODO 同步任务数据 暂不处理 任务数据默认是不可变数据，数据会提交到mq，不应该存在同步数据的问题
    }

    @Override
    public void saveStat(Task task, int newStatus)
    {
        // 统一管理状态
        getTaskStatusManager().taskStatusChanged(task, newStatus);
    }

    @Override
    public int sendTask(List<Task> listSubTask)
    {
        if (listSubTask == null) {
            return 0;
        }

        return TaskComponentFactory.getTaskQueueService().batchProduct(listSubTask);
    }

    @Override
    public Task save(Task task)
    {
        return TaskComponentFactory.getTaskRepository().save(task);
    }

    @Override
    public int delete (Task task)
    {
        return 0;
    }

    @Override
    public void batchSave(List<Task> taskList)
    {
        if (taskList == null) {
            return;
        }

        TaskComponentFactory.getTaskRepository().batchSave(taskList);
    }

    // TODO 启动任务管理器

    /**
     * 1.启动时订阅对应任务的消息
     */
    @Override
    public void start()
    {
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW)
    @Override
    public void onExecComplete(Task task)
    {
        if (task.getTask_status() != TaskConstant.TASK_STATUS.ALREADY_EXEC) {
            TaskComponentFactory.getTaskRepository().insExecCompleteLog(task);
        }
    }

    @Override
    public boolean taskIsExecCompleted(Task task)
    {
        if (task.getRun_mode() == TaskConstant.RUN_MODE.FORCE_RETRY) {
            return false;
        }

        return TaskComponentFactory.getTaskRepository().checkTaskIsExecComplete(task);
    }

    @Override
    public void sendTask(Task task)
    {
        TaskComponentFactory.getTaskQueueService().product(task);
    }
}
