package com.efuture.taskflow.work;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import com.efuture.ocp.common.exception.ServiceException;
import com.efuture.ocp.common.rest.ServiceLogs;
import com.efuture.taskflow.TaskComponentFactory;
import com.efuture.taskflow.TaskExecuter;
import com.efuture.taskflow.TaskManager;
import com.efuture.taskflow.TaskWorker;

@Component("defaultTaskWorkerFactory")
public class TaskWorkerFactory implements ApplicationContextAware, InitializingBean {
    ConcurrentHashMap<String, TaskWorker> workerRegistry = new ConcurrentHashMap<String, TaskWorker>();

    public TaskWorker getWorker(String taskGroup, String taskType) {
        String key = taskGroup + "-" + taskType;
        return workerRegistry.get(key);
    }

    public void registerTaskExecuter(TaskExecuter executer) {
        String key = executer.getTaskGroup() + "-" + executer.getTaskType();
        if (workerRegistry.contains(key)) {
            throw new ServiceException("90001", "[{0}]此任务处理对象已经注册，不能重复注册", key);
        }
        TaskWorker worker = getTaskWorker(executer);
        workerRegistry.put(key, worker);
    }

    private TaskWorker getTaskWorker(TaskExecuter executer) {
        StandardTaskWorker worker = new StandardTaskWorker(executer, getTaskManager());
        worker.start();
        return worker;
    }

    private TaskManager getTaskManager() {
        return TaskComponentFactory.getTaskManager();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        ServiceLogs.debuglog("TaskWorkerFactory", "开始注册执行器", 0);
        Map<String, TaskExecuter> taskExecuters = this.applicationContext.getBeansOfType(TaskExecuter.class);
        if (Objects.nonNull(taskExecuters)) {
            for (TaskExecuter taskExec : taskExecuters.values()) {
                ServiceLogs.debuglog("TaskWorkerFactory", "注册执行器[{0}]-[{1}]", 0, taskExec.getTaskGroup(), taskExec.getTaskType());
                registerTaskExecuter(taskExec);
            }
        }
        ServiceLogs.debuglog("TaskWorkerFactory", "注册执行器完毕,注册[{0}]个", 0, taskExecuters.size());
    }

    ApplicationContext applicationContext;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;

    }

}
