package com.efuture.ocp.taskcore.message;

import java.util.Date;
import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.efuture.ocp.common.util.DateUtils;
import com.efuture.ocp.common.util.SpringBeanFactory;
import com.efuture.ocp.common.util.UniqueID;
import com.efuture.ocp.taskcore.consumer.ConsumerNode;
import com.efuture.omd.storage.FStorageOperations;

/**
 * 数据库消息队列实现--用于基本消息队列场景
 * 
 * @author zhouwudong
 *
 */
public class DBMessageHandle implements IMessageHandle {
	
	private static Logger logger = Logger.getLogger(DBMessageHandle.class);
	public final static String StorageOperation = "StorageOperation_task";
	public final static String message_tablename = "task_message";
	public final static String message_success_tablename = "task_message_success";
	public final static String message_error_tablename = "task_message_error";
	
	public final static int[] delaydate = new int[]{2,4,8,16,32,64,128,256,600,1200};
	
	public FStorageOperations getStorageOperations() {
		return SpringBeanFactory.getBean(StorageOperation, FStorageOperations.class);
	}

	/**
	 * 保存消息到数据库 1.生成message的key
	 * 改成独立事务,20171118
	 */
	@Transactional(propagation = Propagation.REQUIRES_NEW,value = "tasktransactionManager")
	public int produce(Message msg) 
	{
		msg.setId(UniqueID.getUniqueID());
		msg.setCreatedate(new Date());
		msg.setStatus(MessageStatus.NEW.name());
		msg.setKeyvalue(msg.getTopic() + "-" + msg.getKeyvalue());
		getStorageOperations().insert(msg, message_tablename);
		return IMessageHandle.msg_produce_rtncode.SUCCESS;
	}

	/**
	 * 根据消费节点配置,获取要消费的数据 1.先按配置的行数,UPDATE消息表状态，节点号 NEW->ING 2.获取节点数据状态为ING的数据
	 * 3.返回数据
	 * 2018-5-17 修改：
	 * 改为先查询，如果没有数据再update 再查询，避免每次都update
	 */
	public List<Message> pullNewMessage(ConsumerNode node) {
		Date now = new Date();
		Criteria criteria_get = Criteria.where("topic").is(node.getMessageConfig().getTopic()).and("status")
				.is(MessageStatus.ING.name()).and("execnode").is(node.getNodekey());
		Query query_get = new Query(criteria_get);
		query_get.limit(node.getMessageConfig().getPagesize());
		List<Message> rtn = getStorageOperations().select(query_get, Message.class, message_tablename);
		if(rtn.size() > 0){
			return rtn;
		}
		else
		{
			Criteria criteria = Criteria.where("topic").is(node.getMessageConfig().getTopic()).and("status")
					.is(MessageStatus.NEW.name()).and("delaydate").lte(now);
			Query query = new Query(criteria);
			Update update = new Update();
			// status execdate execnode exectransno
			update.set("status", MessageStatus.ING.name());
			update.set("execdate", now);
			update.set("execnode", node.getNodekey());
			query.limit(100);
			getStorageOperations().update(query, update, message_tablename);
		    rtn = getStorageOperations().select(query_get, Message.class, message_tablename);
		}
		return rtn;

	}

	/**
	 * 数据库消息队列处理中PULL数据时，已经处理了状态，所以这里不需要锁定
	 */
	public int lockMessage(ConsumerNode node, Message msg) {
		return 1;
	}

	/**
	 * 消息处理完的回调 1.根据msg中的处理状态，处理消息表数据 2.成功 保存到SUCCESS表中 3.失败 保存到 ERROR表中
	 * 4.删除此消息记录
	 */
	@Transactional(propagation = Propagation.REQUIRES_NEW ,value = "tasktransactionManager")
	public void consumeCallbackMessage(ConsumerNode node, Message msg) {
		// 插入表
		if (msg.getStatus().equalsIgnoreCase(MessageStatus.SUCCESS.name())
				|| msg.getStatus().equalsIgnoreCase(MessageStatus.DUP.name())) {
			getStorageOperations().insert(msg, message_success_tablename);
			// 删除message
			Criteria criteria = Criteria.where("id").is(msg.getId());
			Query query = new Query(criteria);
			getStorageOperations().delete(query, message_tablename);
		} else {
			if(msg.getRetryCount() < node.getMessageConfig().getMaxdelaytime()){
				Criteria criteria = Criteria.where("id").is(msg.getId());
				Query query = new Query(criteria);
				msg.setRetryCount(msg.getRetryCount() + 1);
				Update upt = new Update();
				upt.set("status", MessageStatus.NEW.name());
				upt.set("retryCount", msg.getRetryCount());
				upt.set("msg", msg.getMsg());
				int delaysec = 10;
				if(msg.getRetryCount() >= 10){
					delaysec = delaydate[9];
				}
				else
				{
					delaysec = delaydate[msg.getRetryCount() - 1];
				}
				Date delay = DateUtils.addSeconds(new Date(), delaysec);
				upt.set("delaydate", delay);
				getStorageOperations().update(query, upt, message_tablename);
			}
			else
			{
				getStorageOperations().insert(msg, message_error_tablename);
				// 删除message
				Criteria criteria = Criteria.where("id").is(msg.getId());
				Query query = new Query(criteria);
				getStorageOperations().delete(query, message_tablename);
			}
			
		}
		
	}

}
