/*
 * Copyright 2016 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at http://www.apache.org/licenses/LICENSE-2.0 Unless required by
 * applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS
 * OF ANY KIND, either express or implied. See the License for the specific
 * language governing permissions and limitations under the License.
 */
package com.github.drinkjava2.jdialects;

import java.util.HashMap;
import java.util.Map;

/**
 * Function templates, this file not include some not common functions (<15% percent dialects support)
 * 
 * @author Yong Zhu
 * @since 1.0.1
 */
@SuppressWarnings("all")
public class DialectFunctionTemplate {

    /**
     * Initialize extra function templates
     */
    protected static void initExtraFunctionTemplates() {
        for (Dialect d : Dialect.dialects) //register quote function for all dialects, qt是一个引用符函数
            DialectFunctionTemplate.registerFunction("qt", "1=" + d.ddlFeatures.getOpenQuote() + "$P1" + d.ddlFeatures.getCloseQuote(), d);
        registerFunction("regexlike", "2=$P1 regexp $P2", "MySQL", "H2"); //register regexlike function 给个别方言添加正则查询函数
        registerFunction("regexlike", "2=regex_like($P1, $P2)", "Oracle");
    }

    /**
     * @param databaseFamily
     * @param function
     * @param template
     */
    public static void registerFunction(String function, String template, String... databaseFamily) {
        DialectFunctionTranslator.instance.getFunctionMap().put(function.toUpperCase(), 0);
        for (Dialect dialect : Dialect.dialects) {
            for (String family : databaseFamily) {
                if (dialect.isFamily(family)) {
                    dialect.functions.put(function, template);
                }
            }
        }
    }

    public static void registerFunction(String function, String template, Dialect... dialects) {
        DialectFunctionTranslator.instance.getFunctionMap().put(function.toUpperCase(), 0);
        for (Dialect dialect : Dialect.dialects) {
            for (Dialect d : dialects) {
                if (dialect == d)
                    dialect.functions.put(function, template);
            }
        }
    }

    protected static void initFunctionTemplates() {

        //================SQLiteDialect family===============
        Map<String, String> m = Dialect.SQLiteDialect.functions;
        m.put("abs", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("coalesce", "*");
        m.put("concat", "$Lined_Params");
        m.put("count", "*");
        m.put("day", "1=extract(day from $P1)");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "1=$P1|2=$P1 % $P2");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("quote", "*");
        m.put("random", "0=random()");
        m.put("round", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substr", "*");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("trim", "1=trim($P1)|2=trim($P2, $P1)");
        m.put("upper", "*");
        m.put("year", "1=extract(year from $P1)");

        //================AccessDialect family===============
        m = Dialect.AccessDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bin", "*");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("chr", "char($Params)");
        m.put("coalesce", "*");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("crc32", "*");
        m.put("curdate", "0=curdate()");
        m.put("curtime", "0=curtime()");
        m.put("date", "*");
        m.put("datediff", "*");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("encrypt", "*");
        m.put("exp", "*");
        m.put("extract", "*");
        m.put("floor", "*");
        m.put("from_days", "*");
        m.put("hex", "*");
        m.put("hour", "*");
        m.put("initcap", "*");
        m.put("instr", "*");
        m.put("isnull", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("left", "*");
        m.put("len", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "*");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("log2", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("md5", "*");
        m.put("microsecond", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("oct", "*");
        m.put("octet_length", "*");
        m.put("pi", "0=pi()");
        m.put("power", "*");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("replace", "*");
        m.put("reverse", "*");
        m.put("right", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substr", "stuff($Params)");
        m.put("substring", "*");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate()");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timediff", "*");
        m.put("timestamp", "*");
        m.put("to_days", "*");
        m.put("translate", "*");
        m.put("trim", "*");
        m.put("trunc", "*");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("week", "*");
        m.put("weekofyear", "*");
        m.put("year", "*");

        //================ExcelDialect family===============
        m = Dialect.ExcelDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bin", "*");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("chr", "char($Params)");
        m.put("coalesce", "*");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("crc32", "*");
        m.put("curdate", "0=curdate()");
        m.put("curtime", "0=curtime()");
        m.put("date", "*");
        m.put("datediff", "*");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("encrypt", "*");
        m.put("exp", "*");
        m.put("extract", "*");
        m.put("floor", "*");
        m.put("from_days", "*");
        m.put("hex", "*");
        m.put("hour", "*");
        m.put("initcap", "*");
        m.put("instr", "*");
        m.put("isnull", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("left", "*");
        m.put("len", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "*");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("log2", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("md5", "*");
        m.put("microsecond", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("oct", "*");
        m.put("octet_length", "*");
        m.put("pi", "0=pi()");
        m.put("power", "*");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("replace", "*");
        m.put("reverse", "*");
        m.put("right", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substr", "stuff($Params)");
        m.put("substring", "*");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate()");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timediff", "*");
        m.put("timestamp", "*");
        m.put("to_days", "*");
        m.put("translate", "*");
        m.put("trim", "*");
        m.put("trunc", "*");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("week", "*");
        m.put("weekofyear", "*");
        m.put("year", "*");

        //================TextDialect family===============
        m = Dialect.TextDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bin", "*");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("chr", "char($Params)");
        m.put("coalesce", "*");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("crc32", "*");
        m.put("curdate", "0=curdate()");
        m.put("curtime", "0=curtime()");
        m.put("date", "*");
        m.put("datediff", "*");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("encrypt", "*");
        m.put("exp", "*");
        m.put("extract", "*");
        m.put("floor", "*");
        m.put("from_days", "*");
        m.put("hex", "*");
        m.put("hour", "*");
        m.put("initcap", "*");
        m.put("instr", "*");
        m.put("isnull", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("left", "*");
        m.put("len", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "*");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("log2", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("md5", "*");
        m.put("microsecond", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("oct", "*");
        m.put("octet_length", "*");
        m.put("pi", "0=pi()");
        m.put("power", "*");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("replace", "*");
        m.put("reverse", "*");
        m.put("right", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substr", "stuff($Params)");
        m.put("substring", "*");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate()");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timediff", "*");
        m.put("timestamp", "*");
        m.put("to_days", "*");
        m.put("translate", "*");
        m.put("trim", "*");
        m.put("trunc", "*");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("week", "*");
        m.put("weekofyear", "*");
        m.put("year", "*");

        //================ParadoxDialect family===============
        m = Dialect.ParadoxDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bin", "*");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("chr", "char($Params)");
        m.put("coalesce", "*");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("crc32", "*");
        m.put("curdate", "0=curdate()");
        m.put("curtime", "0=curtime()");
        m.put("date", "*");
        m.put("datediff", "*");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("encrypt", "*");
        m.put("exp", "*");
        m.put("extract", "*");
        m.put("floor", "*");
        m.put("from_days", "*");
        m.put("hex", "*");
        m.put("hour", "*");
        m.put("initcap", "*");
        m.put("instr", "*");
        m.put("isnull", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("left", "*");
        m.put("len", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "*");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("log2", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("md5", "*");
        m.put("microsecond", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("oct", "*");
        m.put("octet_length", "*");
        m.put("pi", "0=pi()");
        m.put("power", "*");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("replace", "*");
        m.put("reverse", "*");
        m.put("right", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substr", "stuff($Params)");
        m.put("substring", "*");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate()");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timediff", "*");
        m.put("timestamp", "*");
        m.put("to_days", "*");
        m.put("translate", "*");
        m.put("trim", "*");
        m.put("trunc", "*");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("week", "*");
        m.put("weekofyear", "*");
        m.put("year", "*");

        //================CobolDialect family===============
        m = Dialect.CobolDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bin", "*");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("chr", "char($Params)");
        m.put("coalesce", "*");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("crc32", "*");
        m.put("curdate", "0=curdate()");
        m.put("curtime", "0=curtime()");
        m.put("date", "*");
        m.put("datediff", "*");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("encrypt", "*");
        m.put("exp", "*");
        m.put("extract", "*");
        m.put("floor", "*");
        m.put("from_days", "*");
        m.put("hex", "*");
        m.put("hour", "*");
        m.put("initcap", "*");
        m.put("instr", "*");
        m.put("isnull", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("left", "*");
        m.put("len", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "*");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("log2", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("md5", "*");
        m.put("microsecond", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("oct", "*");
        m.put("octet_length", "*");
        m.put("pi", "0=pi()");
        m.put("power", "*");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("replace", "*");
        m.put("reverse", "*");
        m.put("right", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substr", "stuff($Params)");
        m.put("substring", "*");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate()");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timediff", "*");
        m.put("timestamp", "*");
        m.put("to_days", "*");
        m.put("translate", "*");
        m.put("trim", "*");
        m.put("trunc", "*");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("week", "*");
        m.put("weekofyear", "*");
        m.put("year", "*");

        //================XMLDialect family===============
        m = Dialect.XMLDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bin", "*");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("chr", "char($Params)");
        m.put("coalesce", "*");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("crc32", "*");
        m.put("curdate", "0=curdate()");
        m.put("curtime", "0=curtime()");
        m.put("date", "*");
        m.put("datediff", "*");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("encrypt", "*");
        m.put("exp", "*");
        m.put("extract", "*");
        m.put("floor", "*");
        m.put("from_days", "*");
        m.put("hex", "*");
        m.put("hour", "*");
        m.put("initcap", "*");
        m.put("instr", "*");
        m.put("isnull", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("left", "*");
        m.put("len", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "*");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("log2", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("md5", "*");
        m.put("microsecond", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("oct", "*");
        m.put("octet_length", "*");
        m.put("pi", "0=pi()");
        m.put("power", "*");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("replace", "*");
        m.put("reverse", "*");
        m.put("right", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substr", "stuff($Params)");
        m.put("substring", "*");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate()");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timediff", "*");
        m.put("timestamp", "*");
        m.put("to_days", "*");
        m.put("translate", "*");
        m.put("trim", "*");
        m.put("trunc", "*");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("week", "*");
        m.put("weekofyear", "*");
        m.put("year", "*");

        //================DbfDialect family===============
        m = Dialect.DbfDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bin", "*");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("chr", "char($Params)");
        m.put("coalesce", "*");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("crc32", "*");
        m.put("curdate", "0=curdate()");
        m.put("curtime", "0=curtime()");
        m.put("date", "*");
        m.put("datediff", "*");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("encrypt", "*");
        m.put("exp", "*");
        m.put("extract", "*");
        m.put("floor", "*");
        m.put("from_days", "*");
        m.put("hex", "*");
        m.put("hour", "*");
        m.put("initcap", "*");
        m.put("instr", "*");
        m.put("isnull", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("left", "*");
        m.put("len", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "*");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("log2", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("md5", "*");
        m.put("microsecond", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("oct", "*");
        m.put("octet_length", "*");
        m.put("pi", "0=pi()");
        m.put("power", "*");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("replace", "*");
        m.put("reverse", "*");
        m.put("right", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substr", "stuff($Params)");
        m.put("substring", "*");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate()");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timediff", "*");
        m.put("timestamp", "*");
        m.put("to_days", "*");
        m.put("translate", "*");
        m.put("trim", "*");
        m.put("trunc", "*");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("week", "*");
        m.put("weekofyear", "*");
        m.put("year", "*");

        //================DamengDialect family===============
        m = Dialect.DamengDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("chr", "*");
        m.put("coalesce", "*");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("curdate", "0=curdate()");
        m.put("current_date", "0=current_date()");
        m.put("current_time", "0=current_time()");
        m.put("current_timestamp", "0=current_timestamp()");
        m.put("curtime", "0=curtime()");
        m.put("datediff", "*");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hour", "*");
        m.put("initcap", "*");
        m.put("instr", "*");
        m.put("isnull", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("left", "*");
        m.put("len", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("localtimestamp", "LOCALTIMESTAMP($Params)");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "NOW($Params)");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("octet_length", "*");
        m.put("pi", "0=pi()");
        m.put("power", "*");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("replace", "*");
        m.put("reverse", "*");
        m.put("right", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("stddev", "*");
        m.put("str", "to_char($Params)");
        m.put("substr", "*");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("sysdate", "*");
        m.put("tan", "*");
        m.put("to_char", "*");
        m.put("to_date", "*");
        m.put("translate", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("trunc", "*");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("user", "*");
        m.put("variance", "*");
        m.put("week", "*");
        m.put("weekday", "*");
        m.put("year", "*");

        //================GBaseDialect family===============
        m = Dialect.GBaseDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "1=vsize($P1)*8");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("chr", "*");
        m.put("coalesce", "$NVL_Params");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("count", "*");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_timestamp");
        m.put("current_timestamp", "0=current_timestamp");
        m.put("day", "1=extract(day from $P1)");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("initcap", "*");
        m.put("instr", "*");
        m.put("last_day", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "2=instr($P2,$P1)");
        m.put("log", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("power", "*");
        m.put("replace", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("sqrt", "*");
        m.put("stddev", "*");
        m.put("str", "to_char($Params)");
        m.put("substr", "*");
        m.put("substring", "substr($Params)");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate");
        m.put("tan", "*");
        m.put("to_char", "*");
        m.put("to_date", "*");
        m.put("translate", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("trunc", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("variance", "*");
        m.put("year", "1=extract(year from $P1)");

        //================Cache71Dialect family===============
        m = Dialect.Cache71Dialect.functions;
        m.put("abs", "*");
        m.put("acos", "{fn acos($Params)}");
        m.put("ascii", "*");
        m.put("asin", "{fn asin($Params)}");
        m.put("atan", "{fn atan($Params)}");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "1=($length($P1)*8)");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceiling", "*");
        m.put("char", "{fn char($Params)}");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("coalesce", "*");
        m.put("concat", "$Lined_Params");
        m.put("cos", "{fn cos($Params)}");
        m.put("cot", "{fn cot($Params)}");
        m.put("count", "*");
        m.put("curdate", "{fn curdate($Params)}");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_time");
        m.put("current_timestamp", "*");
        m.put("curtime", "{fn curtime($Params)}");
        m.put("datediff", "*");
        m.put("day", "*");
        m.put("dayname", "{fn dayname($Params)}");
        m.put("dayofmonth", "{fn dayofmonth($Params)}");
        m.put("dayofweek", "{fn dayofweek($Params)}");
        m.put("dayofyear", "{fn dayofyear($Params)}");
        m.put("exp", "{fn exp($Params)}");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hour", "{fn hour($Params)}");
        m.put("isnull", "*");
        m.put("lcase", "{fn lcase($Params)}");
        m.put("left", "{fn left($Params)}");
        m.put("len", "*");
        m.put("length", "*");
        m.put("locate", "$FIND($Params)");
        m.put("log", "{fn log($Params)}");
        m.put("log10", "{fn log($Params)}");
        m.put("lower", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "{fn minute($Params)}");
        m.put("mod", "{fn mod($Params)}");
        m.put("month", "{fn month($Params)}");
        m.put("monthname", "{fn monthname($Params)}");
        m.put("now", "{fn monthname($Params)}");
        m.put("nullif", "*");
        m.put("nvl", "$NVL_Params");
        m.put("pi", "{fn pi($Params)}");
        m.put("power", "*");
        m.put("quarter", "{fn quarter($Params)}");
        m.put("right", "{fn right($Params)}");
        m.put("round", "*");
        m.put("rtrim", "*");
        m.put("second", "{fn second($Params)}");
        m.put("sign", "*");
        m.put("sin", "{fn sin($Params)}");
        m.put("space", "*");
        m.put("sqrt", "{fn SQRT($Params)}");
        m.put("str", "*");
        m.put("substr", "*");
        m.put("substring", "*");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate");
        m.put("tan", "{fn tan($Params)}");
        m.put("to_char", "*");
        m.put("to_date", "todate($Compact_Params)");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("ucase", "{fn ucase($Params)}");
        m.put("upper", "*");
        m.put("user", "{fn user($Params)}");
        m.put("week", "{fn user($Params)}");
        m.put("year", "{fn year($Params)}");

        //================CUBRIDDialect family===============
        m = Dialect.CUBRIDDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bin", "*");
        m.put("bit_count", "*");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("chr", "*");
        m.put("coalesce", "*");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("curdate", "0=curdate()");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_time");
        m.put("current_timestamp", "0=current_timestamp");
        m.put("curtime", "0=curtime()");
        m.put("date", "*");
        m.put("datediff", "*");
        m.put("day", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("from_days", "*");
        m.put("from_unixtime", "*");
        m.put("hex", "*");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("instr", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("localtime", "0=localtime");
        m.put("localtimestamp", "0=localtimestamp");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("log10", "*");
        m.put("log2", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("md5", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("octet_length", "*");
        m.put("pi", "0=pi()");
        m.put("power", "*");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("random", "0=random()");
        m.put("replace", "*");
        m.put("reverse", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sec_to_time", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("stddev", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substr", "*");
        m.put("substring", "*");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timediff", "*");
        m.put("timestamp", "*");
        m.put("time_to_sec", "*");
        m.put("to_char", "*");
        m.put("to_date", "*");
        m.put("to_days", "*");
        m.put("translate", "*");
        m.put("trim", "*");
        m.put("trunc", "*");
        m.put("ucase", "*");
        m.put("unix_timestamp", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("utc_date", "0=utc_date()");
        m.put("utc_time", "0=utc_time()");
        m.put("variance", "*");
        m.put("week", "*");
        m.put("weekday", "*");
        m.put("year", "*");

        //================DataDirectOracle9Dialect family===============
        m = Dialect.DataDirectOracle9Dialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "1=vsize($P1)*8");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("chr", "*");
        m.put("coalesce", "$NVL_Params");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("count", "*");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_timestamp");
        m.put("current_timestamp", "0=current_timestamp");
        m.put("day", "1=extract(day from $P1)");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("initcap", "*");
        m.put("instr", "*");
        m.put("last_day", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "2=instr($P2,$P1)");
        m.put("log", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("power", "*");
        m.put("replace", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("sqrt", "*");
        m.put("stddev", "*");
        m.put("str", "to_char($Params)");
        m.put("substr", "*");
        m.put("substring", "substr($Params)");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate");
        m.put("tan", "*");
        m.put("to_char", "*");
        m.put("to_date", "*");
        m.put("translate", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("trunc", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("variance", "*");
        m.put("year", "1=extract(year from $P1)");

        //================DB2Dialect family===============
        m = Dialect.DB2Dialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("avg", "1=avg(cast($P1 as double))");
        m.put("bit_length", "1=length($P1)*8");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("chr", "*");
        m.put("coalesce", "*");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("current_date", "0=current date");
        m.put("current_time", "0=current time");
        m.put("current_timestamp", "0=current timestamp");
        m.put("date", "*");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hex", "*");
        m.put("hour", "*");
        m.put("lcase", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("lower", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("microsecond", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("nullif", "*");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("round", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("sqrt", "*");
        m.put("stddev", "*");
        m.put("str", "1=rtrim(char($P1))");
        m.put("substr", "*");
        m.put("substring", "substr($Params)");
        m.put("sum", "*");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timestamp", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("variance", "*");
        m.put("week", "*");
        m.put("year", "*");

        m = Dialect.DB2390Dialect.functions;
        m.putAll(Dialect.DB2Dialect.functions);//extends from DB2Dialect

        m = Dialect.DB2390V8Dialect.functions;
        m.putAll(Dialect.DB2Dialect.functions);//extends from DB2Dialect

        m = Dialect.DB2400Dialect.functions;
        m.putAll(Dialect.DB2Dialect.functions);//extends from DB2Dialect

        m = Dialect.DB297Dialect.functions;
        m.putAll(Dialect.DB2Dialect.functions);//extends from DB2Dialect

        //================DerbyDialect family===============
        m = Dialect.DerbyDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("avg", "1=avg(cast($P1 as double))");
        m.put("bit_length", "1=length($P1)*8");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("chr", "*");
        m.put("coalesce", "*");
        m.put("concat", "($Lined_Params)");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("current_date", "0=current date");
        m.put("current_time", "0=current time");
        m.put("current_timestamp", "0=current timestamp");
        m.put("date", "*");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hex", "*");
        m.put("hour", "*");
        m.put("lcase", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("lower", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("microsecond", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("nullif", "*");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("round", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("sqrt", "*");
        m.put("stddev", "*");
        m.put("str", "1=rtrim(char($P1))");
        m.put("substr", "*");
        m.put("substring", "substr($Params)");
        m.put("sum", "*");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timestamp", "*");
        m.put("trim", "1=trim(both ' ' from $P1)|2=trim(both $P1 from $P2)");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("variance", "*");
        m.put("week", "*");
        m.put("year", "*");

        m = Dialect.DerbyTenFiveDialect.functions;
        m.putAll(Dialect.DerbyDialect.functions);//extends from DerbyDialect

        m = Dialect.DerbyTenSevenDialect.functions;
        m.putAll(Dialect.DerbyDialect.functions);//extends from DerbyDialect

        m = Dialect.DerbyTenSixDialect.functions;
        m.putAll(Dialect.DerbyDialect.functions);//extends from DerbyDialect

        //================FirebirdDialect family===============
        m = Dialect.FirebirdDialect.functions;
        m.put("abs", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("coalesce", "*");
        m.put("concat", "($Lined_Params)");
        m.put("count", "*");
        m.put("current_date", "0=current_date");
        m.put("day", "1=extract(day from $P1)");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("replace", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("upper", "*");
        m.put("year", "1=extract(year from $P1)");

        //================FrontBaseDialect family===============
        m = Dialect.FrontBaseDialect.functions;
        m.put("abs", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("coalesce", "*");
        m.put("count", "*");
        m.put("day", "1=extract(day from $P1)");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("upper", "*");
        m.put("year", "1=extract(year from $P1)");

        //================H2Dialect family===============
        m = Dialect.H2Dialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg(cast($P1 as double))");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("coalesce", "*");
        m.put("concat", "($Lined_Params)");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("curdate", "0=curdate()");
        m.put("current_date", "0=current_date()");
        m.put("current_time", "0=current_time()");
        m.put("current_timestamp", "0=current_timestamp()");
        m.put("curtime", "0=curtime()");
        m.put("datediff", "*");
        m.put("day", "1=extract(day from $P1)");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("encrypt", "*");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("lcase", "*");
        m.put("left", "*");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("lower", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("octet_length", "*");
        m.put("pi", "0=pi()");
        m.put("power", "*");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("replace", "*");
        m.put("right", "*");
        m.put("round", "*");
        m.put("rtrim", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("tan", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("user", "0=user()");
        m.put("week", "*");
        m.put("year", "1=extract(year from $P1)");

        //================HANAColumnStoreDialect family===============
        m = Dialect.HANAColumnStoreDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan2", "atan($Params)");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "1=length(to_binary($P1))*8");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("char", "*");
        m.put("coalesce", "*");
        m.put("concat", "($Lined_Params)");
        m.put("cos", "*");
        m.put("cot", "cos($Params)");
        m.put("count", "*");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_time");
        m.put("current_timestamp", "0=current_timestamp");
        m.put("current_user", "0=current_user");
        m.put("day", "1=extract(day from $P1)");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofyear", "*");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hour", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("left", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "2=locate($P2, $P1)|3=locate($P2, $P1, $P3)");
        m.put("log", "ln($Params)");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("power", "*");
        m.put("quarter", "*");
        m.put("replace", "*");
        m.put("right", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("session_user", "0=session_user");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "*");
        m.put("sum", "*");
        m.put("tan", "*");
        m.put("to_date", "*");
        m.put("trim", "1=trim(both ' ' from $P1)|2=trim(both $P1 from $P2)");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("week", "*");
        m.put("weekday", "*");
        m.put("year", "*");

        m = Dialect.HANARowStoreDialect.functions;
        m.putAll(Dialect.HANAColumnStoreDialect.functions);//extends from HANAColumnStoreDialect

        //================HSQLDialect family===============
        m = Dialect.HSQLDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("avg", "1=avg(cast($P1 as double))");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("coalesce", "*");
        m.put("concat", "($Lined_Params)");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("curdate", "0=curdate()");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_time");
        m.put("current_timestamp", "0=current_timestamp");
        m.put("curtime", "0=curtime()");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hour", "*");
        m.put("lcase", "*");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("lower", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("pi", "0=pi()");
        m.put("quarter", "*");
        m.put("radians", "*");
        m.put("rand", "*");
        m.put("reverse", "*");
        m.put("round", "*");
        m.put("rtrim", "*");
        m.put("second", "1=cast(second($P1) as int)");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as varchar(256))");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate");
        m.put("tan", "*");
        m.put("to_char", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("trunc", "*");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("user", "0=user()");
        m.put("week", "*");
        m.put("year", "*");

        //================InformixDialect family===============
        m = Dialect.InformixDialect.functions;
        m.put("abs", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("coalesce", "$NVL_Params");
        m.put("concat", "($Lined_Params)");
        m.put("count", "*");
        m.put("current_date", "0=today");
        m.put("current_timestamp", "0=current");
        m.put("day", "1=extract(day from $P1)");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("nvl", "$NVL_Params");
        m.put("second", "1=extract(second from $P1)");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substr", "1=substr($P1)|2=substr($P1, $P2)|3=substr($P1, $P2, $P3)");
        m.put("substring", "1=substring($P1)|2=substring($P1 FROM $P2)|3=substring($P1 FROM $P2 FOR $P3)");
        m.put("sum", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("upper", "*");
        m.put("year", "1=extract(year from $P1)");

        m = Dialect.Informix10Dialect.functions;
        m.putAll(Dialect.InformixDialect.functions);//extends from InformixDialect

        //================IngresDialect family===============
        m = Dialect.IngresDialect.functions;
        m.put("abs", "*");
        m.put("atan", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "1=octet_length(hex($P1))*4");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("character_length", "*");
        m.put("coalesce", "*");
        m.put("concat", "($Add_Params)");
        m.put("cos", "*");
        m.put("count", "*");
        m.put("current_date", "0=date('now')");
        m.put("current_time", "0=date('now')");
        m.put("current_timestamp", "0=date('now')");
        m.put("current_user", "0=current_user");
        m.put("day", "*");
        m.put("exp", "*");
        m.put("extract", "1=date_part('$P1)|3=date_part('$P1', $P3)");
        m.put("hex", "*");
        m.put("hour", "*");
        m.put("left", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)");
        m.put("log", "*");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("nullif", "*");
        m.put("octet_length", "*");
        m.put("power", "*");
        m.put("random", "0=random()");
        m.put("right", "*");
        m.put("second", "*");
        m.put("session_user", "0=session_user");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as varchar)");
        m.put("substring", "1=substring($P1)|2=substring($P1 FROM $P2)|3=substring($P1 FROM $P2 FOR $P3)");
        m.put("sum", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("unhex", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("year", "*");

        m = Dialect.Ingres10Dialect.functions;
        m.putAll(Dialect.IngresDialect.functions);//extends from IngresDialect
        m.put("concat", "($Lined_Params)");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_time");
        m.put("current_timestamp", "0=current_timestamp");

        m = Dialect.Ingres9Dialect.functions;
        m.putAll(Dialect.IngresDialect.functions);//extends from IngresDialect
        m.put("concat", "($Lined_Params)");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_time");
        m.put("current_timestamp", "0=current_timestamp");

        //================InterbaseDialect family===============
        m = Dialect.InterbaseDialect.functions;
        m.put("abs", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("coalesce", "*");
        m.put("concat", "($Lined_Params)");
        m.put("count", "*");
        m.put("current_date", "0=current_date");
        m.put("day", "1=extract(day from $P1)");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("upper", "*");
        m.put("year", "1=extract(year from $P1)");

        //================JDataStoreDialect family===============
        m = Dialect.JDataStoreDialect.functions;
        m.put("abs", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("coalesce", "*");
        m.put("count", "*");
        m.put("day", "1=extract(day from $P1)");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("upper", "*");
        m.put("year", "1=extract(year from $P1)");

        //================MariaDBDialect family===============
        m = Dialect.MariaDBDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bin", "*");
        m.put("bit_count", "*");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("ceiling", "*");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("coalesce", "*");
        m.put("concat", "*");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("crc32", "*");
        m.put("curdate", "0=curdate()");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_time");
        m.put("current_timestamp", "0=current_timestamp");
        m.put("curtime", "0=curtime()");
        m.put("date", "*");
        m.put("datediff", "*");
        m.put("date_format", "*");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("encrypt", "*");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("from_days", "*");
        m.put("from_unixtime", "*");
        m.put("hex", "*");
        m.put("hour", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("localtime", "0=localtime()");
        m.put("localtimestamp", "0=localtimestamp()");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("log2", "*");
        m.put("lower", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("md5", "*");
        m.put("microseconds", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("oct", "*");
        m.put("octet_length", "*");
        m.put("ord", "*");
        m.put("pi", "0=pi()");
        m.put("quarter", "*");
        m.put("quote", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("reverse", "*");
        m.put("round", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sec_to_time", "*");
        m.put("sha", "*");
        m.put("sha1", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("stddev", "std($Params)");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate()");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timediff", "*");
        m.put("timestamp", "*");
        m.put("time_to_sec", "*");
        m.put("to_days", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("ucase", "*");
        m.put("unhex", "*");
        m.put("unix_timestamp", "*");
        m.put("upper", "*");
        m.put("utc_date", "0=utc_date()");
        m.put("utc_time", "0=utc_time()");
        m.put("utc_timestamp", "0=utc_timestamp()");
        m.put("week", "*");
        m.put("weekday", "*");
        m.put("weekofyear", "*");
        m.put("year", "*");
        m.put("yearweek", "*");

        m = Dialect.MariaDB53Dialect.functions;
        m.putAll(Dialect.MariaDBDialect.functions);//extends from MariaDBDialect
        m.put("current_timestamp", "0=now(6)");
        m.put("localtime", "0=now(6)");
        m.put("localtimestamp", "0=now(6)");
        m.put("now", "0=now(6)");
        m.put("sysdate", "0=sysdate(6)");

        m = Dialect.MariaDB102Dialect.functions;
        m.putAll(Dialect.MariaDBDialect.functions);//extends from MariaDBDialect
        m.put("current_timestamp", "0=now(6)");
        m.put("localtime", "0=now(6)");
        m.put("localtimestamp", "0=now(6)");
        m.put("now", "0=now(6)");
        m.put("sysdate", "0=sysdate(6)");

        m = Dialect.MariaDB103Dialect.functions;
        m.putAll(Dialect.MariaDBDialect.functions);//extends from MariaDBDialect
        m.put("chr", "*");
        m.put("current_timestamp", "0=now(6)");
        m.put("localtime", "0=now(6)");
        m.put("localtimestamp", "0=now(6)");
        m.put("now", "0=now(6)");
        m.put("sysdate", "0=sysdate(6)");

        m = Dialect.MariaDB10Dialect.functions;
        m.putAll(Dialect.MariaDBDialect.functions);//extends from MariaDBDialect
        m.put("current_timestamp", "0=now(6)");
        m.put("localtime", "0=now(6)");
        m.put("localtimestamp", "0=now(6)");
        m.put("now", "0=now(6)");
        m.put("sysdate", "0=sysdate(6)");

        //================MckoiDialect family===============
        m = Dialect.MckoiDialect.functions;
        m.put("abs", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("coalesce", "*");
        m.put("concat", "*");
        m.put("count", "*");
        m.put("day", "1=extract(day from $P1)");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("round", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sign", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("upper", "*");
        m.put("user", "*");
        m.put("year", "1=extract(year from $P1)");

        //================MimerSQLDialect family===============
        m = Dialect.MimerSQLDialect.functions;
        m.put("abs", "*");
        m.put("acos", "dacos($Params)");
        m.put("asin", "dasin($Params)");
        m.put("atan", "datan($Params)");
        m.put("atan2", "datan2($Params)");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("char_length", "*");
        m.put("coalesce", "*");
        m.put("cos", "dcos($Params)");
        m.put("cot", "dcot($Params)");
        m.put("count", "*");
        m.put("date", "*");
        m.put("day", "1=extract(day from $P1)");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "ddegrees($Params)");
        m.put("exp", "dexp($Params)");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("log", "dlog($Params)");
        m.put("log10", "dlog10($Params)");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("power", "dpower($Params)");
        m.put("round", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sign", "*");
        m.put("sin", "dsin($Params)");
        m.put("soundex", "*");
        m.put("sqrt", "dsqrt($Params)");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("tan", "dtan($Params)");
        m.put("time", "*");
        m.put("timestamp", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("upper", "*");
        m.put("week", "*");
        m.put("year", "1=extract(year from $P1)");

        //================MySQLDialect family===============
        m = Dialect.MySQLDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bin", "*");
        m.put("bit_count", "*");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("ceiling", "*");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("coalesce", "*");
        m.put("concat", "*");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("crc32", "*");
        m.put("curdate", "0=curdate()");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_time");
        m.put("current_timestamp", "0=current_timestamp");
        m.put("curtime", "0=curtime()");
        m.put("date", "*");
        m.put("datediff", "*");
        m.put("date_format", "*");
        m.put("day", "*");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("encrypt", "*");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("from_days", "*");
        m.put("from_unixtime", "*");
        m.put("hex", "*");
        m.put("hour", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("localtime", "0=localtime()");
        m.put("localtimestamp", "0=localtimestamp()");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("log2", "*");
        m.put("lower", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("md5", "*");
        m.put("microseconds", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("oct", "*");
        m.put("octet_length", "*");
        m.put("ord", "*");
        m.put("pi", "0=pi()");
        m.put("quarter", "*");
        m.put("quote", "*");
        m.put("radians", "*");
        m.put("rand", "0=rand()");
        m.put("reverse", "*");
        m.put("round", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sec_to_time", "*");
        m.put("sha", "*");
        m.put("sha1", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("stddev", "std($Params)");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate()");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timediff", "*");
        m.put("timestamp", "*");
        m.put("time_to_sec", "*");
        m.put("to_days", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("ucase", "*");
        m.put("unhex", "*");
        m.put("unix_timestamp", "*");
        m.put("upper", "*");
        m.put("utc_date", "0=utc_date()");
        m.put("utc_time", "0=utc_time()");
        m.put("utc_timestamp", "0=utc_timestamp()");
        m.put("week", "*");
        m.put("weekday", "*");
        m.put("weekofyear", "*");
        m.put("year", "*");
        m.put("yearweek", "*");

        m = Dialect.MySQL5Dialect.functions;
        m.putAll(Dialect.MySQLDialect.functions);//extends from MySQLDialect

        m = Dialect.MySQL55Dialect.functions;
        m.putAll(Dialect.MySQLDialect.functions);//extends from MySQLDialect

        m = Dialect.MySQL57Dialect.functions;
        m.putAll(Dialect.MySQLDialect.functions);//extends from MySQLDialect
        m.put("current_timestamp", "0=now(6)");
        m.put("localtime", "0=now(6)");
        m.put("localtimestamp", "0=now(6)");
        m.put("now", "0=now(6)");
        m.put("sysdate", "0=sysdate(6)");

        m = Dialect.MySQL57InnoDBDialect.functions;
        m.putAll(Dialect.MySQLDialect.functions);//extends from MySQLDialect
        m.put("current_timestamp", "0=now(6)");
        m.put("localtime", "0=now(6)");
        m.put("localtimestamp", "0=now(6)");
        m.put("now", "0=now(6)");
        m.put("sysdate", "0=sysdate(6)");

        m = Dialect.MySQL5InnoDBDialect.functions;
        m.putAll(Dialect.MySQLDialect.functions);//extends from MySQLDialect

        m = Dialect.MySQLInnoDBDialect.functions;
        m.putAll(Dialect.MySQLDialect.functions);//extends from MySQLDialect

        m = Dialect.MySQLMyISAMDialect.functions;
        m.putAll(Dialect.MySQLDialect.functions);//extends from MySQLDialect

        m = Dialect.MySQL8Dialect.functions;
        m.putAll(Dialect.MySQLDialect.functions);//extends from MySQLDialect
        m.put("current_timestamp", "0=now(6)");
        m.put("localtime", "0=now(6)");
        m.put("localtimestamp", "0=now(6)");
        m.put("now", "0=now(6)");
        m.put("sysdate", "0=sysdate(6)");

        //================OracleDialect family===============
        m = Dialect.OracleDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "1=vsize($P1)*8");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("chr", "*");
        m.put("coalesce", "$NVL_Params");
        m.put("concat", "$Lined_Params");
        m.put("cos", "*");
        m.put("count", "*");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_timestamp");
        m.put("current_timestamp", "0=current_timestamp");
        m.put("day", "1=extract(day from $P1)");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("initcap", "*");
        m.put("instr", "*");
        m.put("last_day", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "2=instr($P2,$P1)");
        m.put("log", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("power", "*");
        m.put("replace", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("sqrt", "*");
        m.put("stddev", "*");
        m.put("str", "to_char($Params)");
        m.put("substr", "*");
        m.put("substring", "substr($Params)");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate");
        m.put("tan", "*");
        m.put("to_char", "*");
        m.put("to_date", "*");
        m.put("translate", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("trunc", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("variance", "*");
        m.put("year", "1=extract(year from $P1)");

        m = Dialect.Oracle10gDialect.functions;
        m.putAll(Dialect.OracleDialect.functions);//extends from OracleDialect

        m = Dialect.Oracle12cDialect.functions;
        m.putAll(Dialect.OracleDialect.functions);//extends from OracleDialect

        m = Dialect.Oracle8iDialect.functions;
        m.putAll(Dialect.OracleDialect.functions);//extends from OracleDialect

        m = Dialect.Oracle9Dialect.functions;
        m.putAll(Dialect.OracleDialect.functions);//extends from OracleDialect

        m = Dialect.Oracle9iDialect.functions;
        m.putAll(Dialect.OracleDialect.functions);//extends from OracleDialect

        //================PointbaseDialect family===============
        m = Dialect.PointbaseDialect.functions;
        m.put("abs", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("coalesce", "*");
        m.put("count", "*");
        m.put("day", "1=extract(day from $P1)");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("upper", "*");
        m.put("year", "1=extract(year from $P1)");

        //================PostgreSQLDialect family===============
        m = Dialect.PostgreSQLDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("char_length", "*");
        m.put("chr", "*");
        m.put("coalesce", "*");
        m.put("concat", "($Lined_Params)");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_time");
        m.put("current_timestamp", "0=current_timestamp");
        m.put("current_user", "0=current_user");
        m.put("day", "1=extract(day from $P1)");
        m.put("degrees", "*");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("initcap", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("localtime", "0=localtime");
        m.put("localtimestamp", "0=localtimestamp");
        m.put("locate", "2=position($P1 in $P2)|3=(position($P1 in substring($P2, $P3))+$P3-1)");
        m.put("log", "*");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("md5", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("octet_length", "*");
        m.put("radians", "*");
        m.put("rand", "0=random()");
        m.put("random", "0=random()");
        m.put("round", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("session_user", "0=session_user");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("sqrt", "*");
        m.put("stddev", "*");
        m.put("str", "1=cast($P1 as varchar)");
        m.put("substr", "*");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("tan", "*");
        m.put("to_char", "*");
        m.put("to_date", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("trunc", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("variance", "*");
        m.put("year", "1=extract(year from $P1)");

        m = Dialect.PostgresPlusDialect.functions;
        m.putAll(Dialect.PostgreSQLDialect.functions);//extends from PostgreSQLDialect
        m.put("atan2", "*");
        m.put("coalesce", "$NVL_Params");
        m.put("instr", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("nvl", "*");
        m.put("power", "*");
        m.put("replace", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("soundex", "*");
        m.put("substring", "substr($Params)");
        m.put("sysdate", "0=sysdate");
        m.put("translate", "*");

        m = Dialect.PostgreSQL81Dialect.functions;
        m.putAll(Dialect.PostgreSQLDialect.functions);//extends from PostgreSQLDialect

        m = Dialect.PostgreSQL82Dialect.functions;
        m.putAll(Dialect.PostgreSQLDialect.functions);//extends from PostgreSQLDialect

        m = Dialect.PostgreSQL9Dialect.functions;
        m.putAll(Dialect.PostgreSQLDialect.functions);//extends from PostgreSQLDialect

        m = Dialect.PostgreSQL91Dialect.functions;
        m.putAll(Dialect.PostgreSQLDialect.functions);//extends from PostgreSQLDialect

        m = Dialect.PostgreSQL92Dialect.functions;
        m.putAll(Dialect.PostgreSQLDialect.functions);//extends from PostgreSQLDialect

        m = Dialect.PostgreSQL93Dialect.functions;
        m.putAll(Dialect.PostgreSQLDialect.functions);//extends from PostgreSQLDialect

        m = Dialect.PostgreSQL94Dialect.functions;
        m.putAll(Dialect.PostgreSQLDialect.functions);//extends from PostgreSQLDialect

        m = Dialect.PostgreSQL95Dialect.functions;
        m.putAll(Dialect.PostgreSQLDialect.functions);//extends from PostgreSQLDialect

        //================ProgressDialect family===============
        m = Dialect.ProgressDialect.functions;
        m.put("abs", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("coalesce", "*");
        m.put("count", "*");
        m.put("day", "1=extract(day from $P1)");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("upper", "*");
        m.put("year", "1=extract(year from $P1)");

        //================RDMSOS2200Dialect family===============
        m = Dialect.RDMSOS2200Dialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("character_length", "*");
        m.put("char_length", "*");
        m.put("chr", "*");
        m.put("coalesce", "*");
        m.put("concat", "1=concat($P1)|2=concat($P1, $P2)");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("curdate", "0=curdate()");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_timestamp");
        m.put("current_timestamp", "0=current_timestamp");
        m.put("curtime", "0=curtime()");
        m.put("day", "1=extract(day from $P1)");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hour", "*");
        m.put("initcap", "*");
        m.put("instr", "*");
        m.put("last_day", "*");
        m.put("lcase", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("microsecond", "*");
        m.put("min", "*");
        m.put("minute", "*");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("monthname", "*");
        m.put("now", "0=now()");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("pi", "0=pi()");
        m.put("power", "*");
        m.put("quarter", "*");
        m.put("rand", "0=rand()");
        m.put("replace", "*");
        m.put("reverse", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "*");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substr", "*");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timestamp", "*");
        m.put("trim", "1=ltrim(rtrim($P1))");
        m.put("trunc", "*");
        m.put("ucase", "*");
        m.put("upper", "*");
        m.put("user", "0=user");
        m.put("week", "*");
        m.put("year", "*");

        //================SAPDBDialect family===============
        m = Dialect.SAPDBDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("atan2", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceil", "*");
        m.put("coalesce", "value($Params)");
        m.put("concat", "($Lined_Params)");
        m.put("cos", "*");
        m.put("cot", "cos($Params)");
        m.put("count", "*");
        m.put("date", "*");
        m.put("day", "1=day($P1)");
        m.put("dayname", "*");
        m.put("dayofmonth", "*");
        m.put("dayofweek", "*");
        m.put("dayofyear", "*");
        m.put("degrees", "*");
        m.put("exp", "*");
        m.put("extract", "1=$P1)|3=$P1($P3)");
        m.put("floor", "*");
        m.put("hour", "1=hour($P1)");
        m.put("initcap", "*");
        m.put("length", "*");
        m.put("ln", "*");
        m.put("locate", "index($Params)");
        m.put("log", "ln($Params)");
        m.put("lower", "*");
        m.put("lpad", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("microsecond", "*");
        m.put("min", "*");
        m.put("minute", "1=minute($P1)");
        m.put("mod", "*");
        m.put("month", "1=month($P1)");
        m.put("monthname", "*");
        m.put("nullif", "*");
        m.put("pi", "0=pi()");
        m.put("power", "*");
        m.put("radians", "*");
        m.put("replace", "*");
        m.put("round", "*");
        m.put("rpad", "*");
        m.put("rtrim", "*");
        m.put("second", "1=second($P1)");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("soundex", "*");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substr", "*");
        m.put("substring", "substr($Params)");
        m.put("sum", "*");
        m.put("tan", "*");
        m.put("time", "*");
        m.put("timestamp", "*");
        m.put("translate", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("trunc", "*");
        m.put("upper", "*");
        m.put("weekofyear", "*");
        m.put("year", "1=year($P1)");

        //================SQLServerDialect family===============
        m = Dialect.SQLServerDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "1=datalength($P1) * 8");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("coalesce", "*");
        m.put("concat", "($Add_Params)");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("current_date", "0=getdate()");
        m.put("current_time", "0=getdate()");
        m.put("current_timestamp", "0=getdate()");
        m.put("day", "*");
        m.put("degrees", "*");
        m.put("exp", "*");
        m.put("extract", "1=datepart($P1)|3=datepart($P1, $P3)");
        m.put("floor", "*");
        m.put("hour", "1=datepart(hour, $P1)");
        m.put("isnull", "*");
        m.put("len", "*");
        m.put("length", "len($Params)");
        m.put("locate", "charindex($Params)");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("lower", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=datepart(minute, $P1)");
        m.put("mod", "1=$P1|2=$P1 % $P2");
        m.put("month", "*");
        m.put("nullif", "*");
        m.put("pi", "0=pi()");
        m.put("radians", "*");
        m.put("rand", "*");
        m.put("reverse", "*");
        m.put("round", "*");
        m.put("rtrim", "*");
        m.put("second", "1=datepart(second, $P1)");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("str", "*");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("tan", "*");
        m.put("trim", "1=ltrim(rtrim($P1))|2=replace(replace(ltrim(rtrim(replace(replace($P2,' ','${space}$'),$P1,' '))),' ',$P1),'${space}$',' ')");
        m.put("upper", "*");
        m.put("user", "0=user()");
        m.put("year", "*");

        m = Dialect.SQLServer2005Dialect.functions;
        m.putAll(Dialect.SQLServerDialect.functions);//extends from SQLServerDialect

        m = Dialect.SQLServer2008Dialect.functions;
        m.putAll(Dialect.SQLServerDialect.functions);//extends from SQLServerDialect
        m.put("current_timestamp", "0=current_timestamp");

        m = Dialect.SQLServer2012Dialect.functions;
        m.putAll(Dialect.SQLServerDialect.functions);//extends from SQLServerDialect
        m.put("current_timestamp", "0=current_timestamp");

        //================SybaseDialect family===============
        m = Dialect.SybaseDialect.functions;
        m.put("abs", "*");
        m.put("acos", "*");
        m.put("ascii", "*");
        m.put("asin", "*");
        m.put("atan", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("ceiling", "*");
        m.put("char", "*");
        m.put("coalesce", "*");
        m.put("concat", "($Add_Params)");
        m.put("cos", "*");
        m.put("cot", "*");
        m.put("count", "*");
        m.put("current_date", "0=getdate()");
        m.put("current_time", "0=getdate()");
        m.put("current_timestamp", "0=getdate()");
        m.put("day", "*");
        m.put("degrees", "*");
        m.put("exp", "*");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("floor", "*");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("isnull", "*");
        m.put("len", "*");
        m.put("length", "len($Params)");
        m.put("locate", "2=charindex($P1, $P2)|3=charindex($P1, right($P2, char_length($P2)-($P3-1)))");
        m.put("log", "*");
        m.put("log10", "*");
        m.put("lower", "*");
        m.put("ltrim", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "*");
        m.put("nullif", "*");
        m.put("pi", "0=pi()");
        m.put("radians", "*");
        m.put("rand", "*");
        m.put("reverse", "*");
        m.put("round", "*");
        m.put("rtrim", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sign", "*");
        m.put("sin", "*");
        m.put("space", "*");
        m.put("sqrt", "*");
        m.put("str", "*");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("tan", "*");
        m.put("trim", "1=ltrim(rtrim($P1))");
        m.put("upper", "*");
        m.put("user", "0=user()");
        m.put("year", "*");

        m = Dialect.Sybase11Dialect.functions;
        m.putAll(Dialect.SybaseDialect.functions);//extends from SybaseDialect

        m = Dialect.SybaseAnywhereDialect.functions;
        m.putAll(Dialect.SybaseDialect.functions);//extends from SybaseDialect

        m = Dialect.SybaseASE15Dialect.functions;
        m.putAll(Dialect.SybaseDialect.functions);//extends from SybaseDialect
        m.put("atan2", "1=atn2($P1)|2=atn2($P1, $P2)");
        m.put("bit_length", "1=datalength($P1) * 8");
        m.put("char_length", "1=char_length($P1)");
        m.put("current_date", "0=current_date()");
        m.put("current_time", "0=current_time()");
        m.put("datediff", "1=datediff($P1)|2=datediff($P1, $P2)|3=datediff($P1, $P2, $P3)");
        m.put("extract", "1=datepart($P1)|3=datepart($P1, $P3)");
        m.put("hour", "1=datepart(hour, $P1)");
        m.put("minute", "1=datepart(minute, $P1)");
        m.put("mod", "1=$P1|2=$P1 % $P2");
        m.put("right", "0=right");
        m.put("second", "1=datepart(second, $P1)");
        m.put("soundex", "0=sounded");
        m.put("stddev", "0=stddev");
        m.put("substring", "*");
        m.put("sysdate", "0=getdate()");
        m.put("trim", "1=ltrim(rtrim($P1))|2=str_replace(str_replace(ltrim(rtrim(str_replace(str_replace($P2,' ','${space}$'),$P1,' '))),' ',$P1),'${space}$',' ')");
        m.put("variance", "0=variance");

        m = Dialect.SybaseASE157Dialect.functions;
        m.putAll(Dialect.SybaseDialect.functions);//extends from SybaseDialect
        m.put("atan2", "1=atn2($P1)|2=atn2($P1, $P2)");
        m.put("bit_length", "1=datalength($P1) * 8");
        m.put("char_length", "1=char_length($P1)");
        m.put("current_date", "0=current_date()");
        m.put("current_time", "0=current_time()");
        m.put("datediff", "1=datediff($P1)|2=datediff($P1, $P2)|3=datediff($P1, $P2, $P3)");
        m.put("extract", "1=datepart($P1)|3=datepart($P1, $P3)");
        m.put("hour", "1=datepart(hour, $P1)");
        m.put("minute", "1=datepart(minute, $P1)");
        m.put("mod", "1=$P1|2=$P1 % $P2");
        m.put("right", "0=right");
        m.put("second", "1=datepart(second, $P1)");
        m.put("soundex", "0=sounded");
        m.put("stddev", "0=stddev");
        m.put("substring", "*");
        m.put("sysdate", "0=getdate()");
        m.put("trim", "1=ltrim(rtrim($P1))|2=str_replace(str_replace(ltrim(rtrim(str_replace(str_replace($P2,' ','${space}$'),$P1,' '))),' ',$P1),'${space}$',' ')");
        m.put("variance", "0=variance");

        //================TeradataDialect family===============
        m = Dialect.TeradataDialect.functions;
        m.put("abs", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "1=octet_length(cast($P1 as char))*4");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("coalesce", "*");
        m.put("concat", "($Lined_Params)");
        m.put("count", "*");
        m.put("current_date", "0=current_date");
        m.put("current_time", "0=current_time");
        m.put("current_timestamp", "0=current_timestamp");
        m.put("day", "1=extract(day from $P1)");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("length", "1=character_length($P1)");
        m.put("locate", "1=position($P1)|2=position($P1 in $P2)");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "1=$P1|2=$P1 mod $P2");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as varchar(255))");
        m.put("substring", "1=substring($P1)|2=substring($P1 from $P2)|3=substring($P1 from $P2 for $P3)");
        m.put("sum", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("upper", "*");
        m.put("year", "1=extract(year from $P1)");

        m = Dialect.Teradata14Dialect.functions;
        m.putAll(Dialect.TeradataDialect.functions);//extends from TeradataDialect

        //================TimesTenDialect family===============
        m = Dialect.TimesTenDialect.functions;
        m.put("abs", "*");
        m.put("avg", "1=avg($P1)");
        m.put("bit_length", "*");
        m.put("cast", "1=cast($P1)|2=cast($P1 as $P2)");
        m.put("coalesce", "*");
        m.put("concat", "*");
        m.put("count", "*");
        m.put("day", "1=extract(day from $P1)");
        m.put("extract", "1=extract($P1)|2=extract($P1 $P2)|3=extract($P1 $P2 $P3)");
        m.put("hour", "1=extract(hour from $P1)");
        m.put("length", "*");
        m.put("locate", "1=locate($P1)|2=locate($P1, $P2)|3=locate($P1, $P2, $P3)");
        m.put("lower", "*");
        m.put("max", "*");
        m.put("min", "*");
        m.put("minute", "1=extract(minute from $P1)");
        m.put("mod", "*");
        m.put("month", "1=extract(month from $P1)");
        m.put("nullif", "*");
        m.put("nvl", "*");
        m.put("rtrim", "*");
        m.put("second", "1=extract(second from $P1)");
        m.put("sqrt", "*");
        m.put("str", "1=cast($P1 as char)");
        m.put("substring", "1=substring($P1)|2=substring($P1, $P2)|3=substring($P1, $P2, $P3)");
        m.put("sum", "*");
        m.put("sysdate", "0=sysdate");
        m.put("to_char", "*");
        m.put("to_date", "*");
        m.put("trim", "1=trim($P1)|2=trim($P1 $P2)|3=trim($P1 $P2 $P3)|4=trim($P1 $P2 $P3 $P4)");
        m.put("upper", "*");
        m.put("year", "1=extract(year from $P1)");
    }

}
