package com.github.houbb.paradise.common.util;


import java.io.*;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

/**
 * 文件工具类
 *
 * @author bbhou
 * @version 1.1.0
 */
public final class FileUtil {

    private FileUtil() {
    }

    /**
     * 获取文件内容
     *
     * @param filePath 文件路径
     * @return 文件不存在或异常等, 返回空字符串
     */
    public static String getFileContent(String filePath) {
        File file = new File(filePath);
        if (file.exists()) {
            try {
                InputStream inputStream = new FileInputStream(file);
                return getFileContent(inputStream);
            } catch (FileNotFoundException e) {
                e.printStackTrace();
                System.err.println("文件不存在" + filePath);
                return "";
            }
        }
        return "";
    }

    /**
     * 获取文件内容
     * 默认编码UTF8
     *
     * @param inputStream 输入流
     * @return 文件内容
     */
    public static String getFileContent(InputStream inputStream) {
        return getFileContent(inputStream, "UTF-8");
    }

    /**
     * 获取文件内容
     *
     * @param inputStream 文件输入流
     * @param charset     文件编码
     * @return 文件内容字符串
     */
    public static String getFileContent(InputStream inputStream, String charset) {
        try {
            int size = inputStream.available();
            byte[] bytes = new byte[size];
            int readSize = inputStream.read(bytes);
            inputStream.close();
            String jsonText;
            jsonText = new String(bytes, charset);
            return jsonText;
        } catch (IOException e) {
            e.printStackTrace();
        }

        return "";
    }

    /**
     * 获取文件后缀
     *
     * @param fileName 文件名称
     * @return 文件后缀
     */
    public static String getSuffix(String fileName) {
        return fileName.substring(fileName.lastIndexOf(".") + 1);
    }


    /**
     * 获取指定路径文件的每一行内容
     *
     * @param filePath 文件路径
     * @param initLine 初始化行数
     * @return 内容列表
     */
    public static List<String> getFileContentEachLine(String filePath, int initLine) {
        File file = new File(filePath);
        return getFileContentEachLine(file, initLine);
    }

    /**
     * 获取指定路径文件的每一行内容
     * 1.初始化行数默认为0
     *
     * @param filePath 文件路径
     * @return 内容列表
     * @see #getFileContentEachLine(String, int) 获取指定路径文件的每一行内容
     */
    public static List<String> getFileContentEachLine(String filePath) {
        File file = new File(filePath);
        return getFileContentEachLine(file, 0);
    }


    /**
     * 获取指定文件的每一行内容。并对内容进行trim()操作。
     *
     * @param filePath 文件路径
     * @param initLine 初始化行数
     * @return 内容列表
     */
    public static List<String> getFileContentEachLineTrim(String filePath, int initLine) {
        List<String> stringList = getFileContentEachLine(filePath, initLine);
        List<String> resultList = new LinkedList<>();

        for (String string : stringList) {
            resultList.add(string.trim());
        }

        return resultList;
    }

    /**
     * 获取指定文件的每一行内容
     * 默认初始行数为0
     *
     * @param file 文件
     * @return 内容列表
     */
    public static List<String> getFileContentEachLine(File file) {
        return getFileContentEachLine(file, 0);
    }

    /**
     * 获取指定文件的每一行内容
     * [TWR](http://blog.csdn.net/doctor_who2004/article/details/50901195)
     *
     * @param file     指定读取文件
     * @param initLine 初始读取行数
     * @return 错误返回空列表
     * @since 1.7
     */
    public static List<String> getFileContentEachLine(File file, int initLine) {
        List<String> contentList = new LinkedList<>();

        if (!file.exists()) {
            System.err.println("文件不存在");
            return contentList;
        }

        String charset = "UTF-8";  //暂时使用此编码
        try (FileInputStream fileInputStream = new FileInputStream(file);
             InputStreamReader inputStreamReader = new InputStreamReader(fileInputStream, charset);
             BufferedReader bufferedReader = new BufferedReader(inputStreamReader)
        ) {
            int lineNo = 0;// 用于记录行号
            while (lineNo < initLine) {
                lineNo++;
                bufferedReader.readLine();
            }

            String dataEachLine;   //每一行的内容
            while ((dataEachLine = bufferedReader.readLine()) != null) {
                lineNo++;
                //跳过空白行
                if (Objects.equals("", dataEachLine)) {
                    continue;
                }
                contentList.add(dataEachLine);
            }
        } catch (IOException e) {
            e.printStackTrace();
            return new LinkedList<>();
        }

        return contentList;
    }


    /**
     * 获取文件内容的列表
     *
     * @param file 文件
     * @param initLine 0 开始
     * @param endLine  下标从0开始
     * @param charset 编码
     * @return string list
     */
    public static List<String> getFileContentEachLine(final File file, final int initLine, final int endLine, final String charset) {
        List<String> contentList = new LinkedList<>();

        if (!file.exists()) {
            System.err.println("文件不存在");
            return contentList;
        }

        try (FileInputStream fileInputStream = new FileInputStream(file);
             InputStreamReader inputStreamReader = new InputStreamReader(fileInputStream, charset);
             BufferedReader bufferedReader = new BufferedReader(inputStreamReader)
        ) {
            int lineNo = 0;// 用于记录行号
            while (lineNo < initLine) {
                lineNo++;
                bufferedReader.readLine();
            }

            String dataEachLine;   //每一行的内容
            while ((dataEachLine = bufferedReader.readLine()) != null
                    && lineNo < endLine) {
                lineNo++;
                contentList.add(dataEachLine);
            }
        } catch (IOException e) {
            e.printStackTrace();
            return new LinkedList<>();
        }

        return contentList;
    }


    /**
     * 复制文件夹
     * @param sourceDir 原始文件夹
     * @param targetDir 目标文件夹
     * @throws IOException
     * @since 1.1.2
     */
    public static void copyDir(String sourceDir, String targetDir) throws IOException {
        File file = new File(sourceDir);
        String[] filePath = file.list();

        if (!(new File(targetDir)).exists()) {
            (new File(targetDir)).mkdir();
        }

        if (ArrayUtil.isNotEmpty(filePath)) {

            for (int i = 0; i < filePath.length; i++) {
                if ((new File(sourceDir + File.separator + filePath[i])).isDirectory()) {
                    copyDir(sourceDir + File.separator + filePath[i], targetDir + File.separator + filePath[i]);
                }

                if (new File(sourceDir + File.separator + filePath[i]).isFile()) {
                    copyFile(sourceDir + File.separator + filePath[i], targetDir + File.separator + filePath[i]);
                }

            }

        }

    }


    /**
     * 复制文件
     * @param sourceFile 原始路径
     * @param targetPath 目标路径
     * @throws IOException
     * @since 1.1.2
     */
    public static void copyFile(String sourceFile, String targetPath) throws IOException {
        File oldFile = new File(sourceFile);
        File file = new File(targetPath);

        try (FileInputStream in = new FileInputStream(oldFile);
             FileOutputStream out = new FileOutputStream(file)) {

            byte[] buffer = new byte[2097152];

            while ((in.read(buffer)) != -1) {
                out.write(buffer);
            }

        }

    }


}
