/*
 * Decompiled with CFR 0.152.
 */
package oshi.software.os.linux;

import com.sun.jna.Memory;
import com.sun.jna.Native;
import com.sun.jna.Pointer;
import com.sun.jna.platform.linux.LibC;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import oshi.jna.platform.linux.Libc;
import oshi.software.common.AbstractOperatingSystem;
import oshi.software.os.FileSystem;
import oshi.software.os.NetworkParams;
import oshi.software.os.OSProcess;
import oshi.software.os.OSUser;
import oshi.software.os.OperatingSystem;
import oshi.software.os.linux.LinuxFileSystem;
import oshi.software.os.linux.LinuxNetworkParams;
import oshi.software.os.linux.LinuxOSVersionInfoEx;
import oshi.software.os.linux.LinuxUserGroupInfo;
import oshi.util.ExecutingCommand;
import oshi.util.FileUtil;
import oshi.util.ParseUtil;
import oshi.util.platform.linux.ProcUtil;

public class LinuxOperatingSystem
extends AbstractOperatingSystem {
    private static final long serialVersionUID = 1L;
    private static final Logger LOG = LoggerFactory.getLogger(LinuxOperatingSystem.class);
    private static final long BOOTTIME;
    protected String versionId;
    protected String codeName;
    private final int memoryPageSize;
    private static final int[] PROC_PID_STAT_ORDERS;
    private static final int PROC_PID_STAT_LENGTH;
    private static final long USER_HZ;
    private static final long BOOT_TIME;

    public LinuxOperatingSystem() {
        this.manufacturer = "GNU/Linux";
        this.setFamilyFromReleaseFiles();
        this.version = new LinuxOSVersionInfoEx(this.versionId, this.codeName);
        this.memoryPageSize = ParseUtil.parseIntOrDefault(ExecutingCommand.getFirstAnswer("getconf PAGESIZE"), 4096);
        this.initBitness();
    }

    private void initBitness() {
        if (this.bitness < 64 && ExecutingCommand.getFirstAnswer("uname -m").indexOf("64") != -1) {
            this.bitness = 64;
        }
    }

    @Override
    public FileSystem getFileSystem() {
        return new LinuxFileSystem();
    }

    @Override
    public OSProcess[] getProcesses(int limit, OperatingSystem.ProcessSort sort, boolean slowFields) {
        ArrayList<OSProcess> procs = new ArrayList<OSProcess>();
        File[] pids = ProcUtil.getPidFiles();
        LinuxUserGroupInfo userGroupInfo = new LinuxUserGroupInfo();
        for (File pidFile : pids) {
            int pid = ParseUtil.parseIntOrDefault(pidFile.getName(), 0);
            OSProcess proc = this.getProcess(pid, userGroupInfo, slowFields);
            if (proc == null) continue;
            procs.add(proc);
        }
        List<OSProcess> sorted = this.processSort(procs, limit, sort);
        return sorted.toArray(new OSProcess[0]);
    }

    @Override
    public OSProcess getProcess(int pid) {
        return this.getProcess(pid, new LinuxUserGroupInfo(), true);
    }

    private OSProcess getProcess(int pid, LinuxUserGroupInfo userGroupInfo, boolean slowFields) {
        String path = "";
        Memory buf = new Memory(1024L);
        int size = Libc.INSTANCE.readlink(String.format("/proc/%d/exe", pid), (Pointer)buf, 1023);
        if (size > 0) {
            String tmp;
            path = tmp.substring(0, (tmp = buf.getString(0L)).length() < size ? tmp.length() : size);
        }
        Map<String, String> io = FileUtil.getKeyValueMapFromFile(String.format("/proc/%d/io", pid), ":");
        long now = System.currentTimeMillis();
        String stat = FileUtil.getStringFromFile(String.format("/proc/%d/stat", pid));
        if (stat.isEmpty()) {
            return null;
        }
        long[] statArray = ParseUtil.parseStringToLongArray(stat, PROC_PID_STAT_ORDERS, PROC_PID_STAT_LENGTH, ' ');
        OSProcess proc = new OSProcess();
        proc.setProcessID(pid);
        proc.setCommandLine(FileUtil.getStringFromFile(String.format("/proc/%d/cmdline", pid)));
        long startTime = BOOT_TIME + statArray[ProcPidStat.START_TIME.ordinal()] * 1000L / USER_HZ;
        if (startTime >= now) {
            startTime = now - 1L;
        }
        proc.setStartTime(startTime);
        proc.setParentProcessID((int)statArray[ProcPidStat.PPID.ordinal()]);
        proc.setThreadCount((int)statArray[ProcPidStat.THREAD_COUNT.ordinal()]);
        proc.setPriority((int)statArray[ProcPidStat.PRIORITY.ordinal()]);
        proc.setVirtualSize(statArray[ProcPidStat.VSZ.ordinal()]);
        proc.setResidentSetSize(statArray[ProcPidStat.RSS.ordinal()] * (long)this.memoryPageSize);
        proc.setKernelTime(statArray[ProcPidStat.KERNEL_TIME.ordinal()] * 1000L / USER_HZ);
        proc.setUserTime(statArray[ProcPidStat.USER_TIME.ordinal()] * 1000L / USER_HZ);
        proc.setUpTime(now - proc.getStartTime());
        proc.setBytesRead(ParseUtil.parseLongOrDefault(io.getOrDefault("read_bytes", ""), 0L));
        proc.setBytesWritten(ParseUtil.parseLongOrDefault(io.getOrDefault("write_bytes", ""), 0L));
        if (slowFields) {
            List<String> openFilesList = ExecutingCommand.runNative(String.format("ls -f /proc/%d/fd", pid));
            proc.setOpenFiles((long)openFilesList.size() - 1L);
            byte[] buffer = new byte[5];
            try (FileInputStream is = new FileInputStream(path);){
                if (((InputStream)is).read(buffer) == buffer.length) {
                    proc.setBitness(buffer[4] == 1 ? 32 : 64);
                }
            }
            catch (IOException e) {
                LOG.warn("Failed to read process file: {}", (Object)path);
            }
        }
        Map<String, String> status = FileUtil.getKeyValueMapFromFile(String.format("/proc/%d/status", pid), ":");
        proc.setName(status.getOrDefault("Name", ""));
        proc.setPath(path);
        switch (status.getOrDefault("State", "U").charAt(0)) {
            case 'R': {
                proc.setState(OSProcess.State.RUNNING);
                break;
            }
            case 'S': {
                proc.setState(OSProcess.State.SLEEPING);
                break;
            }
            case 'D': {
                proc.setState(OSProcess.State.WAITING);
                break;
            }
            case 'Z': {
                proc.setState(OSProcess.State.ZOMBIE);
                break;
            }
            case 'T': {
                proc.setState(OSProcess.State.STOPPED);
                break;
            }
            default: {
                proc.setState(OSProcess.State.OTHER);
            }
        }
        proc.setUserID(ParseUtil.whitespaces.split(status.getOrDefault("Uid", ""))[0]);
        proc.setGroupID(ParseUtil.whitespaces.split(status.getOrDefault("Gid", ""))[0]);
        OSUser user = userGroupInfo.getUser(proc.getUserID());
        if (user != null) {
            proc.setUser(user.getUserName());
        }
        proc.setGroup(userGroupInfo.getGroupName(proc.getGroupID()));
        try {
            String cwdLink = String.format("/proc/%d/cwd", pid);
            String cwd = new File(cwdLink).getCanonicalPath();
            if (!cwd.equals(cwdLink)) {
                proc.setCurrentWorkingDirectory(cwd);
            }
        }
        catch (IOException e) {
            LOG.trace("Couldn't find cwd for pid {}: {}", (Object)pid, (Object)e);
        }
        return proc;
    }

    @Override
    public OSProcess[] getChildProcesses(int parentPid, int limit, OperatingSystem.ProcessSort sort) {
        ArrayList<OSProcess> procs = new ArrayList<OSProcess>();
        File[] procFiles = ProcUtil.getPidFiles();
        LinuxUserGroupInfo userGroupInfo = new LinuxUserGroupInfo();
        for (File procFile : procFiles) {
            OSProcess proc;
            int pid = ParseUtil.parseIntOrDefault(procFile.getName(), 0);
            if (parentPid != LinuxOperatingSystem.getParentPidFromProcFile(pid) || (proc = this.getProcess(pid, userGroupInfo, true)) == null) continue;
            procs.add(proc);
        }
        List<OSProcess> sorted = this.processSort(procs, limit, sort);
        return sorted.toArray(new OSProcess[0]);
    }

    private static int getParentPidFromProcFile(int pid) {
        String stat = FileUtil.getStringFromFile(String.format("/proc/%d/stat", pid));
        long[] statArray = ParseUtil.parseStringToLongArray(stat, PROC_PID_STAT_ORDERS, PROC_PID_STAT_LENGTH, ' ');
        return (int)statArray[ProcPidStat.PPID.ordinal()];
    }

    @Override
    public int getProcessId() {
        return Libc.INSTANCE.getpid();
    }

    @Override
    public int getProcessCount() {
        return ProcUtil.getPidFiles().length;
    }

    @Override
    public int getThreadCount() {
        try {
            LibC.Sysinfo info = new LibC.Sysinfo();
            if (0 != LibC.INSTANCE.sysinfo(info)) {
                LOG.error("Failed to get process thread count. Error code: {}", (Object)Native.getLastError());
                return 0;
            }
            return info.procs;
        }
        catch (NoClassDefFoundError | UnsatisfiedLinkError e) {
            LOG.error("Failed to get procs from sysinfo. {}", (Throwable)e);
            return 0;
        }
    }

    @Override
    public long getSystemUptime() {
        return (long)ProcUtil.getSystemUptimeSeconds();
    }

    @Override
    public long getSystemBootTime() {
        return BOOTTIME;
    }

    @Override
    public NetworkParams getNetworkParams() {
        return new LinuxNetworkParams();
    }

    private void setFamilyFromReleaseFiles() {
        if (this.family == null) {
            if (this.readDistribRelease("/etc/system-release")) {
                return;
            }
            if (this.readOsRelease()) {
                return;
            }
            if (this.execLsbRelease()) {
                return;
            }
            if (this.readLsbRelease()) {
                return;
            }
            String etcDistribRelease = LinuxOperatingSystem.getReleaseFilename();
            if (this.readDistribRelease(etcDistribRelease)) {
                return;
            }
            this.family = LinuxOperatingSystem.filenameToFamily(etcDistribRelease.replace("/etc/", "").replace("release", "").replace("version", "").replace("-", "").replace("_", ""));
        }
    }

    private boolean readOsRelease() {
        if (new File("/etc/os-release").exists()) {
            List<String> osRelease = FileUtil.readFile("/etc/os-release");
            for (String line : osRelease) {
                if (line.startsWith("VERSION=")) {
                    LOG.debug("os-release: {}", (Object)line);
                    line = line.replace("VERSION=", "").replaceAll("^\"|\"$", "").trim();
                    String[] split = line.split("[()]");
                    if (split.length <= 1) {
                        split = line.split(", ");
                    }
                    if (split.length > 0) {
                        this.versionId = split[0].trim();
                    }
                    if (split.length <= 1) continue;
                    this.codeName = split[1].trim();
                    continue;
                }
                if (line.startsWith("NAME=") && this.family == null) {
                    LOG.debug("os-release: {}", (Object)line);
                    this.family = line.replace("NAME=", "").replaceAll("^\"|\"$", "").trim();
                    continue;
                }
                if (!line.startsWith("VERSION_ID=") || this.versionId != null) continue;
                LOG.debug("os-release: {}", (Object)line);
                this.versionId = line.replace("VERSION_ID=", "").replaceAll("^\"|\"$", "").trim();
            }
        }
        return this.family != null;
    }

    private boolean execLsbRelease() {
        for (String line : ExecutingCommand.runNative("lsb_release -a")) {
            if (line.startsWith("Description:")) {
                LOG.debug("lsb_release -a: {}", (Object)line);
                if (!(line = line.replace("Description:", "").trim()).contains(" release ")) continue;
                this.family = this.parseRelease(line, " release ");
                continue;
            }
            if (line.startsWith("Distributor ID:") && this.family == null) {
                LOG.debug("lsb_release -a: {}", (Object)line);
                this.family = line.replace("Distributor ID:", "").trim();
                continue;
            }
            if (line.startsWith("Release:") && this.versionId == null) {
                LOG.debug("lsb_release -a: {}", (Object)line);
                this.versionId = line.replace("Release:", "").trim();
                continue;
            }
            if (!line.startsWith("Codename:") || this.codeName != null) continue;
            LOG.debug("lsb_release -a: {}", (Object)line);
            this.codeName = line.replace("Codename:", "").trim();
        }
        return this.family != null;
    }

    private boolean readLsbRelease() {
        if (new File("/etc/lsb-release").exists()) {
            List<String> osRelease = FileUtil.readFile("/etc/lsb-release");
            for (String line : osRelease) {
                if (line.startsWith("DISTRIB_DESCRIPTION=")) {
                    LOG.debug("lsb-release: {}", (Object)line);
                    if (!(line = line.replace("DISTRIB_DESCRIPTION=", "").replaceAll("^\"|\"$", "").trim()).contains(" release ")) continue;
                    this.family = this.parseRelease(line, " release ");
                    continue;
                }
                if (line.startsWith("DISTRIB_ID=") && this.family == null) {
                    LOG.debug("lsb-release: {}", (Object)line);
                    this.family = line.replace("DISTRIB_ID=", "").replaceAll("^\"|\"$", "").trim();
                    continue;
                }
                if (line.startsWith("DISTRIB_RELEASE=") && this.versionId == null) {
                    LOG.debug("lsb-release: {}", (Object)line);
                    this.versionId = line.replace("DISTRIB_RELEASE=", "").replaceAll("^\"|\"$", "").trim();
                    continue;
                }
                if (!line.startsWith("DISTRIB_CODENAME=") || this.codeName != null) continue;
                LOG.debug("lsb-release: {}", (Object)line);
                this.codeName = line.replace("DISTRIB_CODENAME=", "").replaceAll("^\"|\"$", "").trim();
            }
        }
        return this.family != null;
    }

    private boolean readDistribRelease(String filename) {
        if (new File(filename).exists()) {
            List<String> osRelease = FileUtil.readFile(filename);
            for (String line : osRelease) {
                LOG.debug("{}: {}", (Object)filename, (Object)line);
                if (line.contains(" release ")) {
                    this.family = this.parseRelease(line, " release ");
                    break;
                }
                if (!line.contains(" VERSION ")) continue;
                this.family = this.parseRelease(line, " VERSION ");
                break;
            }
        }
        return this.family != null;
    }

    private String parseRelease(String line, String splitLine) {
        String[] split = line.split(splitLine);
        String family = split[0].trim();
        if (split.length > 1) {
            if ((split = split[1].split("[()]")).length > 0) {
                this.versionId = split[0].trim();
            }
            if (split.length > 1) {
                this.codeName = split[1].trim();
            }
        }
        return family;
    }

    protected static String getReleaseFilename() {
        File etc = new File("/etc");
        File[] matchingFiles = etc.listFiles(f -> (f.getName().endsWith("-release") || f.getName().endsWith("-version") || f.getName().endsWith("_release") || f.getName().endsWith("_version")) && !f.getName().endsWith("os-release") && !f.getName().endsWith("lsb-release") && !f.getName().endsWith("system-release"));
        if (matchingFiles != null && matchingFiles.length > 0) {
            return matchingFiles[0].getPath();
        }
        if (new File("/etc/release").exists()) {
            return "/etc/release";
        }
        return "/etc/issue";
    }

    private static String filenameToFamily(String name) {
        switch (name.toLowerCase()) {
            case "": {
                return "Solaris";
            }
            case "blackcat": {
                return "Black Cat";
            }
            case "bluewhite64": {
                return "BlueWhite64";
            }
            case "e-smith": {
                return "SME Server";
            }
            case "eos": {
                return "FreeEOS";
            }
            case "hlfs": {
                return "HLFS";
            }
            case "lfs": {
                return "Linux-From-Scratch";
            }
            case "linuxppc": {
                return "Linux-PPC";
            }
            case "meego": {
                return "MeeGo";
            }
            case "mandakelinux": {
                return "Mandrake";
            }
            case "mklinux": {
                return "MkLinux";
            }
            case "nld": {
                return "Novell Linux Desktop";
            }
            case "novell": 
            case "SuSE": {
                return "SUSE Linux";
            }
            case "pld": {
                return "PLD";
            }
            case "redhat": {
                return "Red Hat Linux";
            }
            case "sles": {
                return "SUSE Linux ES9";
            }
            case "sun": {
                return "Sun JDS";
            }
            case "synoinfo": {
                return "Synology";
            }
            case "tinysofa": {
                return "Tiny Sofa";
            }
            case "turbolinux": {
                return "TurboLinux";
            }
            case "ultrapenguin": {
                return "UltraPenguin";
            }
            case "va": {
                return "VA-Linux";
            }
            case "vmware": {
                return "VMWareESX";
            }
            case "yellowdog": {
                return "Yellow Dog";
            }
            case "issue": {
                return "Unknown";
            }
        }
        return name.substring(0, 1).toUpperCase() + name.substring(1);
    }

    public static long getHz() {
        return USER_HZ;
    }

    static {
        ProcPidStat[] procStat = FileUtil.readFile("/proc/stat");
        long tempBT = 0L;
        for (String string : procStat) {
            if (!string.startsWith("btime")) continue;
            String[] bTime = ParseUtil.whitespaces.split(string);
            tempBT = ParseUtil.parseLongOrDefault(bTime[1], 0L);
            break;
        }
        if (tempBT == 0L) {
            tempBT = System.currentTimeMillis() / 1000L - (long)ProcUtil.getSystemUptimeSeconds();
        }
        BOOTTIME = tempBT;
        PROC_PID_STAT_ORDERS = new int[ProcPidStat.values().length];
        for (ProcPidStat stat : ProcPidStat.values()) {
            LinuxOperatingSystem.PROC_PID_STAT_ORDERS[stat.ordinal()] = stat.getOrder() - 1;
        }
        List<String> stat = FileUtil.readFile("/proc/self/stat", false);
        if (!stat.isEmpty() && stat.get(0).contains(")")) {
            String procPidStat = stat.get(0);
            int parenIndex = procPidStat.lastIndexOf(41);
            String[] split = ParseUtil.whitespaces.split(procPidStat.substring(parenIndex));
            PROC_PID_STAT_LENGTH = split.length + 1;
        } else {
            PROC_PID_STAT_LENGTH = 52;
        }
        USER_HZ = ParseUtil.parseLongOrDefault(ExecutingCommand.getFirstAnswer("getconf CLK_TCK"), 100L);
        double uptime = ProcUtil.getSystemUptimeSeconds();
        long now = System.currentTimeMillis();
        BOOT_TIME = now - (long)(500.0 * (uptime += ProcUtil.getSystemUptimeSeconds()) + 0.5);
    }

    static enum ProcPidStat {
        PPID(4),
        USER_TIME(14),
        KERNEL_TIME(15),
        PRIORITY(18),
        THREAD_COUNT(20),
        START_TIME(22),
        VSZ(23),
        RSS(24);

        private int order;

        public int getOrder() {
            return this.order;
        }

        private ProcPidStat(int order) {
            this.order = order;
        }
    }
}

