/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.raft.cluster.impl;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import io.atomix.cluster.MemberId;
import io.atomix.raft.cluster.RaftCluster;
import io.atomix.raft.cluster.RaftMember;
import io.atomix.raft.cluster.impl.DefaultRaftMember;
import io.atomix.raft.cluster.impl.RaftMemberContext;
import io.atomix.raft.impl.RaftContext;
import io.atomix.raft.storage.system.Configuration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

public final class RaftClusterContext
implements RaftCluster,
AutoCloseable {
    private final RaftContext raft;
    private final DefaultRaftMember localMember;
    private final Map<MemberId, RaftMemberContext> membersMap = new ConcurrentHashMap<MemberId, RaftMemberContext>();
    private final Set<RaftMember> members = new CopyOnWriteArraySet<RaftMember>();
    private final List<RaftMemberContext> remoteMembers = new CopyOnWriteArrayList<RaftMemberContext>();
    private final Map<RaftMember.Type, List<RaftMemberContext>> memberTypes = new EnumMap<RaftMember.Type, List<RaftMemberContext>>(RaftMember.Type.class);
    private final AtomicReference<Configuration> configurationRef = new AtomicReference();
    private final AtomicReference<CompletableFuture<Void>> bootstrapFutureRef = new AtomicReference();

    public RaftClusterContext(MemberId localMemberId, RaftContext raft) {
        Instant time = Instant.now();
        this.localMember = new DefaultRaftMember(localMemberId, RaftMember.Type.PASSIVE, time).setCluster(this);
        this.raft = (RaftContext)Preconditions.checkNotNull((Object)raft, (Object)"context cannot be null");
        this.configurationRef.set(raft.getMetaStore().loadConfiguration());
        Configuration configuration = this.configurationRef.get();
        if (configuration != null) {
            Instant updateTime = Instant.ofEpochMilli(configuration.time());
            for (RaftMember member : configuration.members()) {
                if (member.equals(this.localMember)) {
                    this.localMember.setType(member.getType());
                    this.members.add(this.localMember);
                    continue;
                }
                RaftMemberContext state = new RaftMemberContext(new DefaultRaftMember(member.memberId(), member.getType(), updateTime), this, raft.getMaxAppendsPerFollower());
                state.resetState(raft.getLog());
                this.members.add(state.getMember());
                this.remoteMembers.add(state);
                this.membersMap.put(member.memberId(), state);
                List<RaftMemberContext> memberType = this.memberTypes.get((Object)member.getType());
                if (memberType == null) {
                    memberType = new CopyOnWriteArrayList<RaftMemberContext>();
                    this.memberTypes.put(member.getType(), memberType);
                }
                memberType.add(state);
            }
        }
    }

    @Override
    public DefaultRaftMember getMember(MemberId id) {
        if (this.localMember.memberId().equals(id)) {
            return this.localMember;
        }
        return this.getRemoteMember(id);
    }

    @Override
    public CompletableFuture<Void> bootstrap(Collection<MemberId> cluster) {
        boolean isOnBoostrapCluster;
        CompletableFuture<Void> bootstrapFuture = this.bootstrapFutureRef.get();
        if (bootstrapFuture != null) {
            return bootstrapFuture;
        }
        this.ensureConfigurationIsConsistent(cluster);
        this.bootstrapFutureRef.set(new CompletableFuture());
        boolean bl = isOnBoostrapCluster = this.configurationRef.get() == null;
        if (isOnBoostrapCluster) {
            this.localMember.setType(RaftMember.Type.ACTIVE);
            this.createInitialConfig(cluster);
        }
        this.raft.getThreadContext().execute(() -> {
            this.raft.transition(this.localMember.getType());
            if (isOnBoostrapCluster) {
                this.commit();
            }
            this.completeBootstrapFuture();
        });
        return this.bootstrapFutureRef.get().whenComplete((result, error) -> this.bootstrapFutureRef.set(null));
    }

    @Override
    public RaftMember getLeader() {
        return this.raft.getLeader();
    }

    @Override
    public RaftMember getLocalMember() {
        return this.localMember;
    }

    @Override
    public Collection<RaftMember> getMembers() {
        return new ArrayList<RaftMember>(this.members);
    }

    @Override
    public long getTerm() {
        return this.raft.getTerm();
    }

    private void ensureConfigurationIsConsistent(Collection<MemberId> cluster) {
        boolean hasPersistedConfiguration;
        Configuration configuration = this.configurationRef.get();
        boolean bl = hasPersistedConfiguration = configuration != null;
        if (hasPersistedConfiguration) {
            int newClusterSize = cluster.size();
            int persistedClusterSize = configuration.members().size();
            if (persistedClusterSize != newClusterSize) {
                throw new IllegalStateException(String.format("Expected that persisted cluster size '%d' is equal to given one '%d', but was different. Persisted configuration '%s' is different then given one, new given member id's are: '%s'. Changing the configuration is not supported. Please restart with the same configuration or recreate a new cluster after deleting persisted data.", persistedClusterSize, newClusterSize, configuration, Arrays.toString(cluster.toArray())));
            }
            Collection<RaftMember> persistedMembers = configuration.members();
            for (MemberId memberId : cluster) {
                boolean noMatch = persistedMembers.stream().map(RaftMember::memberId).noneMatch(persistedMemberId -> persistedMemberId.equals(memberId));
                if (!noMatch) continue;
                throw new IllegalStateException(String.format("Expected to find given node id '%s' in persisted members '%s', but was not found. Persisted configuration is different then given one. Changing the configuration is not supported. Please restart with the same configuration or recreate a new cluster after deleting persisted data.", memberId, Arrays.toString(persistedMembers.toArray())));
            }
        }
    }

    private void createInitialConfig(Collection<MemberId> cluster) {
        Set<RaftMember> activeMembers = cluster.stream().filter(m -> !m.equals(this.localMember.memberId())).map(m -> new DefaultRaftMember((MemberId)m, RaftMember.Type.ACTIVE, this.localMember.getLastUpdated())).collect(Collectors.toSet());
        activeMembers.add(this.localMember);
        this.configure(new Configuration(0L, 0L, this.localMember.getLastUpdated().toEpochMilli(), activeMembers));
    }

    public DefaultRaftMember getRemoteMember(MemberId id) {
        RaftMemberContext member = this.membersMap.get(id);
        return member != null ? member.getMember() : null;
    }

    public RaftMemberContext getMemberState(MemberId id) {
        return this.membersMap.get(id);
    }

    public List<RaftMemberContext> getActiveMemberStates(Comparator<RaftMemberContext> comparator) {
        ArrayList<RaftMemberContext> activeMembers = new ArrayList<RaftMemberContext>(this.getActiveMemberStates());
        activeMembers.sort(comparator);
        return activeMembers;
    }

    public List<RaftMemberContext> getActiveMemberStates() {
        return this.getRemoteMemberStates(RaftMember.Type.ACTIVE);
    }

    public List<RaftMemberContext> getRemoteMemberStates(RaftMember.Type type) {
        List<RaftMemberContext> memberContexts = this.memberTypes.get((Object)type);
        return memberContexts != null ? memberContexts : List.of();
    }

    private void completeBootstrapFuture() {
        CompletableFuture<Void> bootstrapFuture = this.bootstrapFutureRef.get();
        if (!this.members.contains(this.localMember)) {
            bootstrapFuture.completeExceptionally(new IllegalStateException("not a member of the cluster"));
        } else {
            bootstrapFuture.complete(null);
        }
    }

    public RaftClusterContext reset() {
        this.configure(this.raft.getMetaStore().loadConfiguration());
        return this;
    }

    public RaftClusterContext configure(Configuration configuration) {
        Preconditions.checkNotNull((Object)configuration, (Object)"configuration cannot be null");
        Configuration currentConfig = this.configurationRef.get();
        if (currentConfig != null && configuration.index() <= currentConfig.index()) {
            return this;
        }
        Instant time = Instant.ofEpochMilli(configuration.time());
        for (RaftMember member : configuration.members()) {
            this.updateMember(member, time);
        }
        if (this.wasPromoted(configuration)) {
            this.raft.transition(this.localMember.getType());
        }
        if (!configuration.members().contains(this.localMember)) {
            this.members.remove(this.localMember);
        }
        this.configurationRef.set(configuration);
        if (this.raft.getCommitIndex() >= configuration.index()) {
            this.raft.getMetaStore().storeConfiguration(configuration);
        }
        return this;
    }

    private boolean wasPromoted(Configuration configuration) {
        return configuration.members().stream().anyMatch(m -> m.equals(this.localMember) && this.localMember.getType().ordinal() < m.getType().ordinal());
    }

    private void updateMember(RaftMember member, Instant time) {
        if (member.equals(this.localMember)) {
            this.localMember.update(member.getType(), time);
            this.members.add(this.localMember);
            return;
        }
        RaftMemberContext state = this.membersMap.get(member.memberId());
        if (state == null) {
            DefaultRaftMember defaultMember = new DefaultRaftMember(member.memberId(), member.getType(), time);
            state = new RaftMemberContext(defaultMember, this, this.raft.getMaxAppendsPerFollower());
            state.resetState(this.raft.getLog());
            this.members.add(state.getMember());
            this.remoteMembers.add(state);
            this.membersMap.put(member.memberId(), state);
        }
        if (state.getMember().getType() != member.getType()) {
            state.getMember().update(member.getType(), time);
            state.resetState(this.raft.getLog());
        }
        for (List list : this.memberTypes.values()) {
            list.remove(state);
        }
        List<RaftMemberContext> memberType = this.memberTypes.get((Object)member.getType());
        if (memberType == null) {
            memberType = new CopyOnWriteArrayList<RaftMemberContext>();
            this.memberTypes.put(member.getType(), memberType);
        }
        memberType.add(state);
    }

    public RaftClusterContext commit() {
        this.raft.transition(this.localMember.getType());
        Configuration configuration = this.configurationRef.get();
        if (this.raft.getMetaStore().loadConfiguration().index() < configuration.index()) {
            this.raft.getMetaStore().storeConfiguration(configuration);
        }
        return this;
    }

    @Override
    public void close() {
        for (RaftMemberContext member : this.remoteMembers) {
            member.getMember().close();
        }
        this.localMember.close();
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("server", (Object)this.raft.getName()).toString();
    }

    public Configuration getConfiguration() {
        return this.configurationRef.get();
    }

    public RaftContext getContext() {
        return this.raft;
    }

    public int getQuorum() {
        return (int)Math.floor((double)(this.getActiveMemberStates().size() + 1) / 2.0) + 1;
    }

    public List<RaftMemberContext> getRemoteMemberStates() {
        return this.remoteMembers;
    }
}

