/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.journal.file;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import io.camunda.zeebe.journal.Journal;
import io.camunda.zeebe.journal.JournalException;
import io.camunda.zeebe.journal.JournalReader;
import io.camunda.zeebe.journal.JournalRecord;
import io.camunda.zeebe.journal.file.JournalIndex;
import io.camunda.zeebe.journal.file.JournalMetrics;
import io.camunda.zeebe.journal.file.JournalSegment;
import io.camunda.zeebe.journal.file.JournalSegmentDescriptor;
import io.camunda.zeebe.journal.file.JournalSegmentFile;
import io.camunda.zeebe.journal.file.SegmentedJournalBuilder;
import io.camunda.zeebe.journal.file.SegmentedJournalReader;
import io.camunda.zeebe.journal.file.SegmentedJournalWriter;
import io.camunda.zeebe.journal.file.UnknownVersionException;
import io.camunda.zeebe.journal.file.record.CorruptedLogException;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.SortedMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.locks.StampedLock;
import org.agrona.DirectBuffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentedJournal
implements Journal {
    public static final long ASQN_IGNORE = -1L;
    private static final int SEGMENT_BUFFER_FACTOR = 3;
    private static final int FIRST_SEGMENT_ID = 1;
    private static final int INITIAL_INDEX = 1;
    private final JournalMetrics journalMetrics;
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final String name;
    private final File directory;
    private final int maxSegmentSize;
    private final NavigableMap<Long, JournalSegment> segments = new ConcurrentSkipListMap<Long, JournalSegment>();
    private final Collection<SegmentedJournalReader> readers = Sets.newConcurrentHashSet();
    private volatile JournalSegment currentSegment;
    private volatile boolean open = true;
    private final long minFreeDiskSpace;
    private final JournalIndex journalIndex;
    private final SegmentedJournalWriter writer;
    private final long lastWrittenIndex;
    private final StampedLock rwlock = new StampedLock();

    public SegmentedJournal(String name, File directory, int maxSegmentSize, long minFreeSpace, JournalIndex journalIndex, long lastWrittenIndex) {
        this.name = (String)Preconditions.checkNotNull((Object)name, (Object)"name cannot be null");
        this.directory = (File)Preconditions.checkNotNull((Object)directory, (Object)"directory cannot be null");
        this.maxSegmentSize = maxSegmentSize;
        this.journalMetrics = new JournalMetrics(name);
        this.minFreeDiskSpace = minFreeSpace;
        this.journalIndex = journalIndex;
        this.lastWrittenIndex = lastWrittenIndex;
        this.open();
        this.writer = new SegmentedJournalWriter(this);
    }

    public static SegmentedJournalBuilder builder() {
        return new SegmentedJournalBuilder();
    }

    @Override
    public JournalRecord append(long asqn, DirectBuffer data) {
        return this.writer.append(asqn, data);
    }

    @Override
    public JournalRecord append(DirectBuffer data) {
        return this.writer.append(-1L, data);
    }

    @Override
    public void append(JournalRecord record) {
        this.writer.append(record);
    }

    @Override
    public void deleteAfter(long indexExclusive) {
        this.journalMetrics.observeSegmentTruncation(() -> {
            long stamp = this.rwlock.writeLock();
            try {
                this.writer.deleteAfter(indexExclusive);
                this.resetAdvancedReaders(indexExclusive + 1L);
            }
            finally {
                this.rwlock.unlockWrite(stamp);
            }
        });
    }

    @Override
    public void deleteUntil(long index) {
        Map.Entry<Long, JournalSegment> segmentEntry = this.segments.floorEntry(index);
        if (segmentEntry != null) {
            SortedMap<Long, JournalSegment> compactSegments = this.segments.headMap(segmentEntry.getValue().index());
            if (compactSegments.isEmpty()) {
                this.log.debug("No segments can be deleted with index < {} (first log index: {})", (Object)index, (Object)this.getFirstIndex());
                return;
            }
            this.log.debug("{} - Deleting log up from {} up to {} (removing {} segments)", new Object[]{this.name, this.getFirstIndex(), ((JournalSegment)compactSegments.get(compactSegments.lastKey())).index(), compactSegments.size()});
            for (JournalSegment segment : compactSegments.values()) {
                this.log.trace("{} - Deleting segment: {}", (Object)this.name, (Object)segment);
                segment.close();
                segment.delete();
                this.journalMetrics.decSegmentCount();
            }
            compactSegments.clear();
            this.journalIndex.deleteUntil(index);
            this.resetHead(this.getFirstSegment().index());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void reset(long nextIndex) {
        long stamp = this.rwlock.writeLock();
        try {
            this.journalIndex.clear();
            this.writer.reset(nextIndex);
            this.resetHead(nextIndex);
        }
        finally {
            this.rwlock.unlockWrite(stamp);
        }
    }

    @Override
    public long getLastIndex() {
        return this.writer.getLastIndex();
    }

    @Override
    public long getFirstIndex() {
        if (!this.segments.isEmpty()) {
            return this.segments.firstEntry().getValue().index();
        }
        return 0L;
    }

    @Override
    public boolean isEmpty() {
        return this.writer.getNextIndex() - this.getFirstSegment().index() == 0L;
    }

    @Override
    public void flush() {
        this.writer.flush();
    }

    @Override
    public JournalReader openReader() {
        SegmentedJournalReader reader = new SegmentedJournalReader(this);
        this.readers.add(reader);
        return reader;
    }

    @Override
    public boolean isOpen() {
        return this.open;
    }

    @Override
    public void close() {
        this.segments.values().forEach(segment -> {
            this.log.debug("Closing segment: {}", segment);
            segment.close();
        });
        this.currentSegment = null;
        this.open = false;
    }

    private synchronized void open() {
        long startTime = System.currentTimeMillis();
        for (JournalSegment segment : this.loadSegments()) {
            this.segments.put(segment.descriptor().index(), segment);
            this.journalMetrics.incSegmentCount();
        }
        if (!this.segments.isEmpty()) {
            this.currentSegment = this.segments.lastEntry().getValue();
        } else {
            JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(1L).withIndex(1L).withMaxSegmentSize(this.maxSegmentSize).build();
            this.currentSegment = this.createSegment(descriptor);
            this.segments.put(1L, this.currentSegment);
            this.journalMetrics.incSegmentCount();
        }
        this.journalMetrics.observeJournalOpenDuration(System.currentTimeMillis() - startTime);
    }

    private void assertOpen() {
        Preconditions.checkState((this.currentSegment != null ? 1 : 0) != 0, (Object)"journal not open");
    }

    private void assertDiskSpace() {
        if (this.directory().getUsableSpace() < Math.max(this.maxSegmentSize() * 3L, this.minFreeDiskSpace)) {
            throw new JournalException.OutOfDiskSpace("Not enough space to allocate a new journal segment");
        }
    }

    private long maxSegmentSize() {
        return this.maxSegmentSize;
    }

    private File directory() {
        return this.directory;
    }

    private synchronized void resetCurrentSegment() {
        JournalSegment lastSegment = this.getLastSegment();
        if (lastSegment != null) {
            this.currentSegment = lastSegment;
        } else {
            JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(1L).withIndex(1L).withMaxSegmentSize(this.maxSegmentSize).build();
            this.currentSegment = this.createSegment(descriptor);
            this.segments.put(1L, this.currentSegment);
            this.journalMetrics.incSegmentCount();
        }
    }

    JournalSegment resetSegments(long index) {
        this.assertOpen();
        for (JournalSegment segment : this.segments.values()) {
            segment.close();
            segment.delete();
            this.journalMetrics.decSegmentCount();
        }
        this.segments.clear();
        JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(1L).withIndex(index).withMaxSegmentSize(this.maxSegmentSize).build();
        this.currentSegment = this.createSegment(descriptor);
        this.segments.put(index, this.currentSegment);
        this.journalMetrics.incSegmentCount();
        return this.currentSegment;
    }

    JournalSegment getFirstSegment() {
        this.assertOpen();
        Map.Entry<Long, JournalSegment> segment = this.segments.firstEntry();
        return segment != null ? segment.getValue() : null;
    }

    JournalSegment getLastSegment() {
        this.assertOpen();
        Map.Entry<Long, JournalSegment> segment = this.segments.lastEntry();
        return segment != null ? segment.getValue() : null;
    }

    synchronized JournalSegment getNextSegment() {
        this.assertOpen();
        this.assertDiskSpace();
        JournalSegment lastSegment = this.getLastSegment();
        JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(lastSegment != null ? lastSegment.descriptor().id() + 1L : 1L).withIndex(this.currentSegment.lastIndex() + 1L).withMaxSegmentSize(this.maxSegmentSize).build();
        this.currentSegment = this.createSegment(descriptor);
        this.segments.put(descriptor.index(), this.currentSegment);
        this.journalMetrics.incSegmentCount();
        return this.currentSegment;
    }

    JournalSegment getNextSegment(long index) {
        Map.Entry<Long, JournalSegment> nextSegment = this.segments.higherEntry(index);
        return nextSegment != null ? nextSegment.getValue() : null;
    }

    synchronized JournalSegment getSegment(long index) {
        this.assertOpen();
        if (this.currentSegment != null && index > this.currentSegment.index()) {
            return this.currentSegment;
        }
        Map.Entry<Long, JournalSegment> segment = this.segments.floorEntry(index);
        if (segment != null) {
            return segment.getValue();
        }
        return this.getFirstSegment();
    }

    synchronized void removeSegment(JournalSegment segment) {
        this.segments.remove(segment.index());
        this.journalMetrics.decSegmentCount();
        segment.close();
        segment.delete();
        this.resetCurrentSegment();
    }

    JournalSegment createSegment(JournalSegmentDescriptor descriptor) {
        FileChannel channel;
        RandomAccessFile raf;
        File segmentFile = JournalSegmentFile.createSegmentFile(this.name, this.directory, descriptor.id());
        try {
            raf = new RandomAccessFile(segmentFile, "rw");
            raf.setLength(descriptor.maxSegmentSize());
            channel = raf.getChannel();
        }
        catch (IOException e) {
            throw new JournalException(e);
        }
        ByteBuffer buffer = ByteBuffer.allocate(JournalSegmentDescriptor.getEncodingLength());
        descriptor.copyTo(buffer);
        try {
            channel.write(buffer);
        }
        catch (IOException e) {
            throw new JournalException(e);
        }
        finally {
            try {
                channel.close();
                raf.close();
            }
            catch (IOException e) {
                this.log.warn("Unexpected IOException on closing", (Throwable)e);
            }
        }
        JournalSegment segment = this.loadSegment(segmentFile, descriptor);
        this.log.debug("Created segment: {}", (Object)segment);
        return segment;
    }

    protected JournalSegment loadSegment(File file, JournalSegmentDescriptor descriptor) {
        JournalSegmentFile segmentFile = new JournalSegmentFile(file);
        return new JournalSegment(segmentFile, descriptor, this.lastWrittenIndex, this.journalIndex);
    }

    private FileChannel openChannel(File file) {
        try {
            return FileChannel.open(file.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        }
        catch (IOException e) {
            throw new JournalException(e);
        }
    }

    protected Collection<JournalSegment> loadSegments() {
        this.directory.mkdirs();
        ArrayList<JournalSegment> segments = new ArrayList<JournalSegment>();
        List<File> files = this.getSortedLogSegments();
        for (int i = 0; i < files.size(); ++i) {
            File file = files.get(i);
            try {
                this.log.debug("Found segment file: {}", (Object)file.getName());
                JournalSegmentDescriptor descriptor = this.readDescriptor(file);
                JournalSegment segment = this.loadSegment(file, descriptor);
                if (i > 0) {
                    this.checkForIndexGaps((JournalSegment)segments.get(i - 1), segment);
                }
                segments.add(segment);
                continue;
            }
            catch (CorruptedLogException e) {
                if (this.handleSegmentCorruption(files, segments, i)) {
                    return segments;
                }
                throw e;
            }
        }
        return segments;
    }

    private void checkForIndexGaps(JournalSegment prevSegment, JournalSegment segment) {
        if (prevSegment.lastIndex() != segment.index() - 1L) {
            throw new CorruptedLogException(String.format("Log segment %s is not aligned with previous segment %s (last index: %d).", segment, prevSegment, prevSegment.lastIndex()));
        }
    }

    private boolean handleSegmentCorruption(List<File> files, List<JournalSegment> segments, int failedIndex) {
        long lastSegmentIndex = 0L;
        if (!segments.isEmpty()) {
            JournalSegment previousSegment = segments.get(segments.size() - 1);
            lastSegmentIndex = previousSegment.lastIndex();
        }
        if (this.lastWrittenIndex > lastSegmentIndex) {
            return false;
        }
        this.log.debug("Found corrupted segment after last ack'ed index {}. Deleting segments {} - {}", new Object[]{this.lastWrittenIndex, files.get(failedIndex).getName(), files.get(files.size() - 1).getName()});
        for (int i = failedIndex; i < files.size(); ++i) {
            File file = files.get(i);
            try {
                Files.delete(file.toPath());
                continue;
            }
            catch (IOException e) {
                throw new JournalException(String.format("Failed to delete log segment '%s' when handling corruption.", file.getName()), e);
            }
        }
        return true;
    }

    private List<File> getSortedLogSegments() {
        File[] files = this.directory.listFiles(file -> file.isFile() && JournalSegmentFile.isSegmentFile(this.name, file));
        if (files == null) {
            throw new IllegalStateException(String.format("Could not list files in directory '%s'. Either the path doesn't point to a directory or an I/O error occurred.", this.directory));
        }
        Arrays.sort(files, Comparator.comparingInt(f -> JournalSegmentFile.getSegmentIdFromPath(f.getName())));
        return Arrays.asList(files);
    }

    private JournalSegmentDescriptor readDescriptor(File file) {
        JournalSegmentDescriptor journalSegmentDescriptor;
        block12: {
            FileChannel channel = this.openChannel(file);
            try {
                byte version = this.readVersion(channel, file.getName());
                int length = JournalSegmentDescriptor.getEncodingLengthForVersion(version);
                if (file.length() < (long)length) {
                    throw new CorruptedLogException(String.format("Expected segment '%s' with version %d to be at least %d bytes long but it only has %d.", file.getName(), version, length, file.length()));
                }
                ByteBuffer buffer = ByteBuffer.allocate(length);
                int readBytes = channel.read(buffer, 0L);
                if (readBytes != -1 && readBytes < length) {
                    throw new JournalException(String.format("Expected to read %d bytes of segment '%s' with %d version but only read %d bytes.", length, file.getName(), version, readBytes));
                }
                buffer.flip();
                journalSegmentDescriptor = new JournalSegmentDescriptor(buffer);
                if (channel == null) break block12;
            }
            catch (Throwable throwable) {
                try {
                    if (channel != null) {
                        try {
                            channel.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IndexOutOfBoundsException e) {
                    throw new JournalException(String.format("Expected to read descriptor of segment '%s', but nothing was read.", file.getName()), e);
                }
                catch (UnknownVersionException e) {
                    throw new CorruptedLogException(String.format("Couldn't read or recognize version of segment '%s'.", file.getName()), e);
                }
                catch (IOException e) {
                    throw new JournalException(e);
                }
            }
            channel.close();
        }
        return journalSegmentDescriptor;
    }

    private byte readVersion(FileChannel channel, String fileName) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(1);
        int readBytes = channel.read(buffer);
        if (readBytes == 0) {
            throw new JournalException(String.format("Expected to read the version byte from segment '%s' but nothing was read.", fileName));
        }
        if (readBytes == -1) {
            throw new CorruptedLogException(String.format("Expected to read the version byte from segment '%s' but got EOF instead.", fileName));
        }
        return buffer.get(0);
    }

    public void closeReader(SegmentedJournalReader segmentedJournalReader) {
        this.readers.remove(segmentedJournalReader);
    }

    void resetHead(long index) {
        for (SegmentedJournalReader reader : this.readers) {
            if (reader.getNextIndex() > index) continue;
            reader.unsafeSeek(index);
        }
    }

    void resetAdvancedReaders(long index) {
        for (SegmentedJournalReader reader : this.readers) {
            if (reader.getNextIndex() <= index) continue;
            reader.unsafeSeek(index);
        }
    }

    public JournalMetrics getJournalMetrics() {
        return this.journalMetrics;
    }

    public JournalIndex getJournalIndex() {
        return this.journalIndex;
    }

    long acquireReadlock() {
        return this.rwlock.readLock();
    }

    void releaseReadlock(long stamp) {
        this.rwlock.unlockRead(stamp);
    }
}

