/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.test.util.record;

import io.camunda.zeebe.protocol.Protocol;
import io.camunda.zeebe.protocol.record.Record;
import io.camunda.zeebe.protocol.record.RecordType;
import io.camunda.zeebe.protocol.record.ValueType;
import io.camunda.zeebe.protocol.record.intent.IncidentIntent;
import io.camunda.zeebe.protocol.record.intent.Intent;
import io.camunda.zeebe.protocol.record.value.DeploymentRecordValue;
import io.camunda.zeebe.protocol.record.value.ErrorRecordValue;
import io.camunda.zeebe.protocol.record.value.IncidentRecordValue;
import io.camunda.zeebe.protocol.record.value.JobBatchRecordValue;
import io.camunda.zeebe.protocol.record.value.JobRecordValue;
import io.camunda.zeebe.protocol.record.value.MessageRecordValue;
import io.camunda.zeebe.protocol.record.value.MessageStartEventSubscriptionRecordValue;
import io.camunda.zeebe.protocol.record.value.MessageSubscriptionRecordValue;
import io.camunda.zeebe.protocol.record.value.ProcessEventRecordValue;
import io.camunda.zeebe.protocol.record.value.ProcessInstanceCreationRecordValue;
import io.camunda.zeebe.protocol.record.value.ProcessInstanceRecordValue;
import io.camunda.zeebe.protocol.record.value.ProcessMessageSubscriptionRecordValue;
import io.camunda.zeebe.protocol.record.value.TimerRecordValue;
import io.camunda.zeebe.protocol.record.value.VariableRecordValue;
import io.camunda.zeebe.protocol.record.value.deployment.Process;
import io.camunda.zeebe.protocol.record.value.deployment.ProcessMetadataValue;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactRecordLogger {
    private static final Logger LOG = LoggerFactory.getLogger((String)"io.camunda.zeebe.test");
    private static final String BLOCK_SEPARATOR = " - ";
    private static final Map<String, String> ABBREVIATIONS = Map.ofEntries(Map.entry("PROCESS", "PROC"), Map.entry("INSTANCE", "INST"), Map.entry("MESSAGE", "MSG"), Map.entry("SUBSCRIPTION", "SUB"), Map.entry("SEQUENCE", "SEQ"), Map.entry("DISTRIBUTED", "DISTR"), Map.entry("VARIABLE", "VAR"), Map.entry("ELEMENT_", ""), Map.entry("_ELEMENT", ""), Map.entry("EVENT", "EVNT"));
    private static final Map<RecordType, Character> RECORD_TYPE_ABBREVIATIONS = Map.ofEntries(Map.entry(RecordType.COMMAND, Character.valueOf('C')), Map.entry(RecordType.EVENT, Character.valueOf('E')), Map.entry(RecordType.COMMAND_REJECTION, Character.valueOf('R')));
    private final Map<ValueType, Function<Record<?>, String>> valueLoggers = new HashMap();
    private final int keyDigits;
    private final int valueTypeChars;
    private final int intentChars;
    private final boolean singlePartition;
    private final Map<Long, String> substitutions = new HashMap<Long, String>();
    private final ArrayList<Record<?>> records;
    private final List<Process> processes = new ArrayList<Process>();

    public CompactRecordLogger(Collection<Record<?>> records) {
        this.valueLoggers.put(ValueType.DEPLOYMENT, this::summarizeDeployment);
        this.valueLoggers.put(ValueType.PROCESS, this::summarizeProcess);
        this.valueLoggers.put(ValueType.INCIDENT, this::summarizeIncident);
        this.valueLoggers.put(ValueType.JOB, this::summarizeJob);
        this.valueLoggers.put(ValueType.JOB_BATCH, this::summarizeJobBatch);
        this.valueLoggers.put(ValueType.MESSAGE, this::summarizeMessage);
        this.valueLoggers.put(ValueType.MESSAGE_START_EVENT_SUBSCRIPTION, this::summarizeMessageStartEventSubscription);
        this.valueLoggers.put(ValueType.MESSAGE_SUBSCRIPTION, this::summarizeMessageSubscription);
        this.valueLoggers.put(ValueType.PROCESS_INSTANCE, this::summarizeProcessInstance);
        this.valueLoggers.put(ValueType.PROCESS_INSTANCE_CREATION, this::summarizeProcessInstanceCreation);
        this.valueLoggers.put(ValueType.PROCESS_MESSAGE_SUBSCRIPTION, this::summarizeProcessInstanceSubscription);
        this.valueLoggers.put(ValueType.VARIABLE, this::summarizeVariable);
        this.valueLoggers.put(ValueType.TIMER, this::summarizeTimer);
        this.valueLoggers.put(ValueType.ERROR, this::summarizeError);
        this.valueLoggers.put(ValueType.PROCESS_EVENT, this::summarizeProcessEvent);
        this.records = new ArrayList(records);
        this.singlePartition = this.records.stream().mapToLong(Record::getKey).filter(key -> key != -1L).map(Protocol::decodePartitionId).distinct().count() < 2L;
        long highestPosition = this.records.get(this.records.size() - 1).getPosition();
        int digits = 0;
        long num = highestPosition;
        while (num != 0L) {
            num /= 10L;
            ++digits;
        }
        this.keyDigits = digits;
        this.valueTypeChars = this.records.stream().map(Record::getValueType).map(Enum::name).map(this::abbreviate).mapToInt(String::length).max().orElse(0);
        this.intentChars = this.records.stream().map(Record::getIntent).map(Intent::name).map(this::abbreviate).mapToInt(String::length).max().orElse(0);
    }

    public void log() {
        StringBuilder bulkMessage = new StringBuilder().append("Compact log representation:\n");
        this.addSummarizedRecords(bulkMessage);
        this.addDeployedProcesses(bulkMessage);
        this.addDecomposedKeys(bulkMessage);
        LOG.info(bulkMessage.toString());
    }

    private void addSummarizedRecords(StringBuilder bulkMessage) {
        bulkMessage.append("--------\n").append("\t['C'ommand/'E'event/'R'ejection] [valueType] [intent] - #[position]->#[source record position]  P[partitionId]K[key] - [summary of value]\n").append("\tP9K999 - key; #999 - record position; \"ID\" element/process id; @\"elementid\"/[P9K999] - element with ID and key\n").append("\tKeys are decomposed into partition id and per partition key (e.g. 2251799813685253 -> P1K005). If single partition, the partition is omitted.\n").append("\tLong IDs are shortened (e.g. 'startEvent_5d56488e-0570-416c-ba2d-36d2a3acea78' -> 'star..acea78'\n").append("--------\n");
        this.records.forEach((Consumer<Record<?>>)((Consumer<Record>)record -> bulkMessage.append((CharSequence)this.summarizeRecord((Record<?>)record)).append("\n")));
    }

    private void addDeployedProcesses(StringBuilder bulkMessage) {
        bulkMessage.append("\n-------------- Deployed Processes ----------------------\n");
        this.processes.forEach(process -> bulkMessage.append(this.summarizeProcessInformation((Process)process)).append(String.format("[%s] ------%n%s%n", this.formatKey(process.getProcessDefinitionKey()), new String(process.getResource()))));
    }

    private void addDecomposedKeys(StringBuilder bulkMessage) {
        bulkMessage.append("\n--------------- Decomposed keys (for debugging) -----------------\n");
        this.substitutions.entrySet().stream().sorted(Comparator.comparing(Map.Entry::getValue)).forEach(entry -> bulkMessage.append((String)entry.getValue()).append(" <-> ").append(entry.getKey()).append("\n"));
    }

    private StringBuilder summarizeRecord(Record<?> record) {
        StringBuilder message = new StringBuilder();
        message.append((CharSequence)this.summarizeIntent(record));
        message.append((CharSequence)this.summarizePositionFields(record));
        message.append(this.summarizeValue(record));
        if (record.getRecordType() == RecordType.COMMAND_REJECTION) {
            message.append(" ");
            message.append((CharSequence)this.summarizeRejection(record));
        }
        return message;
    }

    private StringBuilder summarizePositionFields(Record<?> record) {
        return new StringBuilder().append(this.formatPosition(record.getPosition())).append("->").append(this.formatPosition(record.getSourceRecordPosition())).append(" ").append(this.shortenKey(record.getKey())).append(BLOCK_SEPARATOR);
    }

    private StringBuilder summarizeIntent(Record<?> record) {
        ValueType valueType = record.getValueType();
        return new StringBuilder().append(RECORD_TYPE_ABBREVIATIONS.get(record.getRecordType())).append(" ").append(StringUtils.rightPad((String)this.abbreviate(valueType.name()), (int)this.valueTypeChars)).append(" ").append(StringUtils.rightPad((String)this.abbreviate(record.getIntent().name()), (int)this.intentChars)).append(BLOCK_SEPARATOR);
    }

    private String summarizeValue(Record<?> record) {
        return this.valueLoggers.getOrDefault(record.getValueType(), this::summarizeMiscValue).apply(record);
    }

    private String summarizeMiscValue(Record<?> record) {
        return record.getValue().getClass().getSimpleName() + " " + record.getValue().toJson();
    }

    private String summarizeDeployment(Record<?> record) {
        DeploymentRecordValue value = (DeploymentRecordValue)record.getValue();
        return value.getProcessesMetadata().stream().map(ProcessMetadataValue::getResourceName).collect(Collectors.joining(", "));
    }

    private String summarizeProcess(Record<?> record) {
        Process value = (Process)record.getValue();
        this.processes.add(value);
        return this.summarizeProcessInformation(value);
    }

    private String summarizeProcessInformation(Process value) {
        return String.format("%s -> %s (version:%d)", value.getResourceName(), this.formatId(value.getBpmnProcessId()), value.getVersion());
    }

    private String summarizeElementInformation(String elementId, long elementInstanceKey) {
        return String.format(" @%s[%s]", this.formatId(elementId), this.shortenKey(elementInstanceKey));
    }

    private String summarizeProcessInformation(String bpmnProcessId, long processInstanceKey) {
        String formattedProcessId = StringUtils.isEmpty((CharSequence)bpmnProcessId) ? "?" : this.formatId(bpmnProcessId);
        String formattedInstanceKey = processInstanceKey < 0L ? "?" : this.shortenKey(processInstanceKey);
        return String.format(" in <process %s[%s]>", formattedProcessId, formattedInstanceKey);
    }

    private String summarizeProcessInformation(long processDefinitionKey, long processInstanceKey) {
        String formattedDefinitionKey = processDefinitionKey < 0L ? "?" : this.shortenKey(processDefinitionKey);
        String formattedInstanceKey = processInstanceKey < 0L ? "?" : this.shortenKey(processInstanceKey);
        return String.format(" in <process %s[%s]>", formattedDefinitionKey, formattedInstanceKey);
    }

    private String summarizeVariables(Map<String, Object> variables) {
        if (variables != null && !variables.isEmpty()) {
            return " with variables: " + variables;
        }
        return " (no vars)";
    }

    private String summarizeIncident(Record<?> record) {
        IncidentRecordValue value = (IncidentRecordValue)record.getValue();
        StringBuilder result = new StringBuilder();
        if (record.getIntent() != IncidentIntent.RESOLVE) {
            result.append(value.getErrorType()).append(" ").append(value.getErrorMessage()).append(", ");
            if (value.getJobKey() != -1L) {
                result.append("joBKey: ").append(this.shortenKey(value.getJobKey())).append(" ");
            }
            result.append(this.summarizeElementInformation(value.getElementId(), value.getElementInstanceKey())).append(this.summarizeProcessInformation(value.getBpmnProcessId(), value.getProcessInstanceKey()));
        } else {
            result.append(this.shortenKey(record.getKey()));
        }
        return result.toString();
    }

    private String summarizeJob(Record<?> record) {
        JobRecordValue value = (JobRecordValue)record.getValue();
        return this.summarizeJobRecordValue(record.getKey(), value);
    }

    private String summarizeJobRecordValue(long jobKey, JobRecordValue value) {
        StringBuilder result = new StringBuilder();
        if (jobKey != -1L) {
            result.append(this.shortenKey(jobKey));
        }
        if (!StringUtils.isEmpty((CharSequence)value.getType())) {
            result.append(" \"").append(value.getType()).append("\"").append(this.summarizeElementInformation(value.getElementId(), value.getElementInstanceKey()));
        }
        result.append(" ").append(value.getRetries()).append(" retries,");
        if (!StringUtils.isEmpty((CharSequence)value.getErrorCode())) {
            result.append(" ").append(value.getErrorCode()).append(":").append(value.getErrorMessage());
        }
        result.append(this.summarizeProcessInformation(value.getBpmnProcessId(), value.getProcessInstanceKey())).append(this.summarizeVariables(value.getVariables()));
        return result.toString();
    }

    private String summarizeJobBatch(Record<?> record) {
        JobBatchRecordValue value = (JobBatchRecordValue)record.getValue();
        List jobKeys = value.getJobKeys();
        StringBuilder result = new StringBuilder();
        result.append("\"").append(value.getType()).append("\" ");
        if (jobKeys != null && !jobKeys.isEmpty()) {
            result.append(jobKeys.size()).append("/").append(value.getMaxJobsToActivate());
        } else {
            result.append("max: ").append(value.getMaxJobsToActivate());
        }
        if (value.isTruncated()) {
            result.append(" (truncated)");
        }
        if (jobKeys != null && !jobKeys.isEmpty()) {
            for (int i = 0; i < jobKeys.size(); ++i) {
                Long jobKey = (Long)jobKeys.get(i);
                JobRecordValue job = (JobRecordValue)value.getJobs().get(i);
                result.append(StringUtils.rightPad((String)"\n", (int)(8 + this.valueTypeChars))).append(this.summarizeJobRecordValue(jobKey, job));
            }
        }
        return result.toString();
    }

    private String summarizeMessage(Record<?> record) {
        MessageRecordValue value = (MessageRecordValue)record.getValue();
        StringBuilder result = new StringBuilder().append("\"").append(value.getName()).append("\"");
        if (!StringUtils.isEmpty((CharSequence)value.getCorrelationKey())) {
            result.append(" correlationKey: ").append(value.getCorrelationKey());
        }
        result.append(this.summarizeVariables(value.getVariables()));
        return result.toString();
    }

    private String summarizeMessageStartEventSubscription(Record<?> record) {
        MessageStartEventSubscriptionRecordValue value = (MessageStartEventSubscriptionRecordValue)record.getValue();
        return "\"" + value.getMessageName() + "\"" + " starting <process " + this.formatId(value.getBpmnProcessId()) + this.summarizeVariables(value.getVariables());
    }

    private String summarizeMessageSubscription(Record<?> record) {
        MessageSubscriptionRecordValue value = (MessageSubscriptionRecordValue)record.getValue();
        StringBuilder result = new StringBuilder().append("\"").append(value.getMessageName()).append("\" ");
        if (value.isInterrupting()) {
            result.append("(inter.) ");
        }
        if (!StringUtils.isEmpty((CharSequence)value.getCorrelationKey())) {
            result.append("correlationKey: ").append(value.getCorrelationKey()).append(" ");
        }
        result.append("@[").append(this.shortenKey(value.getElementInstanceKey())).append("]").append(this.summarizeProcessInformation(value.getBpmnProcessId(), value.getProcessInstanceKey())).append(this.summarizeVariables(value.getVariables()));
        return result.toString();
    }

    private String summarizeProcessInstance(Record<?> record) {
        ProcessInstanceRecordValue value = (ProcessInstanceRecordValue)record.getValue();
        return value.getBpmnElementType() + " " + this.formatId(value.getElementId()) + this.summarizeProcessInformation(value.getBpmnProcessId(), value.getProcessInstanceKey());
    }

    private String summarizeProcessInstanceCreation(Record<?> record) {
        ProcessInstanceCreationRecordValue value = (ProcessInstanceCreationRecordValue)record.getValue();
        return "new <process " + this.formatId(value.getBpmnProcessId()) + ">" + this.summarizeVariables(value.getVariables());
    }

    private String summarizeProcessInstanceSubscription(Record<?> record) {
        ProcessMessageSubscriptionRecordValue value = (ProcessMessageSubscriptionRecordValue)record.getValue();
        StringBuilder result = new StringBuilder().append("\"").append(value.getMessageName()).append("\" ");
        if (value.isInterrupting()) {
            result.append("(inter.) ");
        }
        if (!StringUtils.isEmpty((CharSequence)value.getCorrelationKey())) {
            result.append("correlationKey: ").append(value.getCorrelationKey()).append(" ");
        }
        result.append("@[").append(this.shortenKey(value.getElementInstanceKey())).append("]").append(this.summarizeProcessInformation(value.getBpmnProcessId(), value.getProcessInstanceKey())).append(this.summarizeVariables(value.getVariables()));
        return result.toString();
    }

    private String summarizeVariable(Record<?> record) {
        VariableRecordValue value = (VariableRecordValue)record.getValue();
        StringBuilder builder = new StringBuilder();
        builder.append(String.format("%s->%s", value.getName(), value.getValue())).append(String.format(" in <process [%s]", this.shortenKey(value.getProcessInstanceKey())));
        if (value.getProcessInstanceKey() != value.getScopeKey()) {
            builder.append(String.format(" at [%s]", this.shortenKey(value.getScopeKey())));
        }
        return builder.append(">").toString();
    }

    private StringBuilder summarizeRejection(Record<?> record) {
        return new StringBuilder().append("!").append(record.getRejectionType()).append(" (").append(StringUtils.abbreviate((String)record.getRejectionReason(), (String)"..", (int)200)).append(")");
    }

    private String summarizeTimer(Record<?> record) {
        TimerRecordValue value = (TimerRecordValue)record.getValue();
        StringBuilder builder = new StringBuilder();
        ZonedDateTime dueTime = Instant.ofEpochMilli(value.getDueDate()).atZone(ZoneId.systemDefault());
        builder.append(this.summarizeElementInformation(value.getTargetElementId(), value.getElementInstanceKey())).append(" ").append(this.summarizeProcessInformation(this.shortenKey(value.getProcessDefinitionKey()), value.getProcessInstanceKey())).append(" due ").append(this.shortenDateTime(dueTime));
        if (value.getRepetitions() > 1) {
            builder.append(value.getRepetitions()).append(" reps");
        }
        return builder.toString();
    }

    private String summarizeError(Record<?> record) {
        ErrorRecordValue value = (ErrorRecordValue)record.getValue();
        return "\"" + value.getExceptionMessage() + "\"" + " " + this.summarizeProcessInformation(null, value.getProcessInstanceKey()) + " (" + StringUtils.abbreviate((String)value.getStacktrace(), (String)"..", (int)100) + ")";
    }

    private String summarizeProcessEvent(Record<?> record) {
        ProcessEventRecordValue value = (ProcessEventRecordValue)record.getValue();
        return this.summarizeElementInformation(value.getTargetElementId(), value.getScopeKey()) + this.summarizeProcessInformation(value.getProcessDefinitionKey(), value.getProcessInstanceKey()) + this.summarizeVariables(value.getVariables());
    }

    private String shortenKey(long input) {
        return this.substitutions.computeIfAbsent(input, this::formatKey);
    }

    private String formatKey(long key) {
        StringBuilder result = new StringBuilder();
        if (!this.singlePartition) {
            if (key > 0L) {
                result.append("P").append(Protocol.decodePartitionId((long)key));
            } else {
                result.append("  ");
            }
        }
        if (key > 0L) {
            result.append("K" + StringUtils.leftPad((String)Long.toString(Protocol.decodeKeyInPartition((long)key)), (int)this.keyDigits, (char)'0'));
        } else {
            result.append(StringUtils.leftPad((String)Long.toString(key), (int)(this.keyDigits + 1), (char)' '));
        }
        return result.toString();
    }

    private String formatPosition(long input) {
        if (input >= 0L) {
            return "#" + StringUtils.leftPad((String)Long.toString(input), (int)this.keyDigits, (char)'0');
        }
        return StringUtils.leftPad((String)Long.toString(input), (int)(this.keyDigits + 1), (char)' ');
    }

    private String formatId(String input) {
        return "\"" + StringUtils.abbreviateMiddle((String)input, (String)"..", (int)16) + "\"";
    }

    private String abbreviate(String input) {
        String result = input;
        for (String longForm : ABBREVIATIONS.keySet()) {
            result = result.replace(longForm, ABBREVIATIONS.get(longForm));
        }
        return result;
    }

    private String shortenDateTime(ZonedDateTime time) {
        ZonedDateTime now = ZonedDateTime.now();
        StringBuilder builder = new StringBuilder();
        if (!now.toLocalDate().isEqual(time.toLocalDate())) {
            builder.append(DateTimeFormatter.ISO_LOCAL_DATE.format(time));
        }
        builder.append("T").append(DateTimeFormatter.ISO_LOCAL_TIME.format(time));
        return builder.toString();
    }
}

