/*
 * Copyright © 2017 camunda services GmbH (info@camunda.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.zeebe.client.impl.command;

import io.grpc.stub.StreamObserver;
import io.zeebe.client.api.ZeebeFuture;
import io.zeebe.client.api.command.CreateWorkflowInstanceCommandStep1;
import io.zeebe.client.api.command.CreateWorkflowInstanceCommandStep1.CreateWorkflowInstanceCommandStep2;
import io.zeebe.client.api.command.CreateWorkflowInstanceCommandStep1.CreateWorkflowInstanceCommandStep3;
import io.zeebe.client.api.command.FinalCommandStep;
import io.zeebe.client.api.response.WorkflowInstanceEvent;
import io.zeebe.client.impl.RetriableClientFutureImpl;
import io.zeebe.client.impl.ZeebeObjectMapper;
import io.zeebe.client.impl.response.CreateWorkflowInstanceResponseImpl;
import io.zeebe.gateway.protocol.GatewayGrpc.GatewayStub;
import io.zeebe.gateway.protocol.GatewayOuterClass;
import io.zeebe.gateway.protocol.GatewayOuterClass.CreateWorkflowInstanceRequest;
import io.zeebe.gateway.protocol.GatewayOuterClass.CreateWorkflowInstanceRequest.Builder;
import java.io.InputStream;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;

public final class CreateWorkflowInstanceCommandImpl
    implements CreateWorkflowInstanceCommandStep1,
        CreateWorkflowInstanceCommandStep2,
        CreateWorkflowInstanceCommandStep3 {

  private final ZeebeObjectMapper objectMapper;
  private final GatewayStub asyncStub;
  private final Builder builder;
  private final Predicate<Throwable> retryPredicate;
  private Duration requestTimeout;

  public CreateWorkflowInstanceCommandImpl(
      final GatewayStub asyncStub,
      final ZeebeObjectMapper objectMapper,
      final Duration requestTimeout,
      final Predicate<Throwable> retryPredicate) {
    this.asyncStub = asyncStub;
    this.objectMapper = objectMapper;
    this.requestTimeout = requestTimeout;
    this.retryPredicate = retryPredicate;

    this.builder = CreateWorkflowInstanceRequest.newBuilder();
  }

  @Override
  public CreateWorkflowInstanceCommandStep3 variables(final InputStream variables) {
    ArgumentUtil.ensureNotNull("variables", variables);
    return setVariables(objectMapper.validateJson("variables", variables));
  }

  @Override
  public CreateWorkflowInstanceCommandStep3 variables(final String variables) {
    ArgumentUtil.ensureNotNull("variables", variables);
    return setVariables(objectMapper.validateJson("variables", variables));
  }

  @Override
  public CreateWorkflowInstanceCommandStep3 variables(final Map<String, Object> variables) {
    return variables((Object) variables);
  }

  @Override
  public CreateWorkflowInstanceCommandStep3 variables(final Object variables) {
    ArgumentUtil.ensureNotNull("variables", variables);
    return setVariables(objectMapper.toJson(variables));
  }

  @Override
  public CreateWorkflowInstanceWithResultCommandStep1 withResult() {
    return new CreateWorkflowInstanceWithResultCommandImpl(
        objectMapper, asyncStub, builder, retryPredicate, requestTimeout);
  }

  @Override
  public CreateWorkflowInstanceCommandStep2 bpmnProcessId(final String id) {
    builder.setBpmnProcessId(id);
    return this;
  }

  @Override
  public CreateWorkflowInstanceCommandStep3 workflowKey(final long workflowKey) {
    builder.setWorkflowKey(workflowKey);
    return this;
  }

  @Override
  public CreateWorkflowInstanceCommandStep3 version(final int version) {
    builder.setVersion(version);
    return this;
  }

  @Override
  public CreateWorkflowInstanceCommandStep3 latestVersion() {
    return version(LATEST_VERSION);
  }

  @Override
  public FinalCommandStep<WorkflowInstanceEvent> requestTimeout(final Duration requestTimeout) {
    this.requestTimeout = requestTimeout;
    return this;
  }

  @Override
  public ZeebeFuture<WorkflowInstanceEvent> send() {
    final CreateWorkflowInstanceRequest request = builder.build();

    final RetriableClientFutureImpl<
            WorkflowInstanceEvent, GatewayOuterClass.CreateWorkflowInstanceResponse>
        future =
            new RetriableClientFutureImpl<>(
                CreateWorkflowInstanceResponseImpl::new,
                retryPredicate,
                streamObserver -> send(request, streamObserver));

    send(request, future);
    return future;
  }

  private void send(
      final CreateWorkflowInstanceRequest request,
      final StreamObserver<GatewayOuterClass.CreateWorkflowInstanceResponse> future) {
    asyncStub
        .withDeadlineAfter(requestTimeout.toMillis(), TimeUnit.MILLISECONDS)
        .createWorkflowInstance(request, future);
  }

  private CreateWorkflowInstanceCommandStep3 setVariables(final String jsonDocument) {
    builder.setVariables(jsonDocument);
    return this;
  }
}
