/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
package org.elasticsearch.search.aggregations.bucket.range;

import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.search.DocValueFormat;
import org.elasticsearch.search.aggregations.InternalAggregations;
import org.elasticsearch.search.aggregations.support.CoreValuesSourceType;
import org.elasticsearch.search.aggregations.support.ValueType;
import org.elasticsearch.search.aggregations.support.ValuesSourceType;

import java.io.IOException;
import java.util.List;
import java.util.Map;

public class InternalGeoDistance extends InternalRange<InternalGeoDistance.Bucket, InternalGeoDistance> {
    public static final Factory FACTORY = new Factory();

    static class Bucket extends InternalRange.Bucket {

        Bucket(
            String key,
            double from,
            double originalFrom,
            double to,
            double originalTo,
            long docCount,
            InternalAggregations aggregations,
            boolean keyed
        ) {
            super(key, from, originalFrom, to, originalTo, docCount, aggregations, keyed, DocValueFormat.RAW);
        }

        @Override
        protected InternalRange.Factory<Bucket, ?> getFactory() {
            return FACTORY;
        }

        boolean keyed() {
            return keyed;
        }
    }

    public static class Factory extends InternalRange.Factory<InternalGeoDistance.Bucket, InternalGeoDistance> {
        @Override
        public ValuesSourceType getValueSourceType() {
            return CoreValuesSourceType.GEOPOINT;
        }

        @Override
        public ValueType getValueType() {
            return ValueType.GEOPOINT;
        }

        @Override
        public InternalGeoDistance create(
            String name,
            List<Bucket> ranges,
            DocValueFormat format,
            boolean keyed,
            Map<String, Object> metadata
        ) {
            return new InternalGeoDistance(name, ranges, keyed, metadata);
        }

        @Override
        public InternalGeoDistance create(List<Bucket> ranges, InternalGeoDistance prototype) {
            return new InternalGeoDistance(prototype.name, ranges, prototype.keyed, prototype.metadata);
        }

        @Override
        public Bucket createBucket(
            String key,
            double from,
            double originalFrom,
            double to,
            double originalTo,
            long docCount,
            InternalAggregations aggregations,
            boolean keyed,
            DocValueFormat format
        ) {
            return new Bucket(key, from, originalFrom, to, originalTo, docCount, aggregations, keyed);
        }

        @Override
        public Bucket createBucket(InternalAggregations aggregations, Bucket prototype) {
            return new Bucket(
                prototype.getKey(),
                ((Number) prototype.getFrom()).doubleValue(),
                ((Number) prototype.getOriginalFrom()).doubleValue(),
                ((Number) prototype.getTo()).doubleValue(),
                ((Number) prototype.getOriginalTo()).doubleValue(),
                prototype.getDocCount(),
                aggregations,
                prototype.getKeyed()
            );
        }
    }

    public InternalGeoDistance(String name, List<Bucket> ranges, boolean keyed, Map<String, Object> metadata) {
        super(name, ranges, DocValueFormat.RAW, keyed, metadata);
    }

    /**
     * Read from a stream.
     */
    public InternalGeoDistance(StreamInput in) throws IOException {
        super(in);
    }

    @Override
    public InternalRange.Factory<Bucket, InternalGeoDistance> getFactory() {
        return FACTORY;
    }

    @Override
    public String getWriteableName() {
        return GeoDistanceAggregationBuilder.NAME;
    }
}
