/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.engine.processing.common;

import io.camunda.zeebe.el.EvaluationContext;
import io.camunda.zeebe.el.EvaluationResult;
import io.camunda.zeebe.el.Expression;
import io.camunda.zeebe.el.ExpressionLanguage;
import io.camunda.zeebe.el.ResultType;
import io.camunda.zeebe.engine.processing.common.Failure;
import io.camunda.zeebe.engine.processing.message.MessageCorrelationKeyException;
import io.camunda.zeebe.model.bpmn.util.time.Interval;
import io.camunda.zeebe.protocol.record.value.ErrorType;
import io.camunda.zeebe.util.Either;
import io.camunda.zeebe.util.EnsureUtil;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.util.List;
import java.util.function.Function;
import org.agrona.DirectBuffer;
import org.agrona.concurrent.UnsafeBuffer;

public final class ExpressionProcessor {
    private static final EvaluationContext EMPTY_EVALUATION_CONTEXT = x -> null;
    private final DirectBuffer resultView = new UnsafeBuffer();
    private final ExpressionLanguage expressionLanguage;
    private final VariableStateEvaluationContext evaluationContext;

    public ExpressionProcessor(ExpressionLanguage expressionLanguage, VariablesLookup lookup) {
        this.expressionLanguage = expressionLanguage;
        this.evaluationContext = new VariableStateEvaluationContext(lookup);
    }

    public Either<Failure, String> evaluateStringExpression(Expression expression, long scopeKey) {
        return this.evaluateExpressionAsEither(expression, scopeKey).flatMap(result -> this.typeCheck((EvaluationResult)result, ResultType.STRING, scopeKey)).map(EvaluationResult::getString);
    }

    public Either<Failure, DirectBuffer> evaluateStringExpressionAsDirectBuffer(Expression expression, long scopeKey) {
        return this.evaluateStringExpression(expression, scopeKey).map(this::wrapResult);
    }

    public Either<Failure, Long> evaluateLongExpression(Expression expression, long scopeKey) {
        return this.evaluateExpressionAsEither(expression, scopeKey).flatMap(result -> this.typeCheck((EvaluationResult)result, ResultType.NUMBER, scopeKey)).map(EvaluationResult::getNumber).map(Number::longValue);
    }

    public Either<Failure, Boolean> evaluateBooleanExpression(Expression expression, long scopeKey) {
        return this.evaluateExpressionAsEither(expression, scopeKey).flatMap(result -> this.typeCheck((EvaluationResult)result, ResultType.BOOLEAN, scopeKey)).map(EvaluationResult::getBoolean);
    }

    public Either<Failure, Interval> evaluateIntervalExpression(Expression expression, long scopeKey) {
        EvaluationResult result = this.evaluateExpression(expression, scopeKey);
        if (result.isFailure()) {
            return Either.left((Object)new Failure(result.getFailureMessage()));
        }
        switch (result.getType()) {
            case DURATION: {
                return Either.right((Object)new Interval(result.getDuration()));
            }
            case PERIOD: {
                return Either.right((Object)new Interval(result.getPeriod()));
            }
            case STRING: {
                try {
                    return Either.right((Object)Interval.parse((String)result.getString()));
                }
                catch (DateTimeParseException e) {
                    return Either.left((Object)new Failure(String.format("Invalid duration format '%s' for expression '%s'", result.getString(), expression.getExpression())));
                }
            }
        }
        List<ResultType> expected = List.of(ResultType.DURATION, ResultType.PERIOD, ResultType.STRING);
        return Either.left((Object)new Failure(String.format("Expected result of the expression '%s' to be one of '%s', but was '%s'", expression.getExpression(), expected, result.getType())));
    }

    public Either<Failure, ZonedDateTime> evaluateDateTimeExpression(Expression expression, Long scopeKey) {
        EvaluationResult result = this.evaluateExpression(expression, scopeKey);
        if (result.isFailure()) {
            return Either.left((Object)new Failure(result.getFailureMessage()));
        }
        if (result.getType() == ResultType.DATE_TIME) {
            return Either.right((Object)result.getDateTime());
        }
        if (result.getType() == ResultType.STRING) {
            try {
                return Either.right((Object)ZonedDateTime.parse(result.getString()));
            }
            catch (DateTimeParseException e) {
                return Either.left((Object)new Failure(String.format("Invalid date-time format '%s' for expression '%s'", result.getString(), expression.getExpression())));
            }
        }
        List<ResultType> expected = List.of(ResultType.DATE_TIME, ResultType.STRING);
        return Either.left((Object)new Failure(String.format("Expected result of the expression '%s' to be one of '%s', but was '%s'", expression.getExpression(), expected, result.getType())));
    }

    public Either<Failure, DirectBuffer> evaluateAnyExpression(Expression expression, long scopeKey) {
        Either<Failure, EvaluationResult> evaluationResult = this.evaluateExpressionAsEither(expression, scopeKey);
        return evaluationResult.map(EvaluationResult::toBuffer);
    }

    public Either<Failure, List<DirectBuffer>> evaluateArrayExpression(Expression expression, long scopeKey) {
        Either<Failure, EvaluationResult> evaluationResult = this.evaluateExpressionAsEither(expression, scopeKey);
        return evaluationResult.flatMap(result -> this.typeCheck((EvaluationResult)result, ResultType.ARRAY, scopeKey)).map(EvaluationResult::getList);
    }

    public String evaluateMessageCorrelationKeyExpression(Expression expression, long scopeKey) {
        EvaluationResult evaluationResult = this.evaluateExpression(expression, scopeKey);
        CorrelationKeyResultHandler resultHandler = new CorrelationKeyResultHandler(scopeKey);
        return resultHandler.apply(evaluationResult);
    }

    public Either<Failure, DirectBuffer> evaluateVariableMappingExpression(Expression expression, long scopeKey) {
        return this.evaluateExpressionAsEither(expression, scopeKey).flatMap(result -> this.typeCheck((EvaluationResult)result, ResultType.OBJECT, scopeKey)).mapLeft(failure -> new Failure(failure.getMessage(), ErrorType.IO_MAPPING_ERROR, scopeKey)).map(EvaluationResult::toBuffer);
    }

    private Either<Failure, EvaluationResult> typeCheck(EvaluationResult result, ResultType expectedResultType, long scopeKey) {
        if (result.getType() != expectedResultType) {
            return Either.left((Object)new Failure(String.format("Expected result of the expression '%s' to be '%s', but was '%s'.", result.getExpression(), expectedResultType, result.getType()), ErrorType.EXTRACT_VALUE_ERROR, scopeKey));
        }
        return Either.right((Object)result);
    }

    private EvaluationResult evaluateExpression(Expression expression, long variableScopeKey) {
        EvaluationContext context;
        if (variableScopeKey < 0L) {
            context = EMPTY_EVALUATION_CONTEXT;
        } else {
            this.evaluationContext.variableScopeKey = variableScopeKey;
            context = this.evaluationContext;
        }
        return this.expressionLanguage.evaluateExpression(expression, context);
    }

    private Either<Failure, EvaluationResult> evaluateExpressionAsEither(Expression expression, long variableScopeKey) {
        EvaluationResult result = this.evaluateExpression(expression, variableScopeKey);
        return result.isFailure() ? Either.left((Object)new Failure(result.getFailureMessage(), ErrorType.EXTRACT_VALUE_ERROR, variableScopeKey)) : Either.right((Object)result);
    }

    private DirectBuffer wrapResult(String result) {
        this.resultView.wrap(result.getBytes());
        return this.resultView;
    }

    @FunctionalInterface
    public static interface VariablesLookup {
        public DirectBuffer getVariable(long var1, DirectBuffer var3);
    }

    private static class VariableStateEvaluationContext
    implements EvaluationContext {
        private final DirectBuffer variableNameBuffer = new UnsafeBuffer();
        private final VariablesLookup lookup;
        private long variableScopeKey;

        public VariableStateEvaluationContext(VariablesLookup lookup) {
            this.lookup = lookup;
        }

        public DirectBuffer getVariable(String variableName) {
            EnsureUtil.ensureGreaterThan((String)"variable scope key", (long)this.variableScopeKey, (long)0L);
            this.variableNameBuffer.wrap(variableName.getBytes());
            return this.lookup.getVariable(this.variableScopeKey, this.variableNameBuffer);
        }
    }

    protected static final class CorrelationKeyResultHandler
    implements Function<EvaluationResult, String> {
        private final long variableScopeKey;

        protected CorrelationKeyResultHandler(long variableScopeKey) {
            this.variableScopeKey = variableScopeKey;
        }

        @Override
        public String apply(EvaluationResult evaluationResult) {
            if (evaluationResult.isFailure()) {
                throw new MessageCorrelationKeyException(this.variableScopeKey, evaluationResult.getFailureMessage());
            }
            if (evaluationResult.getType() == ResultType.STRING) {
                return evaluationResult.getString();
            }
            if (evaluationResult.getType() == ResultType.NUMBER) {
                Number correlationKeyNumber = evaluationResult.getNumber();
                return Long.toString(correlationKeyNumber.longValue());
            }
            String failureMessage = String.format("Failed to extract the correlation key for '%s': The value must be either a string or a number, but was %s.", evaluationResult.getExpression(), evaluationResult.getType().toString());
            throw new MessageCorrelationKeyException(this.variableScopeKey, failureMessage);
        }
    }

    public static final class EvaluationException
    extends RuntimeException {
        public EvaluationException(String message) {
            super(message);
        }
    }
}

