/*
 * Decompiled with CFR 0.152.
 */
package org.jamel.dbf;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.charset.Charset;
import java.util.Date;
import java.util.GregorianCalendar;
import org.jamel.dbf.exception.DbfException;
import org.jamel.dbf.structure.DbfField;
import org.jamel.dbf.structure.DbfHeader;
import org.jamel.dbf.structure.DbfRow;
import org.jamel.dbf.utils.DbfUtils;

public class DbfReader
implements Closeable {
    protected final byte DATA_ENDED = (byte)26;
    protected final byte DATA_DELETED = (byte)42;
    private Charset charset = Charset.defaultCharset();
    private DataInput dataInput;
    private final DbfHeader header;

    public DbfReader(File file) throws DbfException {
        try {
            this.dataInput = new RandomAccessFile(file, "r");
            this.header = DbfHeader.read(this.dataInput);
            this.skipToDataBeginning();
        }
        catch (IOException e) {
            throw new DbfException("Cannot open Dbf file " + file, e);
        }
    }

    public DbfReader(File file, Charset charset) throws DbfException {
        this(file);
        this.charset = charset;
    }

    public DbfReader(InputStream in) throws DbfException {
        try {
            this.dataInput = new DataInputStream(new BufferedInputStream(in));
            this.header = DbfHeader.read(this.dataInput);
            this.skipToDataBeginning();
        }
        catch (IOException e) {
            throw new DbfException("Cannot read Dbf", e);
        }
    }

    public DbfReader(InputStream in, Charset charset) throws DbfException {
        this(in);
        this.charset = charset;
    }

    private void skipToDataBeginning() throws IOException {
        int dataStartIndex = this.header.getHeaderLength() - 32 * (this.header.getFieldsCount() + 1) - 1;
        if (dataStartIndex > 0) {
            this.dataInput.skipBytes(dataStartIndex);
        }
    }

    public boolean canSeek() {
        return this.dataInput instanceof RandomAccessFile;
    }

    public void seekToRecord(int n) {
        if (!this.canSeek()) {
            throw new DbfException("Seeking is not supported.");
        }
        if (n < 0 || n >= this.header.getNumberOfRecords()) {
            throw new DbfException(String.format("Record index out of range [0, %d]: %d", this.header.getNumberOfRecords(), n));
        }
        long position = this.header.getHeaderLength() + n * this.header.getRecordLength();
        try {
            ((RandomAccessFile)this.dataInput).seek(position);
        }
        catch (IOException e) {
            throw new DbfException(String.format("Failed to seek to record %d of %d", n, this.header.getNumberOfRecords()), e);
        }
    }

    public DbfRow nextRow() {
        Object[] record = this.nextRecord();
        return record == null ? null : new DbfRow(this.header, this.charset, record);
    }

    public Object[] nextRecord() {
        try {
            byte nextByte;
            do {
                if ((nextByte = this.dataInput.readByte()) == 26) {
                    return null;
                }
                if (nextByte != 42) continue;
                this.dataInput.skipBytes(this.header.getRecordLength() - 1);
            } while (nextByte == 42);
            Object[] recordObjects = new Object[this.header.getFieldsCount()];
            for (int i = 0; i < this.header.getFieldsCount(); ++i) {
                recordObjects[i] = this.readFieldValue(this.header.getField(i));
            }
            return recordObjects;
        }
        catch (EOFException e) {
            return null;
        }
        catch (IOException e) {
            throw new DbfException("Cannot read next record form Dbf file", e);
        }
    }

    private Object readFieldValue(DbfField field) throws IOException {
        byte[] buf = new byte[field.getFieldLength()];
        this.dataInput.readFully(buf);
        switch (field.getDataType()) {
            case CHAR: {
                return this.readCharacterValue(field, buf);
            }
            case DATE: {
                return this.readDateValue(field, buf);
            }
            case FLOAT: {
                return this.readFloatValue(field, buf);
            }
            case LOGICAL: {
                return this.readLogicalValue(field, buf);
            }
            case NUMERIC: {
                return this.readNumericValue(field, buf);
            }
            case MEMO: {
                return this.readMemoLink(field, buf);
            }
        }
        return null;
    }

    protected Object readCharacterValue(DbfField field, byte[] buf) throws IOException {
        return buf;
    }

    protected Date readDateValue(DbfField field, byte[] buf) throws IOException {
        int year = DbfUtils.parseInt(buf, 0, 4);
        int month = DbfUtils.parseInt(buf, 4, 6);
        int day = DbfUtils.parseInt(buf, 6, 8);
        return new GregorianCalendar(year, month - 1, day).getTime();
    }

    protected Float readFloatValue(DbfField field, byte[] buf) throws IOException {
        try {
            byte[] floatBuf = DbfUtils.trimLeftSpaces(buf);
            boolean processable = floatBuf.length > 0 && !DbfUtils.contains(floatBuf, (byte)63);
            return processable ? Float.valueOf(new String(floatBuf)) : null;
        }
        catch (NumberFormatException e) {
            throw new DbfException("Failed to parse Float from " + field.getName(), e);
        }
    }

    protected Boolean readLogicalValue(DbfField field, byte[] buf) throws IOException {
        boolean isTrue = buf[0] == 89 || buf[0] == 121 || buf[0] == 84 || buf[0] == 116;
        return isTrue ? Boolean.TRUE : Boolean.FALSE;
    }

    protected Number readNumericValue(DbfField field, byte[] buf) throws IOException {
        try {
            byte[] numericBuf = DbfUtils.trimLeftSpaces(buf);
            boolean processable = numericBuf.length > 0 && !DbfUtils.contains(numericBuf, (byte)63);
            return processable ? Double.valueOf(new String(numericBuf)) : null;
        }
        catch (NumberFormatException e) {
            throw new DbfException("Failed to parse Number from " + field.getName(), e);
        }
    }

    protected Number readMemoLink(DbfField field, byte[] buf) throws IOException {
        switch (field.getFieldLength()) {
            case 4: {
                return DbfUtils.readLittleEndianInt(new DataInputStream(new ByteArrayInputStream(buf)));
            }
            case 10: {
                return this.readNumericValue(field, buf);
            }
        }
        throw new DbfException("Unknown MEMO mode: " + field.getFieldLength());
    }

    public int getRecordCount() {
        return this.header.getNumberOfRecords();
    }

    public DbfHeader getHeader() {
        return this.header;
    }

    @Override
    public void close() {
        try {
            if (this.dataInput instanceof Closeable) {
                ((Closeable)((Object)this.dataInput)).close();
                this.dataInput = null;
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }
}

