package com.efuture.ocp.common.callnumber;

import cn.hutool.core.thread.ExecutorBuilder;
import com.efuture.common.utils.ServiceLogs;
import com.efuture.common.utils.StringUtils;
import com.efuture.ocp.common.exception.ServiceException;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.function.Consumer;

/**
 * 叫号执行器
 * submit ：提交一个任务到线程池，根据maxWorker 创建线程并按maxWorker去获取号码并执行
 * call:直接获取一个号并执行一次任务
 */
public class CallNumberExecutor
{

    CallNumberHandle handle;
    /**
     * 一个叫号器，一个线程池
     */
    ConcurrentMap<String, ExecutorService> executors = new ConcurrentHashMap<>();

    ConcurrentMap<String, List<CallNumberRuner>> workers = new ConcurrentHashMap<>();

    public CallNumberExecutor(CallNumberHandle handle)
    {
        this.handle = handle;
    }

    public void submit(String bizName, int maxNum, int maxWorker, Consumer<NumberInfo> worker)
    {
        ExecutorService executorService = executors.computeIfAbsent(bizName,
        (key) -> {
            return (ExecutorService) ExecutorBuilder.create().setMaxPoolSize(maxWorker).setCorePoolSize(0).build();
        });
        List<CallNumberRuner> workList = workers.computeIfAbsent(bizName,
        (key) -> {
            List<CallNumberRuner> runers = new ArrayList<>();

            for (int i = 0; i < maxWorker; i++)
            {
                CallNumberRuner run = new CallNumberRuner(key, maxNum, i, worker);
                runers.add(run);
            }
            return runers;
        });

        if (workList.size() < maxWorker) {
            for (int i = workList.size() - 1; i < maxWorker; i++) {
                CallNumberRuner run = new CallNumberRuner(bizName, maxNum, i, worker);
                workList.add(run);
            }
        }

        for (CallNumberRuner runer : workList) {
            executorService.submit(runer);
        }
    }

    public boolean watchPrint(String bizName)
    {
        List<CallNumberRuner> workList = workers.get(bizName);
        boolean ibRun = false;

        for (CallNumberRuner runer : workList) {
            ibRun = runer.isRunning();
            ServiceLogs.truedebuglog("watchPrint", runer.watch(), 0);
        }

        return ibRun;
    }

    public List<String> status(String bizName)
    {
        if (StringUtils.isEmpty(bizName)) {
            throw ServiceException.newGroupServiceException(ServiceException.ERR_GROUP.UNKNOWN, "50002", "任务名称不能为空");
        }

        List<String> rtn = new ArrayList<>();
        List<CallNumberRuner> workList = workers.get(bizName);

        if (workList == null) {
            return null;
        }

        for (CallNumberRuner runer : workList) {
            rtn.add(runer.watch());
        }

        return rtn;
    }

    public String stop(String bizName)
    {
        if (StringUtils.isEmpty(bizName)) {
            throw ServiceException.newGroupServiceException(ServiceException.ERR_GROUP.UNKNOWN, "50002", "任务名称不能为空");
        }

        String msg = "任务[" + bizName + "]";
        List<CallNumberRuner> workList = workers.get(bizName);

        if (workList != null) {
            for (CallNumberRuner runer : workList) {
                runer.stop();
            }
        }

        ExecutorService executorService = executors.get(bizName);

        if (executorService == null) {
            return msg + "未提交";
        }

        if (executorService.isShutdown()) {
            return msg + "已经停止或未开始";
        }

        List<Runnable> runnableList = executorService.shutdownNow();
        msg = msg + "停止成功";

        if (runnableList != null && runnableList.size() > 0) {
            return msg + ",还有[" + runnableList.size() + "]条任务未完成";
        }
        else {
            return msg;
        }
    }

    public CallNumberRuner call(String bizName, int maxNumer, Consumer<NumberInfo> worker)
    {
        CallNumberRuner run = new CallNumberRuner(bizName, maxNumer, 0, worker);
        run.run();
        return run;
    }

    /**
     * 获取一个号码
     *
     * @param bizName
     * @param maxNum
     * @param waitSecond
     * @return
     */
    public NumberInfo getNumber(String bizName, int maxNum, int waitSecond)
    {
        NumberInfo num = new NumberInfo(bizName, maxNum, -99);
        int rtn = handle.next(num, waitSecond);

        if (rtn < 0) {
            return null;
        }
        else {
            return num;
        }
    }

    /**
     * 获取下一个号码
     *
     * @param num
     * @param waitSecond
     * @return
     */
    public int getNextNumber(NumberInfo num, int waitSecond)
    {
        if (num.isRunning()) {
            handle.complete(num);
        }

        return handle.next(num, waitSecond);
    }

    public void destroy()
    {
        for (ExecutorService service : executors.values()) {
            service.shutdown();
        }
    }

    private class CallNumberRuner implements Runnable
    {
        public String curJobKey = "";
        private NumberInfo number;
        private Consumer<NumberInfo> worker;
        private String lineSeparator = System.lineSeparator();
        private int flag = 0;

        public CallNumberRuner(NumberInfo number, Consumer<NumberInfo> worker)
        {
            this.number = number;
            this.worker = worker;
        }

        public CallNumberRuner(String bizName, int maxNum, int start, Consumer<NumberInfo> worker)
        {
            this.number = new NumberInfo(bizName, maxNum, start);
            this.worker = worker;
        }

        public NumberInfo getNumber()
        {
            return number;
        }

        public Consumer<NumberInfo> getWorker()
        {
            return worker;
        }

        public String watch()
        {
            StringBuilder sb = new StringBuilder();
            sb.append(lineSeparator);
            sb.append("number:" + number.toString());
            sb.append(lineSeparator);
            sb.append("isRuning:" + number.isRunning());
            return sb.toString();
        }

        public boolean isRunning()
        {
            return number.isRunning();
        }

        public int getFlag()
        {
            return flag;
        }

        public void stop()
        {
            this.flag = 1;
        }

        @Override
        public void run()
        {
            curJobKey = Thread.currentThread().getName() + "-" + this.number.toString();
            ServiceLogs.setCurLogKey(curJobKey);
            ServiceLogs.info("CallNumberRuner", "开始运行[{0}]", 0, this.number.toString());

            if (this.number.isRunning()) {
                ServiceLogs.info("CallNumberRuner", "当前序号[{0}]正在执行,不继续执行", 0, this.number.toString());
                return;
            }

            //循环叫号处理，只要叫不到号了，就直接循环
            int num = -99;
            Random r = new Random(Math.abs(this.number.start));
            int randomAwait = r.nextInt(100) + 1;

            do {
                try {
                    Thread.sleep(randomAwait);
                    num = handle.next(this.number, 10);

                    if (num >= 0) {
                        ServiceLogs.info("CallNumberRuner", "开始执行[{0}]", 0, this.number.toString());
                        this.worker.accept(this.number);
                        //执行完后，随机等待一段时间
                    }
                    else {
                        ServiceLogs.info("CallNumberRuner", "[{0}]叫号返回值[{1}]小于0,准备退出", 0, this.number.toString(), num);
                    }
                }
                catch (InterruptedException e1) {
                    ServiceLogs.errLog("CallNumberRuner", null, "执行[{0}]被强制终止", this.number.toString());
                    return;
                }
                catch (Exception e) {
                    ServiceLogs.errLog("CallNumberRuner", e, "执行[{0}]发生错误:[{1}]", this.number.toString(), e.getMessage());
                }
                finally {
                    handle.complete(this.number);

                    if (Thread.currentThread().isInterrupted()) {
                        ServiceLogs.errLog("CallNumberRuner", null, "执行[{0}]已被强制终止", this.number.toString());
                        return;
                    }
                }

                if (flag != 0) {
                    ServiceLogs.info("CallNumberRuner", "[{0}]:标志不为0,准备终止", 0, this.number.toString());
                    return;
                }
            } while (num >= 0);
        }
    }
}
