package com.efuture.ocp.common.cache.config;


import com.jarvis.cache.ICacheManager;
import com.jarvis.cache.autoconfigure.AutoloadCacheProperties;
import com.jarvis.cache.redis.AbstractRedisCacheManager;
import com.jarvis.cache.redis.LettuceRedisClusterCacheManager;
import com.jarvis.cache.redis.SpringRedisCacheManager;
import com.jarvis.cache.serializer.ISerializer;
import io.lettuce.core.cluster.RedisClusterClient;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.RedisPassword;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.lettuce.LettuceClusterConnection;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.connection.lettuce.LettucePoolingClientConfiguration;
import org.springframework.data.redis.core.RedisConnectionUtils;

import java.lang.reflect.Field;
import java.time.Duration;

@Order(1)
@Configuration
@EnableConfigurationProperties({RedisProperty.class})
@ConditionalOnClass(LettuceConnectionFactory.class)
public class LettuceCacheCacheManagerConfiguration
{

    Logger log = LoggerFactory.getLogger(LettuceCacheCacheManagerConfiguration.class);

    @Bean
    public RedisConnectionFactory getConnectionFactory(RedisProperty redisProperty)
    {
        //单机模式
        RedisStandaloneConfiguration configuration = new RedisStandaloneConfiguration();
        configuration.setHostName(redisProperty.getHost());
        configuration.setPort(redisProperty.getPort());
        configuration.setDatabase(redisProperty.getDatabase());
        configuration.setPassword(RedisPassword.of(redisProperty.getPassword()));
        //哨兵模式
        //RedisSentinelConfiguration configuration1 = new RedisSentinelConfiguration();
        //集群模式
        //RedisClusterConfiguration configuration2 = new RedisClusterConfiguration();
        LettuceConnectionFactory factory = new LettuceConnectionFactory(configuration, getPoolConfig(redisProperty));
        //factory.setShareNativeConnection(false);//是否允许多个线程操作共用同一个缓存连接，默认true，false时每个操作都将开辟新的连接
        return factory;
    }

    /**
     * 获取缓存连接池
     *
     * @return
     */
    @Bean
    public LettucePoolingClientConfiguration getPoolConfig(RedisProperty redisProperty)
    {
        GenericObjectPoolConfig config = new GenericObjectPoolConfig();
        config.setMaxTotal(redisProperty.getMaxTotal());
        config.setMaxWaitMillis(redisProperty.getMaxWait());
        config.setMaxIdle(redisProperty.getMaxIdle());
        config.setMinIdle(redisProperty.getMinIdle());
        LettucePoolingClientConfiguration pool = LettucePoolingClientConfiguration.builder()
                .poolConfig(config)
                .commandTimeout(Duration.ofMillis(redisProperty.getTimeout()))
                .shutdownTimeout(Duration.ofMillis(redisProperty.getShutdown()))
                .build();
        return pool;
    }

    /**
     * 默认只支持{@link LettuceRedisClusterCacheManager}<br>
     *
     * @param config
     * @param serializer
     * @param connectionFactory
     * @return
     */
    @Bean
    @ConditionalOnMissingBean(ICacheManager.class)
    @ConditionalOnBean(RedisConnectionFactory.class)
    public ICacheManager LettuceCacheCacheManager(AutoloadCacheProperties config, ISerializer<Object> serializer,
            RedisConnectionFactory connectionFactory)
    {
        return createRedisCacheManager(config, serializer, (LettuceConnectionFactory) connectionFactory);
    }

    private ICacheManager createRedisCacheManager(AutoloadCacheProperties config, ISerializer<Object> serializer, LettuceConnectionFactory connectionFactory)
    {
        RedisConnection redisConnection = null;

        try {
            redisConnection = connectionFactory.getConnection();
            AbstractRedisCacheManager cacheManager = null;

            if (redisConnection instanceof LettuceClusterConnection) {
                LettuceClusterConnection lettuceClusterConnection = (LettuceClusterConnection) redisConnection;

                try {
                    Field clusterClientField = LettuceClusterConnection.class.getDeclaredField("clusterClient");
                    clusterClientField.setAccessible(true);
                    RedisClusterClient redisClusterClient = (RedisClusterClient) clusterClientField.get(lettuceClusterConnection);
                    cacheManager = new LettuceRedisClusterCacheManager(redisClusterClient, serializer);
                }
                catch (Exception e) {
                    log.error(e.getMessage(), e);
                }
            }
            else {
                cacheManager = new SpringRedisCacheManager(connectionFactory, serializer);
            }

            // 根据需要自行配置
            //cacheManager.setHashExpire(config.getJedis().getHashExpire());
            return cacheManager;
        }
        catch (Throwable e) {
            log.error(e.getMessage(), e);
            throw e;
        }
        finally {
            RedisConnectionUtils.releaseConnection(redisConnection, connectionFactory, false);
        }
    }
}
