package com.efuture.ocp.common.distributedLock;

import cn.hutool.extra.spring.SpringUtil;
import com.efuture.common.utils.ServiceLogs;
import com.efuture.ocp.common.cache.config.CacheConfig;

import java.util.HashMap;

public abstract class AbstractDistributedLockHandle implements DistributedLockHandle
{

    private final ThreadLocal<HashMap<String, DLockInfo>> LocalTrans = new ThreadLocal<HashMap<String, DLockInfo>>();
    LockWatchdog dog;

    protected DLockInfo newTransId(String key, int lockExpire)
    {
        return new DLockInfo(key, lockExpire * 1000);
    }

    protected DLockInfo getThisTransId(String key)
    {
        HashMap<String, DLockInfo> trans = LocalTrans.get();

        if (trans == null) {
            return null;
        }

        return trans.get(key);
    }

    protected void setThisTransId(String key, DLockInfo id)
    {
        HashMap<String, DLockInfo> trans = LocalTrans.get();

        if (trans == null) {
            trans = new HashMap<String, DLockInfo>();
            LocalTrans.set(trans);
        }

        trans.put(key, id);
    }

    public void addWatchdog(DLockInfo lockInfo)
    {
        if (dog == null) {
            dog = SpringUtil.getBean(LockWatchdog.class);
        }

        if (dog == null) {
            return;
        }

        dog.add(lockInfo);
    }

    public void removeWatchdog(DLockInfo lockInfo)
    {
        if (dog == null) {
            dog = SpringUtil.getBean(LockWatchdog.class);
        }

        if (dog == null) {
            return;
        }

        dog.remove(lockInfo);
    }

    /**
     * 1.调用子类的缓存实现把key和随机生成的key放入缓存 2.如果成功则放入线程对象，等待解锁时判断
     */
    @Override
    public DLockInfo tryLock(String key, int lockExpire, int waitMillSeconds)
    {
        //线程变量中已经存在，表示当前线程已经加过锁，不用再加
        DLockInfo lockInfo = getThisTransId(key);

        if (lockInfo != null) {
            return lockInfo;
        }

        lockInfo = newTransId(key, lockExpire);
        boolean ok = doNaiveCachePut(key, lockInfo, lockExpire, waitMillSeconds);

        if (!ok) {
            return null;
        }

        setThisTransId(key, lockInfo);
        addWatchdog(lockInfo);
        return lockInfo;
    }


    public abstract boolean doNaiveCachePut(String key, DLockInfo lockInfo, int lockExpire, int waitMillSeconds);

    public abstract boolean doNaiveCacheRenewal(DLockInfo lockInfo);

    public void renewal(DLockInfo lockInfo)
    {
        String key = lockInfo.getKey();

        try {
            doNaiveCacheRenewal(lockInfo);
        }
        catch (Exception e) {
            ServiceLogs.error("DistributedLock", e, "延期时失败,key[{0}]", 0, key);
        }
    }

    /**
     * 1.先检查当前进程有没有缓存成功的数据 2.如果有，调用子类的缓存实现，获取缓存的ID 3.如果分布式缓存ID=进程缓存的ID，则表示配对成功
     * 4.配对成功，则调用子类的缓存实现，删除缓存 5.配对不成功，则应该只有如下这一种情况，这种情况下，线程A的处理 都已经完成了，可以不做任何处理
     * <p>
     * -- 1.线程A加锁成功 -- 2.线程A的锁到期,没有自动延期机制,缓存到期清除 -- 3.线程B加锁成功 --
     * 4.线程A执行完，开始解锁，发现缓存的ID不是开始加锁的ID
     */
    @Override
    public void unlock(String key, int lockExpire)
    {
        DLockInfo lockInfo = getThisTransId(key);

        if (lockInfo == null) {
            //本地线程没有，就不应该解锁key
            ServiceLogs.debug("DistributedLock", "解锁时key[" + key + "]的线程变量数据不存在,不执行解锁", 0);
            return;
        }

        unlock(lockInfo);
    }

    @Override
    public void unlock(String key)
    {
        unlock(key, CacheConfig.CacheTimeOut.Min);
    }

    @Override
    public void unlock(DLockInfo lockInfo)
    {
        String key = lockInfo.getKey();

        try {
            //value = 0 缓存中没有数据
            long value = doNaiveCacheGet(key);
            long id = lockInfo.getLockKey();

            if (id == value && value != 0) {
                doNaiveCacheDelete(key);
            }
            else {
                ServiceLogs.debug("DistributedLock", "解锁时key[" + key + "]的缓存ID不匹配,不做处理,[" + id + "]-[" + value + "]", 0);
            }
        }
        catch (Exception e) {
            ServiceLogs.error("DistributedLock", e, "解锁是失败,key[{0}]", 0, key);
        }
        finally {
            clearThisTransId(key);
            removeWatchdog(lockInfo);
        }
    }

    protected void clearThisTransId(String key)
    {
        HashMap<String, DLockInfo> trans = LocalTrans.get();

        if (null != trans) {
            trans.remove(key);

            if (trans.isEmpty()) {
                LocalTrans.remove();
            }
        }
    }

    public abstract void doNaiveCacheDelete(String key);

    public abstract long doNaiveCacheGet(String key);

}
