package com.bstek.ureport.console.designer;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.antlr.v4.runtime.ANTLRInputStream;
import org.antlr.v4.runtime.CommonTokenStream;
import org.apache.commons.io.IOUtils;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;

import com.bstek.ureport.cache.CacheUtils;
import com.bstek.ureport.console.RenderPageServletAction;
import com.bstek.ureport.console.cache.TempObjectCache;
import com.bstek.ureport.console.exception.ReportDesignException;
import com.bstek.ureport.definition.ReportDefinition;
import com.bstek.ureport.dsl.ReportParserLexer;
import com.bstek.ureport.dsl.ReportParserParser;
import com.bstek.ureport.dsl.ReportParserParser.DatasetContext;
import com.bstek.ureport.export.ReportRender;
import com.bstek.ureport.expression.ErrorInfo;
import com.bstek.ureport.expression.ScriptErrorListener;
import com.bstek.ureport.parser.ReportParser;
import com.bstek.ureport.provider.report.ReportProvider;

/**
 * 石基设计页面对应的action
 *
 * @author 吴德
 */
public class ShijiDesignerServletAction extends RenderPageServletAction {
	private ReportRender reportRender;
	private ReportParser reportParser;
	private List<ReportProvider> reportProviders=new ArrayList<ReportProvider>();

	/**
	 * 执行
	 *
	 * @param req  请求·
	 * @param resp 响应
	 * @throws ServletException servlet异常
	 * @throws IOException      ioexception
	 */
	@Override
	public void execute(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
		String method=retriveMethod(req);
		if(method!=null){
			invokeMethod(method, req, resp);
		}else{
			VelocityContext context = new VelocityContext();
			context.put("contextPath", req.getContextPath());
			resp.setContentType("text/html");
			resp.setCharacterEncoding("utf-8");
			Template template=ve.getTemplate("ureport-html/designer.html","utf-8");
			PrintWriter writer=resp.getWriter();
			template.merge(context, writer);
			writer.close();
		}
	}

	/**
	 * 脚本验证
	 *
	 * @param req  请求
	 * @param resp 响应
	 * @throws ServletException servlet异常
	 * @throws IOException      ioexception
	 */
	public void scriptValidation(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
		String content=req.getParameter("content");
		ANTLRInputStream antlrInputStream=new ANTLRInputStream(content);
		ReportParserLexer lexer=new ReportParserLexer(antlrInputStream);
		CommonTokenStream tokenStream=new CommonTokenStream(lexer);
		ReportParserParser parser=new ReportParserParser(tokenStream);
		ScriptErrorListener errorListener=new ScriptErrorListener();
		parser.removeErrorListeners();
		parser.addErrorListener(errorListener);
		parser.expression();
		List<ErrorInfo> infos=errorListener.getInfos();
		writeObjectToJson(resp, infos);
	}

	/**
	 * 条件脚本验证
	 *
	 * @param req  请求
	 * @param resp 响应
	 * @throws ServletException servlet异常
	 * @throws IOException      ioexception
	 */
	public void conditionScriptValidation(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
		String content=req.getParameter("content");
		ANTLRInputStream antlrInputStream=new ANTLRInputStream(content);
		ReportParserLexer lexer=new ReportParserLexer(antlrInputStream);
		CommonTokenStream tokenStream=new CommonTokenStream(lexer);
		ReportParserParser parser=new ReportParserParser(tokenStream);
		ScriptErrorListener errorListener=new ScriptErrorListener();
		parser.removeErrorListeners();
		parser.addErrorListener(errorListener);
		parser.expr();
		List<ErrorInfo> infos=errorListener.getInfos();
		writeObjectToJson(resp, infos);
	}


	/**
	 * 解析数据集名称
	 *
	 * @param req  请求
	 * @param resp 响应
	 * @throws ServletException servlet异常
	 * @throws IOException      ioexception
	 */
	public void parseDatasetName(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
		String expr=req.getParameter("expr");
		ANTLRInputStream antlrInputStream=new ANTLRInputStream(expr);
		ReportParserLexer lexer=new ReportParserLexer(antlrInputStream);
		CommonTokenStream tokenStream=new CommonTokenStream(lexer);
		ReportParserParser parser=new ReportParserParser(tokenStream);
		parser.removeErrorListeners();
		DatasetContext ctx=parser.dataset();
		String datasetName=ctx.Identifier().getText();
		Map<String,String> result=new HashMap<String,String>(2);
		result.put("datasetName", datasetName);
		writeObjectToJson(resp, result);
	}

	/**
	 * 保存预览数据
	 *
	 * @param req  请求
	 * @param resp 响应
	 * @throws ServletException servlet异常
	 * @throws IOException      ioexception
	 */
	public void savePreviewData(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
		String content=req.getParameter("content");
		content=decodeContent(content);
		String metadata = req.getParameter("metadata");
		metadata=decodeContent(metadata);
		InputStream inputStream=IOUtils.toInputStream(content,"utf-8");

		ReportDefinition reportDef=reportParser.parse(inputStream,"p",metadata);
		reportRender.rebuildReportDefinition(reportDef);
		IOUtils.closeQuietly(inputStream);
		TempObjectCache.putObject(PREVIEW_KEY, reportDef);
	}

	/**
	 * 负载报告
	 *
	 * @param req  请求
	 * @param resp 响应
	 * @throws ServletException servlet异常
	 * @throws IOException      ioexception
	 */
	public void loadReport(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
		String file=req.getParameter("file");
		if(file==null){
			throw new ReportDesignException("Report file can not be null.");
		}
		file=ReportUtils.decodeFileName(file);
		Object obj=TempObjectCache.getObject(file);
		if(obj!=null && obj instanceof ReportDefinition){
			ReportDefinition reportDef=(ReportDefinition)obj;
			TempObjectCache.removeObject(file);
			writeObjectToJson(resp, new ReportDefinitionWrapper(encodeMeta(reportDef)));
		}else{
			ReportDefinition reportDef=reportRender.parseReportMeta(file);
			writeObjectToJson(resp, new ReportDefinitionWrapper(encodeMeta(reportDef)));			
		}
	}


	/**
	 * 删除报告文件
	 *
	 * @param req  请求
	 * @param resp 响应
	 * @throws ServletException servlet异常
	 * @throws IOException      ioexception
	 */
	public void deleteReportFile(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
		String file=req.getParameter("file");
		if(file==null){
			throw new ReportDesignException("Report file can not be null.");
		}
		ReportProvider targetReportProvider=null;
		for(ReportProvider provider:reportProviders){
			if(file.startsWith(provider.getPrefix())){
				targetReportProvider=provider;
				break;
			}
		}
		if(targetReportProvider==null){
			throw new ReportDesignException("File ["+file+"] not found available report provider.");
		}
		targetReportProvider.deleteReport(file);
	}

	/**
	 * 保存报告文件
	 *
	 * @param req  请求
	 * @param resp 响应
	 * @throws ServletException servlet异常
	 * @throws IOException      ioexception
	 */
	public void saveReportFile(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
		String file=req.getParameter("file");
		file=ReportUtils.decodeFileName(file);
		String content=req.getParameter("content");
		content=decodeContent(content);
		String metadata = req.getParameter("metadata");
		metadata = decodeContent(metadata);

		ReportProvider targetReportProvider=null;
		for(ReportProvider provider:reportProviders){
			if(file.startsWith(provider.getPrefix())){
				targetReportProvider=provider;
				break;
			}
		}
		if(targetReportProvider==null){
			throw new ReportDesignException("File ["+file+"] not found available report provider.");
		}
		targetReportProvider.saveReport(file,content,metadata);
		InputStream inputStream=IOUtils.toInputStream(content,"utf-8");
		ReportDefinition reportDef=reportParser.parse(inputStream, file,metadata);
		reportRender.rebuildReportDefinition(reportDef);
		CacheUtils.cacheReportDefinition(file, reportDef);
		IOUtils.closeQuietly(inputStream);
	}

	/**
	 * 负载报告提供者
	 *
	 * @param req  请求
	 * @param resp 响应
	 * @throws ServletException servlet异常
	 * @throws IOException      ioexception
	 */
	public void loadReportProviders(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
		writeObjectToJson(resp, reportProviders);
	}

	/**
	 * 设置报告呈现
	 *
	 * @param reportRender 报告呈现
	 */
	public void setReportRender(ReportRender reportRender) {
		this.reportRender = reportRender;
	}

	/**
	 * 设置报告解析器
	 *
	 * @param reportParser 报告解析器
	 */
	public void setReportParser(ReportParser reportParser) {
		this.reportParser = reportParser;
	}

	/**
	 * 设置应用程序上下文
	 *
	 * @param applicationContext 应用程序上下文
	 * @throws BeansException beans异常
	 */
	@Override
	public void setApplicationContext(ApplicationContext applicationContext)throws BeansException {
		super.setApplicationContext(applicationContext);
		Collection<ReportProvider> providers=applicationContext.getBeansOfType(ReportProvider.class).values();
		for(ReportProvider provider:providers){
			if(provider.disabled() || provider.getName()==null){
				continue;
			}
			reportProviders.add(provider);
		}
	}

	/**
	 * url
	 *
	 * @return {@link String}
	 */
	@Override
	public String url() {
		return "/designer";
	}
}
