package com.efuture.ocp.common.cache.config;

import com.efuture.ocp.common.cache.template.CacheTemplate;
import com.jarvis.cache.CacheHandler;
import com.jarvis.cache.ComboCacheManager;
import com.jarvis.cache.ICacheManager;
import com.jarvis.cache.autoconfigure.AutoloadCacheManageConfiguration;
import com.jarvis.cache.autoconfigure.AutoloadCacheProperties;
import com.jarvis.cache.autoconfigure.DistributedLockConfiguration;
import com.jarvis.cache.clone.ICloner;
import com.jarvis.cache.lock.ILock;
import com.jarvis.cache.map.MapCacheManager;
import com.jarvis.cache.script.AbstractScriptParser;
import com.jarvis.cache.script.SpringELParser;
import com.jarvis.cache.serializer.HessianSerializer;
import com.jarvis.cache.serializer.ISerializer;
import com.jarvis.cache.serializer.JdkSerializer;
import com.jarvis.cache.serializer.KryoSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.util.ClassUtils;

@Order(0)
@Configuration
@AutoConfigureAfter({FutureCacheManagerConfig.class, AutoloadCacheManageConfiguration.class, DistributedLockConfiguration.class})
@EnableConfigurationProperties({AutoloadCacheProperties.class})
public class FutureCacheAutoConfig
{

    private static final boolean hessianPresent = ClassUtils.isPresent(
                "com.caucho.hessian.io.AbstractSerializerFactory", AutoloadCacheManageConfiguration.class.getClassLoader());
    private static final boolean kryoPresent = ClassUtils.isPresent(
                "com.esotericsoftware.kryo.Kryo", AutoloadCacheManageConfiguration.class.getClassLoader());
    private final ILock lock;

    //    @Bean
    //    public ICacheManager mapCacheManager(AutoloadCacheProperties config, ICloner cloner) {
    //        return new MapCacheManager(config.getConfig(), cloner);
    //    }

    //    @Bean
    //    public RedisProperty getRedisProperty() {
    //        return new RedisProperty();
    //    }
    Logger log = LoggerFactory.getLogger(FutureCacheAutoConfig.class);

    public FutureCacheAutoConfig(ObjectProvider<ILock> lockProvider)
    {
        if (log.isDebugEnabled()) {
            log.debug("FutureCacheAutoConfig------->>>>缓存自动装配启动成功");
        }

        if (null != lockProvider) {
            lock = lockProvider.getIfAvailable();
        }
        else {
            lock = null;
        }
    }


    //    /**
    //     * * 序列化工具{@link ISerializer ISerializer} 注入规则：<br>
    //     * 如果导入了Hessian的jar包，优先使用Hessian：{@link HessianSerializer
    //     * HessianSerializer},否则使用{@link JdkSerializer JdkSerializer}<br>
    //     *
    //     * @return
    //     */
    @Bean
    @ConditionalOnMissingBean(ISerializer.class)
    public ISerializer<Object> getCacheSerializer()
    {
        ISerializer<Object> res;

        // 推荐优先使用：Hessian
        if (hessianPresent) {
            res = new HessianSerializer();
            log.debug("HessianSerializer auto-configured");
        }
        else if (kryoPresent) {
            res = new KryoSerializer();
            log.debug("KryoSerializer auto-configured");
        }
        else {
            res = new JdkSerializer();
            log.debug("JdkSerializer auto-configured");
        }

        return res;
    }

    @Bean
    @ConditionalOnMissingBean(AbstractScriptParser.class)
    public AbstractScriptParser autoloadCacheScriptParser()
    {
        return new SpringELParser();
    }

    @Bean(destroyMethod = "destroy")
    @ConditionalOnMissingBean(CacheHandler.class)
    //@ConditionalOnBean()
    public CacheHandler cacheHandler(ICacheManager cacheManager, AutoloadCacheProperties config)
    {
        AbstractScriptParser scriptParser = autoloadCacheScriptParser();
        ICloner cloner = getCacheSerializer();
        ICacheManager comboCacheManager = comboCacheManager(cacheManager, config, cloner, scriptParser);
        CacheHandler cacheHandler = new CacheHandler(comboCacheManager, scriptParser, config.getConfig(), cloner);
        cacheHandler.setLock(lock);
        return cacheHandler;
    }

    @Bean
    @ConditionalOnMissingBean(CacheTemplate.class)
    public CacheTemplate cacheTemplate(CacheHandler cacheHandler)
    {
        return new CacheTemplate(cacheHandler.getCacheManager(), cacheHandler.getAutoLoadConfig());
    }

    /**
     * 创建CacheManager
     * 1.默认自动创建localCache
     * 2.如果没有远程缓存，则直接使用localCache
     *
     * @param config
     * @param cloner
     * @param scriptParser
     * @return
     */
    public ICacheManager comboCacheManager(ICacheManager remoteCache, AutoloadCacheProperties config, ICloner cloner, AbstractScriptParser scriptParser)
    {
        ICacheManager localCache = getLocalCache(config, cloner, scriptParser);

        if (remoteCache == null) {
            return localCache;
        }

        return new ComboCacheManager(localCache, remoteCache, scriptParser);
    }

    private ICacheManager getLocalCache(AutoloadCacheProperties config, ICloner cloner, AbstractScriptParser scriptParser)
    {
        return new MapCacheManager(config.getConfig(), cloner);
    }


}
