/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.engine.processing.streamprocessor;

import io.camunda.zeebe.db.TransactionContext;
import io.camunda.zeebe.db.ZeebeDb;
import io.camunda.zeebe.engine.metrics.StreamProcessorMetrics;
import io.camunda.zeebe.engine.processing.streamprocessor.ProcessingContext;
import io.camunda.zeebe.engine.processing.streamprocessor.ProcessingStateMachine;
import io.camunda.zeebe.engine.processing.streamprocessor.RecordProcessorMap;
import io.camunda.zeebe.engine.processing.streamprocessor.RecordValues;
import io.camunda.zeebe.engine.processing.streamprocessor.ReplayStateMachine;
import io.camunda.zeebe.engine.processing.streamprocessor.StreamProcessorBuilder;
import io.camunda.zeebe.engine.processing.streamprocessor.StreamProcessorLifecycleAware;
import io.camunda.zeebe.engine.processing.streamprocessor.StreamProcessorMode;
import io.camunda.zeebe.engine.processing.streamprocessor.TypedRecordProcessorFactory;
import io.camunda.zeebe.engine.processing.streamprocessor.TypedRecordProcessors;
import io.camunda.zeebe.engine.processing.streamprocessor.writers.TypedStreamWriterImpl;
import io.camunda.zeebe.engine.state.EventApplier;
import io.camunda.zeebe.engine.state.ZbColumnFamilies;
import io.camunda.zeebe.engine.state.ZeebeDbState;
import io.camunda.zeebe.engine.state.mutable.MutableZeebeState;
import io.camunda.zeebe.logstreams.impl.Loggers;
import io.camunda.zeebe.logstreams.log.LogRecordAwaiter;
import io.camunda.zeebe.logstreams.log.LogStream;
import io.camunda.zeebe.logstreams.log.LogStreamBatchWriter;
import io.camunda.zeebe.logstreams.log.LogStreamReader;
import io.camunda.zeebe.util.exception.UnrecoverableException;
import io.camunda.zeebe.util.health.FailureListener;
import io.camunda.zeebe.util.health.HealthMonitorable;
import io.camunda.zeebe.util.health.HealthStatus;
import io.camunda.zeebe.util.sched.Actor;
import io.camunda.zeebe.util.sched.ActorSchedulingService;
import io.camunda.zeebe.util.sched.clock.ActorClock;
import io.camunda.zeebe.util.sched.future.ActorFuture;
import io.camunda.zeebe.util.sched.future.CompletableActorFuture;
import java.time.Duration;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import org.slf4j.Logger;

public class StreamProcessor
extends Actor
implements HealthMonitorable,
LogRecordAwaiter {
    public static final long UNSET_POSITION = -1L;
    public static final Duration HEALTH_CHECK_TICK_DURATION = Duration.ofSeconds(5L);
    private static final String ERROR_MESSAGE_RECOVER_FROM_SNAPSHOT_FAILED = "Expected to find event with the snapshot position %s in log stream, but nothing was found. Failed to recover '%s'.";
    private static final Logger LOG = Loggers.LOGSTREAMS_LOGGER;
    private final ActorSchedulingService actorSchedulingService;
    private final AtomicBoolean isOpened = new AtomicBoolean(false);
    private final List<StreamProcessorLifecycleAware> lifecycleAwareListeners;
    private final Function<MutableZeebeState, EventApplier> eventApplierFactory;
    private final Set<FailureListener> failureListeners = new HashSet<FailureListener>();
    private final StreamProcessorMetrics metrics;
    private final LogStream logStream;
    private final int partitionId;
    private final ZeebeDb zeebeDb;
    private final ProcessingContext processingContext;
    private final TypedRecordProcessorFactory typedRecordProcessorFactory;
    private final String actorName;
    private LogStreamReader logStreamReader;
    private long snapshotPosition = -1L;
    private ProcessingStateMachine processingStateMachine;
    private ReplayStateMachine replayStateMachine;
    private volatile Phase phase = Phase.REPLAY;
    private CompletableActorFuture<Void> openFuture;
    private CompletableActorFuture<Void> closeFuture = CompletableActorFuture.completed(null);
    private volatile long lastTickTime;
    private boolean shouldProcess = true;
    private ActorFuture<Long> replayCompletedFuture;

    protected StreamProcessor(StreamProcessorBuilder processorBuilder) {
        this.actorSchedulingService = processorBuilder.getActorSchedulingService();
        this.lifecycleAwareListeners = processorBuilder.getLifecycleListeners();
        this.typedRecordProcessorFactory = processorBuilder.getTypedRecordProcessorFactory();
        this.zeebeDb = processorBuilder.getZeebeDb();
        this.eventApplierFactory = processorBuilder.getEventApplierFactory();
        this.processingContext = processorBuilder.getProcessingContext().eventCache(new RecordValues()).actor(this.actor).abortCondition(this::isClosed);
        this.logStream = this.processingContext.getLogStream();
        this.partitionId = this.logStream.getPartitionId();
        this.actorName = StreamProcessor.buildActorName((int)processorBuilder.getNodeId(), (String)"StreamProcessor", (int)this.partitionId);
        this.metrics = new StreamProcessorMetrics(this.partitionId);
    }

    public static StreamProcessorBuilder builder() {
        return new StreamProcessorBuilder();
    }

    protected Map<String, String> createContext() {
        Map context = super.createContext();
        context.put("partitionId", Integer.toString(this.partitionId));
        return context;
    }

    public String getName() {
        return this.actorName;
    }

    protected void onActorStarting() {
        this.actor.runOnCompletionBlockingCurrentPhase(this.logStream.newLogStreamBatchWriter(), this::onRetrievingWriter);
    }

    protected void onActorStarted() {
        try {
            LOG.debug("Recovering state of partition {} from snapshot", (Object)this.partitionId);
            long startTime = ActorClock.currentTimeMillis();
            this.snapshotPosition = this.recoverFromSnapshot();
            this.initProcessors();
            this.processingStateMachine = new ProcessingStateMachine(this.processingContext, this::shouldProcessNext);
            this.healthCheckTick();
            this.replayStateMachine = new ReplayStateMachine(this.processingContext, this::shouldProcessNext);
            this.processingContext.disableLogStreamWriter();
            this.openFuture.complete(null);
            this.replayCompletedFuture = this.replayStateMachine.startRecover(this.snapshotPosition);
            if (!this.shouldProcess) {
                this.setStateToPausedAndNotifyListeners();
            }
            if (this.isInReplayOnlyMode()) {
                this.replayCompletedFuture.onComplete((v, error) -> {
                    if (error != null) {
                        LOG.error("The replay of events failed.", error);
                        this.onFailure((Throwable)error);
                    }
                });
            } else {
                this.replayCompletedFuture.onComplete((lastSourceEventPosition, error) -> {
                    if (error != null) {
                        LOG.error("The replay of events failed.", error);
                        this.onFailure((Throwable)error);
                    } else {
                        this.onRecovered((long)lastSourceEventPosition);
                        this.metrics.recoveryTime(ActorClock.currentTimeMillis() - startTime);
                    }
                });
            }
        }
        catch (RuntimeException e) {
            this.onFailure(e);
        }
    }

    protected void onActorClosing() {
        this.tearDown();
    }

    protected void onActorClosed() {
        this.closeFuture.complete(null);
        LOG.debug("Closed stream processor controller {}.", (Object)this.getName());
    }

    protected void onActorCloseRequested() {
        if (!this.isFailed()) {
            this.lifecycleAwareListeners.forEach(StreamProcessorLifecycleAware::onClose);
        }
    }

    public ActorFuture<Void> closeAsync() {
        if (this.isOpened.compareAndSet(true, false)) {
            this.closeFuture = new CompletableActorFuture();
            this.actor.close();
        }
        return this.closeFuture;
    }

    protected void handleFailure(Exception failure) {
        this.onFailure(failure);
    }

    public void onActorFailed() {
        this.phase = Phase.FAILED;
        this.closeFuture = CompletableActorFuture.completed(null);
        this.isOpened.set(false);
        this.lifecycleAwareListeners.forEach(StreamProcessorLifecycleAware::onFailed);
        this.tearDown();
    }

    private boolean shouldProcessNext() {
        return this.isOpened() && this.shouldProcess;
    }

    private void tearDown() {
        this.processingContext.getLogStreamReader().close();
        this.logStream.removeRecordAvailableListener((LogRecordAwaiter)this);
        this.replayStateMachine.close();
    }

    private void healthCheckTick() {
        this.lastTickTime = ActorClock.currentTimeMillis();
        this.actor.runDelayed(HEALTH_CHECK_TICK_DURATION, this::healthCheckTick);
    }

    private void onRetrievingWriter(LogStreamBatchWriter batchWriter, Throwable errorOnReceivingWriter) {
        if (errorOnReceivingWriter == null) {
            this.processingContext.maxFragmentSize(batchWriter.getMaxFragmentLength()).logStreamWriter(new TypedStreamWriterImpl(batchWriter));
            this.actor.runOnCompletionBlockingCurrentPhase(this.logStream.newLogStreamReader(), this::onRetrievingReader);
        } else {
            LOG.error("Unexpected error on retrieving batch writer from log stream.", errorOnReceivingWriter);
            this.actor.close();
        }
    }

    private void onRetrievingReader(LogStreamReader reader, Throwable errorOnReceivingReader) {
        if (errorOnReceivingReader == null) {
            this.logStreamReader = reader;
            this.processingContext.logStreamReader(reader);
        } else {
            LOG.error("Unexpected error on retrieving reader from log stream.", errorOnReceivingReader);
            this.actor.close();
        }
    }

    public ActorFuture<Void> openAsync(boolean pauseOnStart) {
        if (this.isOpened.compareAndSet(false, true)) {
            this.shouldProcess = !pauseOnStart;
            this.openFuture = new CompletableActorFuture();
            this.actorSchedulingService.submitActor((Actor)this);
        }
        return this.openFuture;
    }

    private void initProcessors() {
        TypedRecordProcessors typedRecordProcessors = this.typedRecordProcessorFactory.createProcessors(this.processingContext);
        this.lifecycleAwareListeners.addAll(typedRecordProcessors.getLifecycleListeners());
        RecordProcessorMap recordProcessorMap = typedRecordProcessors.getRecordProcessorMap();
        recordProcessorMap.values().forEachRemaining(this.lifecycleAwareListeners::add);
        this.processingContext.recordProcessorMap(recordProcessorMap);
    }

    private long recoverFromSnapshot() {
        boolean failedToRecoverReader;
        ZeebeDbState zeebeState = this.recoverState();
        long snapshotPosition = zeebeState.getLastProcessedPositionState().getLastSuccessfulProcessedRecordPosition();
        boolean bl = failedToRecoverReader = !this.logStreamReader.seekToNextEvent(snapshotPosition);
        if (failedToRecoverReader && this.processingContext.getProcessorMode() == StreamProcessorMode.PROCESSING) {
            throw new IllegalStateException(String.format(ERROR_MESSAGE_RECOVER_FROM_SNAPSHOT_FAILED, snapshotPosition, this.getName()));
        }
        LOG.info("Recovered state of partition {} from snapshot at position {}", (Object)this.partitionId, (Object)snapshotPosition);
        return snapshotPosition;
    }

    private ZeebeDbState recoverState() {
        TransactionContext transactionContext = this.zeebeDb.createContext();
        ZeebeDbState zeebeState = new ZeebeDbState(this.partitionId, (ZeebeDb<ZbColumnFamilies>)this.zeebeDb, transactionContext);
        this.processingContext.transactionContext(transactionContext);
        this.processingContext.zeebeState(zeebeState);
        this.processingContext.eventApplier(this.eventApplierFactory.apply(zeebeState));
        return zeebeState;
    }

    private void onRecovered(long lastSourceEventPosition) {
        this.phase = Phase.PROCESSING;
        this.processingContext.enableLogStreamWriter();
        this.logStream.registerRecordAvailableListener((LogRecordAwaiter)this);
        this.lifecycleAwareListeners.forEach(l -> l.onRecovered(this.processingContext));
        this.processingStateMachine.startProcessing(lastSourceEventPosition);
        if (!this.shouldProcess) {
            this.setStateToPausedAndNotifyListeners();
        }
    }

    private void onFailure(Throwable throwable) {
        LOG.error("Actor {} failed in phase {}.", new Object[]{this.actorName, this.actor.getLifecyclePhase(), throwable});
        this.actor.fail();
        if (!this.openFuture.isDone()) {
            this.openFuture.completeExceptionally(throwable);
        }
        if (throwable instanceof UnrecoverableException) {
            this.failureListeners.forEach(FailureListener::onUnrecoverableFailure);
        } else {
            this.failureListeners.forEach(FailureListener::onFailure);
        }
    }

    public boolean isOpened() {
        return this.isOpened.get();
    }

    public boolean isClosed() {
        return !this.isOpened.get();
    }

    public boolean isFailed() {
        return this.phase == Phase.FAILED;
    }

    public ActorFuture<Long> getLastProcessedPositionAsync() {
        return this.actor.call(() -> {
            if (this.isInReplayOnlyMode()) {
                return this.replayStateMachine.getLastSourceEventPosition();
            }
            return this.processingStateMachine.getLastSuccessfulProcessedEventPosition();
        });
    }

    private boolean isInReplayOnlyMode() {
        return this.processingContext.getProcessorMode() == StreamProcessorMode.REPLAY;
    }

    public ActorFuture<Long> getLastWrittenPositionAsync() {
        return this.actor.call(() -> {
            if (this.isInReplayOnlyMode()) {
                return this.replayStateMachine.getLastReplayedEventPosition();
            }
            return this.processingStateMachine.getLastWrittenEventPosition();
        });
    }

    public HealthStatus getHealthStatus() {
        if (this.actor.isClosed()) {
            return HealthStatus.UNHEALTHY;
        }
        if (this.processingStateMachine == null || !this.processingStateMachine.isMakingProgress()) {
            return HealthStatus.UNHEALTHY;
        }
        if (ActorClock.currentTimeMillis() - this.lastTickTime > HEALTH_CHECK_TICK_DURATION.toMillis() * 2L) {
            return HealthStatus.UNHEALTHY;
        }
        if (this.phase == Phase.FAILED) {
            return HealthStatus.UNHEALTHY;
        }
        return HealthStatus.HEALTHY;
    }

    public void addFailureListener(FailureListener failureListener) {
        this.actor.run(() -> this.failureListeners.add(failureListener));
    }

    public void removeFailureListener(FailureListener failureListener) {
        this.actor.run(() -> this.failureListeners.remove(failureListener));
    }

    public ActorFuture<Phase> getCurrentPhase() {
        return this.actor.call(() -> this.phase);
    }

    public ActorFuture<Void> pauseProcessing() {
        return this.actor.call(() -> {
            if (this.shouldProcess) {
                this.setStateToPausedAndNotifyListeners();
            }
        });
    }

    private void setStateToPausedAndNotifyListeners() {
        if (this.isInReplayOnlyMode() || !this.replayCompletedFuture.isDone()) {
            LOG.debug("Paused replay for partition {}", (Object)this.partitionId);
        } else {
            this.lifecycleAwareListeners.forEach(StreamProcessorLifecycleAware::onPaused);
            LOG.debug("Paused processing for partition {}", (Object)this.partitionId);
        }
        this.shouldProcess = false;
        this.phase = Phase.PAUSED;
    }

    public void resumeProcessing() {
        this.actor.call(() -> {
            if (!this.shouldProcess) {
                this.shouldProcess = true;
                if (this.isInReplayOnlyMode() || !this.replayCompletedFuture.isDone()) {
                    this.phase = Phase.REPLAY;
                    this.actor.submit(this.replayStateMachine::replayNextEvent);
                    LOG.debug("Resumed replay for partition {}", (Object)this.partitionId);
                } else {
                    this.lifecycleAwareListeners.forEach(StreamProcessorLifecycleAware::onResumed);
                    this.phase = Phase.PROCESSING;
                    this.actor.submit(this.processingStateMachine::readNextEvent);
                    LOG.debug("Resumed processing for partition {}", (Object)this.partitionId);
                }
            }
        });
    }

    public void onRecordAvailable() {
        this.actor.run(this.processingStateMachine::readNextEvent);
    }

    public static enum Phase {
        REPLAY,
        PROCESSING,
        FAILED,
        PAUSED;

    }
}

